package com.netease.mail.yanxuan.change.integration.client;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.parser.ParserConfig;
import com.netease.mail.yanxuan.change.integration.email.exception.RpcException;
import com.netease.mail.yanxuan.change.integration.email.exception.RpcTimeoutException;
import com.netease.mail.yanxuan.change.integration.email.util.EncodeUtil;
import com.netease.mail.yanxuan.change.integration.email.util.HttpClientUtil;
import com.netease.yanxuan.missa.pressure.TracerClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.util.List;
import java.util.Map;

@Component
public class RpcTemplate {
    private static final Logger LOG = LoggerFactory.getLogger(RpcTemplate.class);

    @Value("${act.rpc.get.retry:1}")
    private int rpcGetRetry;

    @Value("${act.rpc.post.retry:0}")
    private int rpcPostRetry;

    @Value("${act.rpc.collect:false}")
    private boolean rpcCollect;

    public RpcTemplate() {
        ParserConfig.getGlobalInstance().setAutoTypeSupport(true);
    }

    /**
     * Get请求
     *
     * @param url
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T get(String url, CallHandler<T> handler) {
        return get(url, null, null, -1, handler);
    }

    /**
     * Get请求
     *
     * @param url
     * @param params
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T get(String url, Map<String, Object> params, CallHandler<T> handler) {
        return get(url, params, null, -1, handler);
    }

    /**
     * Get请求
     *
     * @param url
     * @param params
     * @param header
     * @param timeout
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T get(String url, Map<String, Object> params, Map<String, String> header,
        int timeout, CallHandler<T> handler) {
        CallTemplate template = () -> HttpClientUtil.get(url, params, header, timeout);
        return this.execute(url, params, handler, template, rpcGetRetry, Method.GET);
    }

    /**
     * Post请求
     *
     * @param url
     * @param params
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T post(String url, Map<String, Object> params, CallHandler<T> handler) {
        return post(url, params, -1, handler);
    }

    /**
     * Post请求
     *
     * @param url
     * @param params
     * @param timeout
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T post(String url, Map<String, Object> params, int timeout, CallHandler<T> handler) {
        CallTemplate template = () -> HttpClientUtil.post(url, params, timeout);
        return this.execute(url, params, handler, template, rpcPostRetry, Method.POST);
    }

    /**
     * Post Json请求
     *
     * @param url
     * @param params
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T postJson(String url, String params, CallHandler<T> handler) {
        return postJson(url, params, -1, handler);
    }

    /**
     * Post Json请求
     *
     * @param url
     * @param params
     * @param timeout
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T postJson(String url, String params, int timeout, CallHandler<T> handler) {
        CallTemplate template = () -> HttpClientUtil.postJson(url, params, timeout);
        return this.execute(url, params, handler, template, rpcPostRetry, Method.BODY);
    }

    /**
     * 执行模板调用
     *
     * @param url
     *            请求地址
     * @param params
     *            请求报文
     * @param handler
     *            返回解析
     * @param callTemplate
     *            模板调用
     * @param retryTime
     *            最多重复次数，0默认（3次）
     */
    private <T> T execute(String url, Object params, CallHandler<T> handler,
        CallTemplate callTemplate, int retryTime, Method method) {
        int tryTimes = 0;
        RpcException rpcException = null;
        String respBody = null;
        RpcStatus status = RpcStatus.SUCCESS;

        while (tryTimes <= retryTime) {
            long startTime = System.currentTimeMillis();
            try {
                respBody = callTemplate.executeCall();
                if (StringUtils.isEmpty(respBody)) {
                    throw new RpcException("response empty");
                }

                T t = handler.handle(respBody);
                status = checkResp(handler, t);
                return t;
            } catch (RpcTimeoutException e) {
                rpcException = e;
                status = RpcStatus.TIMEOUT;
            } catch (RpcException e) {
                rpcException = e;
                status = RpcStatus.EXCEPTION;
            } catch (Exception e) {
                rpcException = new RpcException(e);
                status = RpcStatus.EXCEPTION;
            } finally {
                try {
                    if (LOG.isDebugEnabled() || rpcCollect) {
                        long costTime = System.currentTimeMillis() - startTime;
                        String reqParam = convertParam(params);

                        // 根据需要开启rpc日志
                        if (LOG.isDebugEnabled()) {

                            LOG.debug(
                                "process request, yx-tracer-flowType={}, url={}, method={}, params={}, response={}, cost={}ms",
                                TracerClient.getFlowType(), url, method.name(), reqParam, respBody,
                                costTime);
                        }
                    }
                } catch (Exception e) {
                    LOG.error("collect error, msg={}", e.getMessage());
                }
            }
            tryTimes++;
        }

        LOG.error("request error, url={}, params={}, respBody={}", url,
            params instanceof String ? params : JSON.toJSONString(params), respBody, rpcException);

        if (rpcException == null) {
            rpcException = new RpcException("服务异常");
        }
        throw rpcException;
    }

    /**
     * 校验resp
     *
     * @param t
     * @param <T>
     * @return
     */
    private <T> RpcStatus checkResp(CallHandler<T> handler, T t) {
        RpcStatus status = RpcStatus.SUCCESS;
        try {
            // 为空，或handlerCheck为false直接判断FAIL
            if (t == null || !handler.check(t)) {
                return RpcStatus.WARNING;
            }
            if (t instanceof Checker) {
                status = ((Checker) t).check() ? RpcStatus.SUCCESS : RpcStatus.WARNING;
            } else if (t instanceof List) {
                // 为空认为不是理想结果
                if (CollectionUtils.isEmpty((List) t)) {
                    status = RpcStatus.WARNING;
                } else {
                    // 列表数据简单检测第一个元素
                    Object obj = ((List) t).get(0);
                    if (obj instanceof Checker) {
                        status = ((Checker) obj).check() ? RpcStatus.SUCCESS : RpcStatus.WARNING;
                    }
                }
            }
        } catch (Exception e) {
            LOG.error("collect error, msg={}", e.getMessage());
        }
        return status;
    }

    private String convertParam(Object param) {
        if (param instanceof String) {
            return (String) param;
        }
        String reqParam = "";
        if (param instanceof Map) {
            reqParam = EncodeUtil.encodeMap((Map) param);
        } else if (param != null) {
            reqParam = JSON.toJSONString(param);
        }
        return reqParam;
    }

    enum RpcStatus {

        /**
         * 成功
         */
        SUCCESS(1),

        /**
         * 告警（请求成功，但非理想值）
         */
        WARNING(2),

        /**
         * 异常
         */
        EXCEPTION(3),

        /**
         * 超时
         */
        TIMEOUT(4);

        private final int value;

        RpcStatus(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }

    enum Method {

        /**
         * Get请求
         */
        GET,

        /**
         * Post表单请求
         */
        POST,

        /**
         * 数据体（一般是post json）
         */
        BODY
    }

    /**
     * 模板调用
     */
    @FunctionalInterface
    public interface CallTemplate {

        /**
         * 模板调用类
         */
        String executeCall();
    }

    /**
     * 报文解析
     */
    @FunctionalInterface
    public interface CallHandler<T> {

        /**
         * 报文解析
         *
         * @param resp
         *            response body
         */
        T handle(String resp) throws IOException;

        /**
         * 校验数据体
         *
         * @param t
         * @return
         */
        default boolean check(T t) {
            return true;
        }
    }
}
