/**
 * @(#)FlowService.java, 2022/11/29.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.mail.yanxuan.change.biz.service.rpc;

import java.util.List;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.netease.mail.yanxuan.change.biz.meta.exception.ExceptionFactory;
import com.netease.mail.yanxuan.change.common.bean.CommonConstants;
import com.netease.mail.yanxuan.change.common.bean.ResponseCode;
import com.netease.mail.yanxuan.change.common.bean.ResponseCodeEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeFlowEnum;
import com.netease.mail.yanxuan.change.common.enums.FlowxOperationEnum;
import com.netease.mail.yanxuan.change.integration.flow.FlowRpcService;
import com.netease.yanxuan.flowx.sdk.meta.controller.communal.AjaxResponse;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.FlowDataDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.UserBaseDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.flow.FlowCreateReqDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.flow.NodeSubmitReqDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.flow.UpdateNodeApproverDTO;

import lombok.extern.slf4j.Slf4j;

/**
 * @Author zcwang
 * @Date 2022/11/29
 * 工单流转相关操作
 */
@Slf4j
@Component
public class FlowService {

    @Autowired
    private FlowRpcService flowRpcService;

    /**
     * 创建工单
     * @param flowCreateReqDTO
     * @return
     */
    public String createFlow(FlowCreateReqDTO flowCreateReqDTO) {
        // 先创建工单，创建完成后直接提交
        log.info("[createFlow] flowCreateReqDTO={}", flowCreateReqDTO);
        AjaxResponse<String> flowCreateResponse = flowRpcService.create(CommonConstants.FLOWX_PRODUCT,
                flowCreateReqDTO);
        String flowId;
        if (ResponseCodeEnum.SUCCESS.getCode() == flowCreateResponse.getCode()) {
            flowId = flowCreateResponse.getData();
            log.info("[createFlow] flowId ={}", JSON.toJSONString(flowId));
        } else {
            log.error("[createFlow] create flow failed, query={}, errMsg={}", JSON.toJSONString(flowCreateReqDTO),
                    JSON.toJSONString(flowCreateResponse));
            throw ExceptionFactory.createBiz(ResponseCode.CREATE_FLOW_ERROR, "创建工单错误");
        }
        return flowId;
    }

    /**
     * 提交工单
     * @param flowId
     * @param flowDataDTO
     * @param uid
     * @param topoId
     * @param content
     * @param approved
     * @return
     */
    public String submitFlow(String flowId, FlowDataDTO flowDataDTO, String uid, String topoId, String content,
        boolean approved, String operateResult, String remark) {
        NodeSubmitReqDTO nodeSubmitReqDTO = new NodeSubmitReqDTO();
        nodeSubmitReqDTO.setFlowId(flowId);
        nodeSubmitReqDTO.setNodeId(flowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId());
        nodeSubmitReqDTO.setUid(uid);
        nodeSubmitReqDTO.setUserName(uid);
        nodeSubmitReqDTO.setTopoId(topoId);
        nodeSubmitReqDTO.setContent(content);
        nodeSubmitReqDTO.setOperateResult(operateResult);
        nodeSubmitReqDTO.setRemark(remark);
        nodeSubmitReqDTO.setCreateTime(System.currentTimeMillis());
        nodeSubmitReqDTO.setApproved(approved);
        nodeSubmitReqDTO.setOccLock(flowDataDTO.getOccLock() + 1);
        log.info("[submitFlow] nodeSubmitReqDTO={}", JSON.toJSONString(nodeSubmitReqDTO));
        List<String> nextNodeIdList;
        // 提交工单
        AjaxResponse<List<String>> submitResponse = flowRpcService.submit(CommonConstants.FLOWX_PRODUCT,
            nodeSubmitReqDTO);
        if (ResponseCodeEnum.SUCCESS.getCode() == submitResponse.getCode()) {
            nextNodeIdList = submitResponse.getData();
            log.info("[submitFlow] nextNodeIdList={}", JSON.toJSONString(nextNodeIdList));
        } else {
            log.error("[submitFlow] submit flow failed, query={}, errMsg={}", JSON.toJSONString(nodeSubmitReqDTO),
                JSON.toJSONString(submitResponse));
            throw ExceptionFactory.createBiz(ResponseCode.SUBMIT_FLOW_ERROR, "提交工单错误");
        }
        if (CollectionUtils.isEmpty(nextNodeIdList)) {
            return ChangeFlowEnum.END.getNodeId();
        }
        return nextNodeIdList.get(0);
    }

    /**
     * 查询工单详情
     * @param flowId
     * @return
     */
    public FlowDataDTO flowDetail(String flowId) {
        // 查询工单详情，根据详情提交
        FlowDataDTO flowDataDTO;
        AjaxResponse<FlowDataDTO> flowDetailResponse = flowRpcService.getDetail(CommonConstants.FLOWX_PRODUCT, flowId);
        if (ResponseCodeEnum.SUCCESS.getCode() == flowDetailResponse.getCode()) {
            flowDataDTO = flowDetailResponse.getData();
            log.info("[detailFlow] flowDataDTO={}", JSON.toJSONString(flowDataDTO));
        } else {
            log.error("[detailFlow] get flow detail failed, query={}, errMsg={}", JSON.toJSONString(flowId),
                    JSON.toJSONString(flowDetailResponse));
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "查询工单详情错误");
        }
        return flowDataDTO;
    }

    /**
     * 更新待办人
     * @param topoId
     * @param flowId
     * @param currentNodeId
     * @param newApproverList
     * @param uid
     * @param userName
     * @param remark
     * @param content
     */
    public void updateApprovers(String topoId, String flowId, String currentNodeId, List<UserBaseDTO> newApproverList,
                         String uid, String userName, String remark, String content){
        FlowDataDTO flowDataDTO = this.flowDetail(flowId);
        UpdateNodeApproverDTO approverDTO = new UpdateNodeApproverDTO();
        approverDTO.setOccLock(flowDataDTO.getOccLock() + 1);
        approverDTO.setTopoId(topoId);
        approverDTO.setNodeId(currentNodeId);
        approverDTO.setFlowId(flowId);
        approverDTO.setNewApproverList(newApproverList);
        approverDTO.setUid(uid);
        approverDTO.setUserName(userName);
        approverDTO.setRemark(remark);
        approverDTO.setContent(content);
        approverDTO.setOperateResult(FlowxOperationEnum.DELIVER.getName());
        flowRpcService.updateApprovers(CommonConstants.FLOWX_PRODUCT, approverDTO);
    }
}