package com.netease.yanxuan.wx.store.sharer.biz.interceptor;

import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserContextHolder;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserHelper;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserInfo;
import com.netease.yanxuan.wx.store.sharer.common.annotation.LoginRequired;
import com.netease.yanxuan.wx.store.sharer.common.constant.CoreConstant;
import com.netease.yanxuan.wx.store.sharer.common.exception.NoAuthException;
import com.netease.yanxuan.wx.store.sharer.common.handler.RedisClient;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;

/**
 * @Description 鉴权拦截器
 * @Author fanjiaxin
 * @Date 2025/3/10 10:20
 */
@Slf4j
@RequiredArgsConstructor
@Component
public class AuthInterceptor implements HandlerInterceptor {
    private final RedisClient redisClient;
    private final LoginUserHelper jwtHelper;


    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        log.info("AuthInterceptor preHandle execute, url:{}", request.getRequestURI());
        HandlerMethod handlerMethod = (HandlerMethod) handler;
        Method method = handlerMethod.getMethod();
        LoginRequired validateAnnotation = method.getAnnotation(LoginRequired.class);
        String token = request.getHeader(CoreConstant.AUTH_HEADER_TOKEN_KEY);
        log.info("AuthInterceptor preHandle token:{}", token);
        LoginUserInfo loginUserInfo;
        if (null != validateAnnotation) {
            if (StringUtils.isBlank(token)) {
                throw new NoAuthException("用户认证失败，TOKEN为空");
            }
            loginUserInfo = jwtHelper.getLoginUserInfo(token);
            if (null == loginUserInfo) {
                log.info("用户登录状态过期，token缓存失效,path:{}", request.getServletPath());
                throw new NoAuthException("用户登录状态过期");
            }
        } else if (StringUtils.isNotBlank(token)) {
            loginUserInfo = jwtHelper.getLoginUserInfo(token);
            if (null != loginUserInfo) {
                // 刷新缓存
                jwtHelper.setLoginUserInfo(token, loginUserInfo);
            }
        }
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response,
                                Object handler, Exception ex) {
        LoginUserContextHolder.remove();
    }

    /**
     * 刷新token缓存过期时间
     */
    private void resetCacheExpire(String token, String userInfo) {
        redisClient.setStr(CoreConstant.REDIS_TOKEN_USER_KEY + token, userInfo, CoreConstant.REDIS_TOKEN_EXPIRE_SECONDS);
    }
}
