/**
 * @(#)DrmSharerConfig.java, 2025/3/7.
 * <p/>
 * Copyright 2025 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.yanxuan.wx.store.sharer.biz.config;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ctrip.framework.apollo.model.ConfigChange;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.ctrip.framework.apollo.spring.annotation.EnableAutoUpdateApolloConfig;
import com.ctrip.framework.apollo.spring.annotation.ValueMapping;
import com.netease.yanxuan.wx.store.sharer.biz.meta.enums.CommissionChangeOptTypeEnum;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.bo.CommissionSharerBO;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.bo.CommissionSharerProductBO;
import com.netease.yanxuan.wx.store.sharer.dal.mapper.SharerInfoMapper;
import com.netease.yanxuan.wx.store.sharer.dal.mapper.SharerProductCommissionMapper;
import com.netease.yanxuan.wx.store.sharer.dal.mapper.SharerProductCommissionRecordMapper;
import com.netease.yanxuan.wx.store.sharer.dal.meta.model.po.SharerInfo;
import com.netease.yanxuan.wx.store.sharer.dal.meta.model.po.SharerProductCommission;
import com.netease.yanxuan.wx.store.sharer.dal.meta.model.po.SharerProductCommissionRecord;
import com.netease.yanxuan.wx.store.sharer.integration.constant.WeChatApi;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopSetSharerCommissionRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopSetSharerProductCommissionRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopSharerListRequest;
import com.netease.yanxuan.wx.store.sharer.integration.meta.enums.CommissionTypeEnum;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 推客配置apo
 *
 * @author 刘运星 (liuyunxing01@corp.netease.com)
 */
@Slf4j
@Data
@Component
@RequiredArgsConstructor
@EnableAutoUpdateApolloConfig("sharer-config")
public class DrmSharerConfig {
    private final SharerInfoMapper sharerInfoMapper;
    private final SharerProductCommissionMapper sharerProductCommissionMapper;
    private final SharerProductCommissionRecordMapper sharerProdutcCommissionRecordMapper;
    private final WeChatShopSharerListRequest weChatShopSharerListRequest;
    private final WeChatShopSetSharerCommissionRequest weChatShopSetSharerCommissionRequest;
    private final WeChatShopSetSharerProductCommissionRequest weChatShopSetSharerProductCommissionRequest;

    private final Map<String, SharerInfo> sharerInfoMap = new ConcurrentHashMap<>(0);

    /**
     * 默认分佣比例
     */
    @Value("${commissionratio.default:10}")
    private BigDecimal defaultCommissionRatio;
    /**
     * 用户分佣比例
     */
    @ValueMapping("${commissionratio.sharer:}")
    private List<CommissionSharerBO> commissionSharerList;
    /**
     * 用户商品分佣比例
     */
    @ValueMapping("${commissionratio.sharerproduct:}")
    private List<CommissionSharerProductBO> commissionSharerProductList;

    /**
     * 监听
     */
    @ApolloConfigChangeListener("sharer-config")
    public void onChange(ConfigChangeEvent changeEvent) {
        log.info("ProductCommissionConfig:onChange, changeEvent = {}", JSON.toJSONString(changeEvent));

        // 默认分佣比例
        String defaultCommissionratio = "commissionratio.default";
        if (changeEvent.isChanged(defaultCommissionratio)) {
            log.info("ProductCommissionConfig:onChange, changeKey = {}", defaultCommissionratio);
            ConfigChange change = changeEvent.getChange(defaultCommissionratio);
            defaultCommissionratioChange(change);
        }

        // 用户分佣比例
        String sharerCommissionratio = "commissionratio.sharer";
        if (changeEvent.isChanged(sharerCommissionratio)) {
            log.info("ProductCommissionConfig:onChange, changeKey = {}", sharerCommissionratio);
            ConfigChange change = changeEvent.getChange(sharerCommissionratio);
            sharerCommissionRatioChange(change);
        }

        // 用户商品分佣比例
        String sharerproductCommissionratio = "commissionratio.sharerproduct";
        if (changeEvent.isChanged(sharerproductCommissionratio)) {
            log.info("ProductCommissionConfig:onChange, changeKey = {}", sharerproductCommissionratio);
            ConfigChange change = changeEvent.getChange(sharerproductCommissionratio);
            sharerProductCommissionRatioChange(change);
        }
    }

    /**
     * 默认分佣比例变更
     */
    private void defaultCommissionratioChange(ConfigChange change) {
        log.info("ProductCommissionConfig:defaultCommissionratioChange, change = {}", JSON.toJSONString(change));
        if (StringUtils.isBlank(change.getNewValue()) || change.getNewValue().equals(change.getOldValue())) {
            return;
        }
        int pageNum = 1;
        int pageSize = 10;
        Page<SharerInfo> sharerInfoPage;
        while (true) {
            sharerInfoPage = sharerInfoMapper.selectPage(new Page<>(pageNum++, pageSize), null);
            if (null == sharerInfoPage || CollectionUtils.isEmpty(sharerInfoPage.getRecords())) {
                // 如果返回值为空，直接终止循环
                break;
            }
            for (SharerInfo sharerInfo : sharerInfoPage.getRecords()) {
                // 处理每个 sharerInfoVO
                setSharerDefaultCommissionInfo(change, sharerInfo);
            }
        }
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerDefaultCommissionInfo(ConfigChange change, SharerInfo sharerInfo) {
        // 分佣比例为小数,  微信接口参数应当分佣比例*倍数 为整数
        BigDecimal commissionRatio = new BigDecimal(change.getNewValue())
                .multiply(new BigDecimal(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER))
                .setScale(0, RoundingMode.DOWN);
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            if (StringUtils.isNotBlank(sharerInfo.getSharerAppid())) {
                weChatShopSetSharerCommissionRequest.handle(sharerInfo.getSharerAppid(), commissionRatio.longValue());
            }
            updateSharerInfo(sharerInfo, change.getNewValue());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerInfo.getOpenId(), sharerInfo.getSharerAppid(), null,
                    new BigDecimal(change.getOldValue()), new BigDecimal(change.getNewValue()),
                    CommissionChangeOptTypeEnum.DEFAULT_CHANGE, isSuccess);
        }
    }

    /**
     * 用户分佣比例变更
     */
    private void sharerCommissionRatioChange(ConfigChange change) {
        log.info("ProductCommissionConfig:sharerCommissionRatioChange, change = {}", JSON.toJSONString(change));
        List<CommissionSharerBO> oldCommissionSharerList = JSON.parseArray(change.getOldValue(), CommissionSharerBO.class);
        if (null == oldCommissionSharerList) {
            oldCommissionSharerList = new ArrayList<>();
        }
        List<CommissionSharerBO> newCommissionSharerList = JSON.parseArray(change.getNewValue(), CommissionSharerBO.class);
        if (null == newCommissionSharerList) {
            newCommissionSharerList = Collections.emptyList();
        }
        // 将 newCommissionSharerList 转换为 Map，以减少流操作的开销
        Map<String, CommissionSharerBO> newCommissionSharerMap = newCommissionSharerList.stream()
                .collect(Collectors.toMap(CommissionSharerBO::getOpenId, Function.identity()));
        for (CommissionSharerBO oldInfo : oldCommissionSharerList) {
            CommissionSharerBO newInfo = newCommissionSharerMap.get(oldInfo.getOpenId());
            if (newInfo != null) {
                // 修改
                if (newInfo.getCommissionRatio().compareTo(oldInfo.getCommissionRatio()) != 0) {
                    setSharerCommissionInfo(newInfo, oldInfo);
                }
            } else {
                // 删除
                deleteSharerCommissionInfo(oldInfo);
            }
        }
        for (CommissionSharerBO newInfo : newCommissionSharerList) {
            // 新增
            if (oldCommissionSharerList.stream().noneMatch(item -> item.getOpenId().equals(newInfo.getOpenId()))) {
                setSharerCommissionInfo(newInfo, null);
            }
        }
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerCommissionInfo(CommissionSharerBO newCommissionSharer,
                                         CommissionSharerBO oldCommissionSharer) {
        SharerInfo sharerInfo = getSharerInfoByOpenId(newCommissionSharer.getOpenId());
        if (null == sharerInfo) {
            return;
        }
        // 分佣比例为小数,  微信接口参数应当分佣比例*1000 为整数
        BigDecimal commissionRatio = newCommissionSharer.getCommissionRatio()
                .multiply(new BigDecimal(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER))
                .setScale(0, RoundingMode.DOWN);
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            if (StringUtils.isNotBlank(sharerInfo.getSharerAppid())) {
                // 设置推客分佣比例
                weChatShopSetSharerCommissionRequest.handle(sharerInfo.getSharerAppid(), commissionRatio.longValue());
            }
            // 更新推客分佣比例
            updateSharerInfo(sharerInfo, newCommissionSharer.getCommissionRatio()
                    .setScale(2, RoundingMode.HALF_UP).toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerInfo.getOpenId(), sharerInfo.getSharerAppid(), null,
                    oldCommissionSharer.getCommissionRatio(), newCommissionSharer.getCommissionRatio(),
                    CommissionChangeOptTypeEnum.SET, isSuccess);
        }
    }

    /**
     * 删除推客的的分佣类型和比例信息
     */
    private void deleteSharerCommissionInfo(CommissionSharerBO oldCommissionSharer) {
        SharerInfo sharerInfo = getSharerInfoByOpenId(oldCommissionSharer.getOpenId());
        if (null == sharerInfo) {
            return;
        }
        // 分佣比例为小数,  微信接口参数应当分佣比例*倍数 为整数
        BigDecimal sharerCommissionRatio = getSharerCommissionRatio(oldCommissionSharer.getOpenId());
        BigDecimal commissionRatio = sharerCommissionRatio
                .multiply(new BigDecimal(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER))
                .setScale(0, RoundingMode.DOWN);
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            if (StringUtils.isNotBlank(sharerInfo.getSharerAppid())) {
                // 设置推客分佣比例
                weChatShopSetSharerCommissionRequest.handle(sharerInfo.getSharerAppid(), commissionRatio.longValue());
            }
            // 更新推客分佣比例
            updateSharerInfo(sharerInfo, sharerCommissionRatio.setScale(2, RoundingMode.HALF_UP).toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerInfo.getOpenId(), sharerInfo.getSharerAppid(), null,
                    oldCommissionSharer.getCommissionRatio(), sharerCommissionRatio,
                    CommissionChangeOptTypeEnum.UNSET, isSuccess);
        }
    }

    /**
     * 用户商品分佣比例变更
     */
    private void sharerProductCommissionRatioChange(ConfigChange change) {
        log.info("ProductCommissionConfig:sharerProductCommissionRatioChange, change = {}", JSON.toJSONString(change));
        List<CommissionSharerProductBO> oldCommissionSharerList = JSON.parseArray(change.getOldValue(), CommissionSharerProductBO.class);
        if (null == oldCommissionSharerList) {
            oldCommissionSharerList = new ArrayList<>();
        }
        List<CommissionSharerProductBO> newCommissionSharerList = JSON.parseArray(change.getNewValue(), CommissionSharerProductBO.class);
        if (null == newCommissionSharerList) {
            newCommissionSharerList = Collections.emptyList();
        }
        // 将 newCommissionSharerList 转换为 Map，以减少流操作的开销
        Map<String, CommissionSharerProductBO> newCommissionSharerMap = newCommissionSharerList.stream()
                .collect(Collectors.toMap(this::getSharerProductCommissionMapKey, Function.identity()));
        for (CommissionSharerProductBO oldInfo : oldCommissionSharerList) {
            CommissionSharerProductBO newInfo = newCommissionSharerMap.get(getSharerProductCommissionMapKey(oldInfo));
            if (newInfo != null) {
                // 修改
                if (newInfo.getCommissionRatio().compareTo(oldInfo.getCommissionRatio()) != 0) {
                    setSharerProductCommissionInfo(newInfo, oldInfo);
                }
            } else {
                // 删除
                deleteSharerProductCommissionInfo(oldInfo);
            }
        }
        for (CommissionSharerProductBO newInfo : newCommissionSharerList) {
            // 新增
            if (oldCommissionSharerList.stream().noneMatch(item -> item.getOpenId().equals(newInfo.getOpenId())
                    && item.getProductId().equals(newInfo.getProductId()))) {
                setSharerProductCommissionInfo(newInfo, null);
            }
        }
    }

    private String getSharerProductCommissionMapKey(CommissionSharerProductBO bo) {
        return bo.getOpenId() + "&" + bo.getProductId();
    }

    /**
     * 获取推客分佣比例
     */
    private BigDecimal getSharerCommissionRatio(String openId, Long productId) {
        Optional<CommissionSharerProductBO> sharerProductCommissionOpt = Optional.ofNullable(commissionSharerProductList)
                .orElseGet(ArrayList::new)
                .stream()
                .filter(item -> item.getOpenId().equals(openId))
                .filter(item -> null == productId || item.getProductId().equals(String.valueOf(productId)))
                .findFirst();
        if (sharerProductCommissionOpt.isPresent()) {
            // 返回推客商品分佣
            return sharerProductCommissionOpt.get().getCommissionRatio();
        }
        return getSharerCommissionRatio(openId);
    }

    /**
     * 获取推客分佣比例
     */
    private BigDecimal getSharerCommissionRatio(String openId) {
        Optional<CommissionSharerBO> sharerCommissionOpt = Optional.ofNullable(commissionSharerList)
                .orElseGet(ArrayList::new)
                .stream()
                .filter(item -> item.getOpenId().equals(openId))
                .findFirst();
        if (sharerCommissionOpt.isPresent()) {
            // 返回推客分佣
            return sharerCommissionOpt.get().getCommissionRatio();
        }
        // 返回默认分佣金
        return defaultCommissionRatio;
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerProductCommissionInfo(CommissionSharerProductBO newBo,
                                                CommissionSharerProductBO oldBo) {
        SharerInfo sharerInfo = getSharerInfoByOpenId(newBo.getOpenId());
        if (null == sharerInfo) {
            return;
        }
        // 分佣比例为小数,  微信接口参数应当分佣比例*倍数 为整数
        BigDecimal commissionRatio = newBo.getCommissionRatio()
                .multiply(new BigDecimal(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER))
                .setScale(0, RoundingMode.DOWN);
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            if (StringUtils.isNotBlank(sharerInfo.getSharerAppid())) {
                // 设置推客分佣比例
                weChatShopSetSharerProductCommissionRequest.handle(sharerInfo.getSharerAppid(),
                        Long.valueOf(newBo.getProductId()),
                        commissionRatio.longValue());
            }
            // 保存推客商品记录
            saveSharerProductCommissionInfo(sharerInfo.getOpenId(), newBo.getProductId(),
                    commissionRatio, SharerProductCommission.UnsetEnum.SET);
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerInfo.getOpenId(),
                    sharerInfo.getSharerAppid(), newBo.getProductId(),
                    null != oldBo ? oldBo.getCommissionRatio() : null,
                    newBo.getCommissionRatio(),
                    CommissionChangeOptTypeEnum.SET, isSuccess);
        }
    }

    /**
     * 删除推客的的分佣类型和比例信息
     */
    private void deleteSharerProductCommissionInfo(CommissionSharerProductBO oldBo) {
        SharerInfo sharerInfo = getSharerInfoByOpenId(oldBo.getOpenId());
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            if (StringUtils.isNotBlank(sharerInfo.getSharerAppid())) {
                // 设置推客分佣比例
                weChatShopSetSharerProductCommissionRequest.handle(sharerInfo.getSharerAppid(),
                        Long.valueOf(oldBo.getProductId()), null);
            }
            // 获取推客的分佣比例
            BigDecimal sharerCommissionRatio = getSharerCommissionRatio(sharerInfo.getOpenId(), null);
            // 保存推客商品记录
            saveSharerProductCommissionInfo(sharerInfo.getOpenId(), oldBo.getProductId(),
                    sharerCommissionRatio, SharerProductCommission.UnsetEnum.UNSET);
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerInfo.getOpenId(),
                    sharerInfo.getSharerAppid(), oldBo.getProductId(),
                    oldBo.getCommissionRatio(), null,
                    CommissionChangeOptTypeEnum.UNSET, isSuccess);
        }
    }


    /**
     * 保存推客商品分佣记录
     */
    private void saveSharerProductCommissionRecord(String openId, String sharerAppid, String productId,
                                                   BigDecimal oldCommissionRatio, BigDecimal newCommissionRatio,
                                                   CommissionChangeOptTypeEnum optTypeEnum, boolean isSuccess) {
        Date now = new Date();
        SharerProductCommissionRecord record = new SharerProductCommissionRecord();
        record.setOpenId(openId);
        record.setSharerAppid(sharerAppid);
        record.setProductId(productId);
        record.setOptType(optTypeEnum.getCode());
        record.setOptInfo((null == oldCommissionRatio ? "" : oldCommissionRatio.toPlainString()) + "&"
                + (null == newCommissionRatio ? "" : newCommissionRatio.toPlainString()));
        record.setOptTime(now);
        record.setOptStatus(isSuccess ? "1" : "2");
        record.setExtInfo("");
        record.setCreateTime(now);
        record.setUpdateTime(now);
        sharerProdutcCommissionRecordMapper.insert(record);
    }

    /**
     * 保存推客商品分佣金表
     */
    private void saveSharerProductCommissionInfo(String openId,
                                                 String productId,
                                                 BigDecimal commissionRatio,
                                                 SharerProductCommission.UnsetEnum unsetEnum) {
        String sharerAppid = getSharerAppidByOpenId(openId);
        if (StringUtils.isBlank(sharerAppid)) {
            log.error("微信用户暂未绑定微信小店，openId={}", openId);
            return;
        }
        LambdaQueryWrapper<SharerProductCommission> lqw = Wrappers.lambdaQuery();
        lqw.eq(SharerProductCommission::getOpenId, openId);
        lqw.eq(SharerProductCommission::getProductId, productId);
        lqw.last("LIMIT 1");
        SharerProductCommission sharerProductCommission = sharerProductCommissionMapper.selectOne(lqw);
        Date now = new Date();
        if (null == sharerProductCommission) {
            sharerProductCommission = new SharerProductCommission();
            sharerProductCommission.setOpenId(openId);
            sharerProductCommission.setProductId(productId);
            sharerProductCommission.setCreateTime(now);
        }
        sharerProductCommission.setSharerAppid(sharerAppid);
        sharerProductCommission.setCommissionRatio(commissionRatio);
        sharerProductCommission.setCommissionType(CommissionTypeEnum.PLATFORM.getCode().toString());
        sharerProductCommission.setUnset(unsetEnum.getCode());
        sharerProductCommission.setUpdateTime(now);
        if (null == sharerProductCommission.getId()) {
            sharerProductCommissionMapper.insert(sharerProductCommission);
        } else {
            sharerProductCommissionMapper.updateById(sharerProductCommission);
        }
    }

    /**
     * 根据 openId 获取推客信息
     */
    private SharerInfo getSharerInfoByOpenId(String openId) {
        SharerInfo sharerInfo = sharerInfoMap.get(openId);
        if (null == sharerInfo) {
            LambdaQueryWrapper<SharerInfo> sharerInfoLqw = Wrappers.lambdaQuery();
            sharerInfoLqw.eq(SharerInfo::getOpenId, openId);
            sharerInfoLqw.last("LIMIT 1");
            sharerInfo = sharerInfoMapper.selectOne(sharerInfoLqw);
            if (null != sharerInfo) {
                sharerInfoMap.put(openId, sharerInfo);
            }
        }
        return sharerInfo;
    }

    /**
     * 根据 openId 获取推客的 SharerAppid
     */
    private String getSharerAppidByOpenId(String openId) {
        SharerInfo sharerInfo = sharerInfoMap.get(openId);
        if (null == sharerInfo) {
            LambdaQueryWrapper<SharerInfo> sharerInfoLqw = Wrappers.lambdaQuery();
            sharerInfoLqw.eq(SharerInfo::getOpenId, openId);
            sharerInfoLqw.last("LIMIT 1");
            sharerInfo = sharerInfoMapper.selectOne(sharerInfoLqw);
            if (null != sharerInfo) {
                sharerInfoMap.put(openId, sharerInfo);
            }
        }
        return null != sharerInfo ? sharerInfo.getSharerAppid() : null;
    }

    private void updateSharerInfo(SharerInfo sharerInfo, String commissionRatio) {
        String openId = sharerInfo.getOpenId();
        if (StringUtils.isNotBlank(openId)) {
            BigDecimal sharerCommissionRatio = getSharerCommissionRatio(openId);
            commissionRatio = sharerCommissionRatio.setScale(2, RoundingMode.HALF_UP).toPlainString();
        }
        LambdaUpdateWrapper<SharerInfo> sharerInfoLuw = Wrappers.lambdaUpdate();
        sharerInfoLuw.set(SharerInfo::getCommissionRatio, commissionRatio);
        sharerInfoLuw.set(SharerInfo::getUpdateTime, new Date());
        sharerInfoLuw.eq(SharerInfo::getId, sharerInfo.getId());
        sharerInfoMapper.update(null, sharerInfoLuw);
    }
}