package com.netease.yanxuan.wx.store.sharer.common.util;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;

/**
 * @Description 异步处理工具类
 * @Author fanjiaxin
 * @Date 2024/6/13 09:31
 */
public class AsyncUtils {

    /**
     * 异步处理列表中的每个元素，并收集所有结果。
     *
     * @param <T>       列表中的元素类型
     * @param <R>       异步处理结果类型
     * @param itemList  要处理的列表
     * @param asyncTask 定义如何异步处理每个元素的函数
     * @return 异步处理后所有结果的列表
     */
    public static <T, R> List<R> processListAsync(List<T> itemList, Function<T, R> asyncTask) {
        CompletableFuture<R>[] futures = new CompletableFuture[itemList.size()];

        for (int i = 0; i < itemList.size(); i++) {
            T item = itemList.get(i);
            CompletableFuture<R> future = CompletableFuture.supplyAsync(() -> asyncTask.apply(item));
            futures[i] = future;
        }

        CompletableFuture.allOf(futures).join();

        List<R> resultList = new ArrayList<>(itemList.size());
        for (CompletableFuture<R> future : futures) {
            resultList.add(future.join());
        }

        return resultList;
    }

    /**
     * 异步处理列表中的每个元素，并收集所有结果。
     *
     * @param <T>       列表中的元素类型
     * @param <R>       异步处理结果类型
     * @param itemList  要处理的列表
     * @param asyncTask 定义如何异步处理每个元素的函数
     * @param executor  异步执行器
     * @return 异步处理后所有结果的列表
     */
    public static <T, R> List<R> processListAsync(List<T> itemList, Function<T, R> asyncTask, Executor executor) {
        CompletableFuture<R>[] futures = new CompletableFuture[itemList.size()];

        for (int i = 0; i < itemList.size(); i++) {
            T item = itemList.get(i);
            CompletableFuture<R> future = CompletableFuture.supplyAsync(() -> asyncTask.apply(item), executor);
            futures[i] = future;
        }

        CompletableFuture.allOf(futures).join();

        List<R> resultList = new ArrayList<>(itemList.size());
        for (CompletableFuture<R> future : futures) {
            resultList.add(future.join());
        }

        return resultList;
    }
}
