package com.netease.yanxuan.wx.store.sharer.biz.service.impl;

import com.netease.yanxuan.wx.store.sharer.biz.config.DrmSharerConfig;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserContextHolder;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserInfo;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.vo.ProductListVO;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.vo.ProductPromotionLinkVO;
import com.netease.yanxuan.wx.store.sharer.biz.meta.page.PageQuery;
import com.netease.yanxuan.wx.store.sharer.biz.meta.page.PageVO;
import com.netease.yanxuan.wx.store.sharer.biz.service.IProductService;
import com.netease.yanxuan.wx.store.sharer.common.exception.NoBindSharerException;
import com.netease.yanxuan.wx.store.sharer.common.util.AsyncUtils;
import com.netease.yanxuan.wx.store.sharer.integration.constant.WeChatApi;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopPromoteProductDetailRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopPromoteProductLinkRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopPromoteProductListRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopSharerListRequest;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @Description 商品-业务实现类
 * @Author fanjiaxin
 * @Date 2025/3/10 12:31
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class ProductServiceImpl implements IProductService {
    private final WeChatShopPromoteProductListRequest weChatShopPromoteProductListRequest;
    private final WeChatShopPromoteProductDetailRequest weChatShopPromoteProductDetailRequest;
    private final WeChatShopPromoteProductLinkRequest weChatShopPromoteProductLinkRequest;
    private final WeChatShopSharerListRequest weChatShopSharerListRequest;
    private final DrmSharerConfig drmSharerConfig;


    @Override
    public PageVO<ProductListVO> getProductPageList(PageQuery pageQuery, String keyword) {
        PageVO<ProductListVO> pageVO = new PageVO<>();
        WeChatPromoteProductListVO productListVO = weChatShopPromoteProductListRequest.handle(keyword,
                pageQuery.getNextKey(), pageQuery.getPageSize());
        pageVO.setHasMore(false);
        // 商品列表
        if (!CollectionUtils.isEmpty(productListVO.getProduct_list())) {
            // 使用异步多线程处理
            List<ProductListVO> productList = AsyncUtils.processListAsync(productListVO.getProduct_list(), item -> {
                WeChatPromoteProductDetailVO detailVO = weChatShopPromoteProductDetailRequest.handle(item.getShop_appid(),
                        Long.valueOf(item.getProduct_id()));
                if (null == detailVO) {
                    return null;
                }
                WeChatPromoteProductDetailVO.WeChatPromoteProductVO product = detailVO.getProduct();
                if (null == product) {
                    return null;
                }
                WeChatPromoteProductDetailVO.WeChatPromoteProductVO.WeChatPromoteProductInfoVO productInfo = product.getProduct_info();
                if (null == productInfo) {
                    return null;
                }
                ProductListVO listVO = new ProductListVO();
                listVO.setTitle(productInfo.getTitle());
                listVO.setHeadImg(CollectionUtils.isEmpty(productInfo.getHead_imgs()) ? null : productInfo.getHead_imgs().get(0));
                // 取参数SkuInfo中sale_price的最小
                BigDecimal price = null;
                Optional<BigDecimal> priceOpt = Optional.ofNullable(productInfo.getSkus()).orElseGet(ArrayList::new)
                        .stream()
                        .map(WeChatPromoteProductDetailVO.WeChatPromoteProductVO.WeChatPromoteProductInfoVO.WeChatPromoteProductSkuInfoVO::getSale_price)
                        .min(BigDecimal::compareTo);
                if (priceOpt.isPresent()) {
                    // 以分为单位，进行元转换
                    price = priceOpt.get().divide(new BigDecimal("100"));
                    listVO.setPrice(price.setScale(2, RoundingMode.HALF_UP));
                }
                // 推客分佣比例
                BigDecimal commissionRatio = drmSharerConfig.getDefaultCommissionRatio();
                log.info("默认商品佣金比例:{}", commissionRatio);
                // 获取用户信息
                LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
                if (null != loginUserInfo && StringUtils.isNotBlank(loginUserInfo.getOpenId())) {
                    WeChatSharerListVO sharerListVO = weChatShopSharerListRequest.handle(loginUserInfo.getOpenId(), "", 1);
                    if (null != sharerListVO && !CollectionUtils.isEmpty(sharerListVO.getSharer_info_list())) {
                        WeChatSharerInfoVO sharerInfoVO = sharerListVO.getSharer_info_list().get(0);
                        if (null != sharerInfoVO && null != sharerInfoVO.getCommission_ratio()) {
                            log.info("微信返回商品佣金比例:{}", sharerInfoVO.getCommission_ratio());
                            // 微信分佣比例倍数
                            BigDecimal wechatMultiplier = new BigDecimal(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER);
                            commissionRatio = sharerInfoVO.getCommission_ratio().divide(wechatMultiplier);
                        }
                    }
                }
                WeChatPromoteProductDetailVO.WeChatPromoteProductVO.WeChatPromoteProductCommissionInfoVO commissionInfo
                        = product.getCommission_info();
                // 服务费率
                BigDecimal serviceRatio = null;
                if (null != commissionInfo && null != commissionInfo.getService_ratio()) {
                    serviceRatio = new BigDecimal(commissionInfo.getService_ratio())
                            .divide(new BigDecimal(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER));
                }
                // 佣金=商品价格*服务费率*推客分佣比例，四舍五入保留小数点后两位
                BigDecimal commission = BigDecimal.ZERO;
                if (null != price && null != commissionRatio && null != serviceRatio) {
                    commission = price.multiply(serviceRatio).multiply(commissionRatio)
                            .setScale(2, RoundingMode.HALF_UP);
                    log.info("计算佣金，价格：{}，服务费率：{}，推客分佣比例：{}，佣金：{}", price, serviceRatio, commissionRatio, commission);
                }
                listVO.setCommission(commission);
                listVO.setShopAppid(item.getShop_appid());
                listVO.setProductId(item.getProduct_id());
                listVO.setProductPromotionLink(productInfo.getProduct_promotion_link());
                return listVO;
            });
            // 过滤空
            productList = Optional.of(productList).orElseGet(ArrayList::new)
                    .stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
            pageVO.setList(productList);
            pageVO.setHasMore(!CollectionUtils.isEmpty(productList)
                    && productList.size() >= pageQuery.getPageSize());
            pageVO.setNextKey(productListVO.getNext_key());
        }
        return pageVO;
    }

    @Override
    public ProductPromotionLinkVO getProductPromotionLink(String shopAppid, String productId) {
        // 获取用户信息
        LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
        if (StringUtils.isBlank(loginUserInfo.getSharerAppId())) {
            throw new NoBindSharerException("未绑定推客");
        }
        WeChatPromoteProductLinkVO promoteProductLinkVO = weChatShopPromoteProductLinkRequest.handle(
                loginUserInfo.getSharerAppId(),
                Long.valueOf(productId),
                shopAppid);
        if (null != promoteProductLinkVO && StringUtils.isNotBlank(promoteProductLinkVO.getShort_link())) {
            ProductPromotionLinkVO result = new ProductPromotionLinkVO();
            result.setPromotionLink(promoteProductLinkVO.getShort_link());
            return result;
        }
        return null;
    }
}
