/**
 * @(#)DrmSharerConfig.java, 2025/3/7.
 * <p/>
 * Copyright 2025 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.yanxuan.wx.store.sharer.biz.config;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ctrip.framework.apollo.model.ConfigChange;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.ctrip.framework.apollo.spring.annotation.EnableAutoUpdateApolloConfig;
import com.ctrip.framework.apollo.spring.annotation.ValueMapping;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserContextHolder;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserInfo;
import com.netease.yanxuan.wx.store.sharer.biz.meta.enums.CommissionChangeOptTypeEnum;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.bo.CommissionSharerBO;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.bo.CommissionSharerProductBO;
import com.netease.yanxuan.wx.store.sharer.dal.mapper.SharerInfoMapper;
import com.netease.yanxuan.wx.store.sharer.dal.mapper.SharerProductCommissionMapper;
import com.netease.yanxuan.wx.store.sharer.dal.mapper.SharerProductCommissionRecordMapper;
import com.netease.yanxuan.wx.store.sharer.dal.meta.model.po.SharerInfo;
import com.netease.yanxuan.wx.store.sharer.dal.meta.model.po.SharerProductCommission;
import com.netease.yanxuan.wx.store.sharer.dal.meta.model.po.SharerProductCommissionRecord;
import com.netease.yanxuan.wx.store.sharer.integration.constant.WeChatApi;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopSetSharerCommissionRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopSetSharerProductCommissionRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopSharerListRequest;
import com.netease.yanxuan.wx.store.sharer.integration.meta.enums.CommissionTypeEnum;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatSharerInfoVO;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatSharerListVO;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 推客配置apo
 *
 * @author 刘运星 (liuyunxing01@corp.netease.com)
 */
@Slf4j
@Data
@Component
@RequiredArgsConstructor
@EnableAutoUpdateApolloConfig("sharer-config")
public class DrmSharerConfig {
    private final SharerInfoMapper sharerInfoMapper;
    private final SharerProductCommissionMapper sharerProductCommissionMapper;
    private final SharerProductCommissionRecordMapper sharerProdutcCommissionRecordMapper;
    private final WeChatShopSharerListRequest weChatShopSharerListRequest;
    private final WeChatShopSetSharerCommissionRequest weChatShopSetSharerCommissionRequest;
    private final WeChatShopSetSharerProductCommissionRequest weChatShopSetSharerProductCommissionRequest;

    private final Map<String, String> sharerAppidMap = new ConcurrentHashMap<>(0);

    /**
     * 默认分佣比例
     */
    @Value("${commissionratio.default:10}")
    private BigDecimal defaultCommissionRatio;
    /**
     * 用户分佣比例
     */
    @ValueMapping("${commissionratio.sharer:}")
    private List<CommissionSharerBO> commissionSharerList;
    /**
     * 用户商品分佣比例
     */
    @ValueMapping("${commissionratio.sharerproduct:}")
    private List<CommissionSharerProductBO> commissionSharerProductList;

    /**
     * 监听
     */
    @ApolloConfigChangeListener("sharer-config")
    public void onChange(ConfigChangeEvent changeEvent) {
        log.info("ProductCommissionConfig:onChange, changeEvent = {}", JSON.toJSONString(changeEvent));

        // 默认分佣比例
        String defaultCommissionratio = "application.commissionratio.default";
        if (changeEvent.isChanged(defaultCommissionratio)) {
            ConfigChange change = changeEvent.getChange(defaultCommissionratio);
            defaultCommissionratioChange(change);
        }

        // 用户分佣比例
        String sharerCommissionratio = "application.commissionratio.sharer";
        if (changeEvent.isChanged(sharerCommissionratio)) {
            ConfigChange change = changeEvent.getChange(sharerCommissionratio);
            sharerCommissionRatioChange(change);
        }

        // 用户商品分佣比例
        String sharerproductCommissionratio = "application.commissionratio.sharerproduct";
        if (changeEvent.isChanged(sharerproductCommissionratio)) {
            ConfigChange change = changeEvent.getChange(sharerproductCommissionratio);
            sharerProductCommissionRatioChange(change);
        }
    }

    /**
     * 默认分佣比例变更
     */
    private void defaultCommissionratioChange(ConfigChange change) {
        if (StringUtils.isBlank(change.getNewValue()) || change.getNewValue().equals(change.getOldValue())) {
            return;
        }
        // 获取推客绑定的所有用户
        String nextKey = null;
        int pageSize = 10;
        while (true) {
            WeChatSharerListVO sharerListVO = weChatShopSharerListRequest.handle(null, nextKey, pageSize);
            if (sharerListVO == null) {
                break; // 如果返回值为空，直接终止循环
            }
            nextKey = sharerListVO.getNext_key();
            if (!CollectionUtils.isEmpty(sharerListVO.getSharer_info_list())) {
                for (WeChatSharerInfoVO sharerInfoVO : sharerListVO.getSharer_info_list()) {
                    // 处理每个 sharerInfoVO
                    setSharerDefaultCommissionInfo(change, sharerInfoVO);
                }
            }
            if (CollectionUtils.isEmpty(sharerListVO.getSharer_info_list()) || sharerListVO.getSharer_info_list().size() < pageSize) {
                break; // 如果列表为空或小于 pageSize，终止循环
            }
        }
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerDefaultCommissionInfo(ConfigChange change, WeChatSharerInfoVO sharerInfoVO) {
        // 分佣比例为小数,  微信接口参数应当分佣比例*倍数 为整数
        BigDecimal commissionRatio = new BigDecimal(change.getNewValue()).divide(BigDecimal.valueOf(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER));
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatShopSetSharerCommissionRequest.handle(sharerInfoVO.getSharer_appid(), commissionRatio.toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerInfoVO.getSharer_appid(), null,
                    new BigDecimal(change.getOldValue()), new BigDecimal(change.getNewValue()),
                    CommissionChangeOptTypeEnum.DEFAULT_CHANGE, isSuccess);
        }
    }

    /**
     * 用户分佣比例变更
     */
    private void sharerCommissionRatioChange(ConfigChange change) {
        List<CommissionSharerBO> oldCommissionSharerList = JSON.parseArray(change.getOldValue(), CommissionSharerBO.class);
        if (null == oldCommissionSharerList) {
            oldCommissionSharerList = new ArrayList<>();
        }
        List<CommissionSharerBO> newCommissionSharerList = JSON.parseArray(change.getNewValue(), CommissionSharerBO.class);
        if (null == newCommissionSharerList) {
            newCommissionSharerList = Collections.emptyList();
        }
        // 将 newCommissionSharerList 转换为 Map，以减少流操作的开销
        Map<String, CommissionSharerBO> newCommissionSharerMap = newCommissionSharerList.stream()
                .collect(Collectors.toMap(CommissionSharerBO::getOpenId, Function.identity()));
        for (CommissionSharerBO oldInfo : oldCommissionSharerList) {
            CommissionSharerBO newInfo = newCommissionSharerMap.get(oldInfo.getOpenId());
            if (newInfo != null) {
                // 修改
                if (newInfo.getCommissionRatio().compareTo(oldInfo.getCommissionRatio()) != 0) {
                    setSharerCommissionInfo(newInfo, oldInfo);
                }
            } else {
                // 删除
                deleteSharerCommissionInfo(oldInfo);
            }
        }
        for (CommissionSharerBO newInfo : newCommissionSharerList) {
            // 新增
            if (oldCommissionSharerList.stream().noneMatch(item -> item.getOpenId().equals(newInfo.getOpenId()))) {
                setSharerCommissionInfo(newInfo, null);
            }
        }
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerCommissionInfo(CommissionSharerBO newCommissionSharer,
                                         CommissionSharerBO oldCommissionSharer) {
        String sharerAppid = getSharerAppidByOpenId(newCommissionSharer.getOpenId());
        if (StringUtils.isBlank(sharerAppid)) {
            log.error("微信用户暂未绑定微信小店，openId={}", newCommissionSharer.getOpenId());
            return;
        }
        // 分佣比例为小数,  微信接口参数应当分佣比例*1000 为整数
        BigDecimal commissionRatio = newCommissionSharer.getCommissionRatio().divide(BigDecimal.valueOf(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER));
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatShopSetSharerCommissionRequest.handle(sharerAppid, commissionRatio.toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerAppid, null,
                    oldCommissionSharer.getCommissionRatio(), newCommissionSharer.getCommissionRatio(),
                    CommissionChangeOptTypeEnum.SET, isSuccess);
        }
    }

    /**
     * 删除推客的的分佣类型和比例信息
     */
    private void deleteSharerCommissionInfo(CommissionSharerBO oldCommissionSharer) {
        String sharerAppid = getSharerAppidByOpenId(oldCommissionSharer.getOpenId());
        if (StringUtils.isBlank(sharerAppid)) {
            log.error("微信用户暂未绑定微信小店，openId={}", oldCommissionSharer.getOpenId());
            return;
        }
        // 分佣比例为小数,  微信接口参数应当分佣比例*倍数 为整数
        BigDecimal sharerCommissionRatio = getSharerCommissionRatio(oldCommissionSharer.getOpenId(), null);
        BigDecimal commissionRatio = sharerCommissionRatio.divide(BigDecimal.valueOf(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER));
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatShopSetSharerCommissionRequest.handle(sharerAppid, commissionRatio.toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerAppid, null,
                    oldCommissionSharer.getCommissionRatio(), sharerCommissionRatio,
                    CommissionChangeOptTypeEnum.UNSET, isSuccess);
        }
    }

    /**
     * 用户商品分佣比例变更
     */
    private void sharerProductCommissionRatioChange(ConfigChange change) {
        List<CommissionSharerProductBO> oldCommissionSharerList = JSON.parseArray(change.getOldValue(), CommissionSharerProductBO.class);
        if (null == oldCommissionSharerList) {
            oldCommissionSharerList = new ArrayList<>();
        }
        List<CommissionSharerProductBO> newCommissionSharerList = JSON.parseArray(change.getNewValue(), CommissionSharerProductBO.class);
        if (null == newCommissionSharerList) {
            newCommissionSharerList = Collections.emptyList();
        }
        // 将 newCommissionSharerList 转换为 Map，以减少流操作的开销
        Map<String, CommissionSharerProductBO> newCommissionSharerMap = newCommissionSharerList.stream()
                .collect(Collectors.toMap(this::getSharerProductCommissionMapKey, Function.identity()));
        for (CommissionSharerProductBO oldInfo : oldCommissionSharerList) {
            CommissionSharerProductBO newInfo = newCommissionSharerMap.get(getSharerProductCommissionMapKey(oldInfo));
            if (newInfo != null) {
                // 修改
                if (newInfo.getCommissionRatio().compareTo(oldInfo.getCommissionRatio()) != 0) {
                    setSharerProductCommissionInfo(newInfo, oldInfo);
                    // 保存推客商品记录
                    saveSharerProductCommissionInfo(newInfo.getOpenId(), newInfo.getProductId(),
                            newInfo.getCommissionRatio(), SharerProductCommission.UnsetEnum.SET);
                }
            } else {
                // 删除
                deleteSharerProductCommissionInfo(oldInfo);
                // 获取推客的分佣比例
                BigDecimal sharerCommissionRatio = getSharerCommissionRatio(oldInfo.getOpenId(), null);
                // 保存推客商品记录
                saveSharerProductCommissionInfo(oldInfo.getOpenId(), oldInfo.getProductId(),
                        sharerCommissionRatio, SharerProductCommission.UnsetEnum.UNSET);
            }
        }
        for (CommissionSharerProductBO newInfo : newCommissionSharerList) {
            // 新增
            if (oldCommissionSharerList.stream().noneMatch(item -> item.getOpenId().equals(newInfo.getOpenId())
                    && item.getProductId().equals(newInfo.getProductId()))) {
                setSharerProductCommissionInfo(newInfo, null);
                // 保存推客商品记录
                saveSharerProductCommissionInfo(newInfo.getOpenId(), newInfo.getProductId(),
                        newInfo.getCommissionRatio(), SharerProductCommission.UnsetEnum.SET);
            }
        }
    }


    /**
     * 根据 openId 获取推客的 SharerAppid
     */
    private String getSharerAppidByOpenId(String openId) {
        String sharerAppid = sharerAppidMap.get(openId);
        if (StringUtils.isBlank(sharerAppid)) {
            LambdaQueryWrapper<SharerInfo> sharerInfoLqw = Wrappers.lambdaQuery();
            sharerInfoLqw.eq(SharerInfo::getOpenId, openId);
            sharerInfoLqw.last("LIMIT 1");
            SharerInfo sharerInfo = sharerInfoMapper.selectOne(sharerInfoLqw);
            if (null != sharerInfo && StringUtils.isNotBlank(sharerInfo.getSharerAppid())) {
                sharerAppid = sharerInfo.getSharerAppid();
                sharerAppidMap.put(openId, sharerAppid);
            }
        }
        return sharerAppid;
    }

    private String getSharerProductCommissionMapKey(CommissionSharerProductBO bo) {
        return bo.getOpenId() + "&" + bo.getProductId();
    }

    /**
     * 获取推客分佣比例
     */
    private BigDecimal getSharerCommissionRatio(String openId, Integer productId) {
        Optional<CommissionSharerProductBO> sharerProductCommissionOpt = Optional.ofNullable(commissionSharerProductList)
                .orElseGet(ArrayList::new)
                .stream()
                .filter(item -> item.getOpenId().equals(openId))
                .filter(item -> null == productId || item.getProductId().equals(productId))
                .findFirst();
        if(sharerProductCommissionOpt.isPresent()){
            // 返回推客商品分佣
            return sharerProductCommissionOpt.get().getCommissionRatio();
        }
        Optional<CommissionSharerBO> sharerCommissionOpt = Optional.ofNullable(commissionSharerList)
                .orElseGet(ArrayList::new)
                .stream()
                .filter(item -> item.getOpenId().equals(openId))
                .findFirst();
        if(sharerCommissionOpt.isPresent()){
            // 返回推客分佣
            return sharerCommissionOpt.get().getCommissionRatio();
        }
        // 返回默认分佣金
        return defaultCommissionRatio;
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerProductCommissionInfo(CommissionSharerProductBO newBo,
                                                CommissionSharerProductBO oldBo) {
        String sharerAppid = getSharerAppidByOpenId(newBo.getOpenId());
        if (StringUtils.isBlank(sharerAppid)) {
            log.error("微信用户暂未绑定微信小店，openId={}", newBo.getOpenId());
            return;
        }
        // 分佣比例为小数,  微信接口参数应当分佣比例*倍数 为整数
        BigDecimal commissionRatio = newBo.getCommissionRatio().divide(BigDecimal.valueOf(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER));
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatShopSetSharerProductCommissionRequest.handle(sharerAppid, Integer.valueOf(newBo.getProductId()), commissionRatio.toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerAppid, oldBo.getProductId(),
                    oldBo.getCommissionRatio(), newBo.getCommissionRatio(),
                    CommissionChangeOptTypeEnum.SET, isSuccess);
        }
    }

    /**
     * 删除推客的的分佣类型和比例信息
     */
    private void deleteSharerProductCommissionInfo(CommissionSharerProductBO oldBo) {
        String sharerAppid = getSharerAppidByOpenId(oldBo.getOpenId());
        if (StringUtils.isBlank(sharerAppid)) {
            log.error("微信用户暂未绑定微信小店，openId={}", oldBo.getOpenId());
            return;
        }
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatShopSetSharerProductCommissionRequest.handle(sharerAppid,
                    Integer.valueOf(oldBo.getProductId()), null);
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            // 保存记录
            saveSharerProductCommissionRecord(sharerAppid, oldBo.getProductId(),
                    oldBo.getCommissionRatio(), null,
                    CommissionChangeOptTypeEnum.UNSET, isSuccess);
        }
    }


    /**
     * 保存推客商品分佣记录
     */
    private void saveSharerProductCommissionRecord(String sharerAppid, String productId,
                                                   BigDecimal oldCommissionRatio,
                                                   BigDecimal newCommissionRatio,
                                                   CommissionChangeOptTypeEnum optTypeEnum,
                                                   boolean isSuccess) {
        LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
        if (null != loginUserInfo) {
            Date now = new Date();
            SharerProductCommissionRecord record = new SharerProductCommissionRecord();
            record.setOpenId(loginUserInfo.getOpenId());
            record.setSharerAppid(sharerAppid);
            record.setProductId(productId);
            record.setOptType(optTypeEnum.getCode());
            record.setOptInfo(oldCommissionRatio.toPlainString() + "&"
                    + (null == newCommissionRatio ? "" : newCommissionRatio.toPlainString()));
            record.setOptTime(now);
            record.setOptStatus(isSuccess ? "1" : "2");
            record.setExtInfo("");
            record.setCreateTime(now);
            record.setUpdateTime(now);
            sharerProdutcCommissionRecordMapper.insert(record);
        } else {
            log.error("用户登录状态过期，保存记录失败");
        }
    }

    /**
     * 保存推客商品分佣金表
     */
    private void saveSharerProductCommissionInfo(String openId,
                                                 String productId,
                                                 BigDecimal commissionRatio,
                                                 SharerProductCommission.UnsetEnum unsetEnum) {
        String sharerAppid = getSharerAppidByOpenId(openId);
        if (StringUtils.isBlank(sharerAppid)) {
            log.error("微信用户暂未绑定微信小店，openId={}", openId);
            return;
        }
        LambdaQueryWrapper<SharerProductCommission> lqw = Wrappers.lambdaQuery();
        lqw.eq(SharerProductCommission::getOpenId, openId);
        lqw.eq(SharerProductCommission::getProductId, productId);
        lqw.last("LIMIT 1");
        SharerProductCommission sharerProductCommission = sharerProductCommissionMapper.selectOne(lqw);
        Date now = new Date();
        if (null == sharerProductCommission) {
            sharerProductCommission = new SharerProductCommission();
            sharerProductCommission.setOpenId(openId);
            sharerProductCommission.setSharerAppid(sharerAppid);
            sharerProductCommission.setProductId(productId);
            sharerProductCommission.setCommissionRatio(commissionRatio);
            sharerProductCommission.setCommissionType(CommissionTypeEnum.PLATFORM.getCode());
            sharerProductCommission.setUnset(unsetEnum.getCode());
            sharerProductCommission.setCreateTime(now);
            sharerProductCommission.setUpdateTime(now);
            sharerProductCommissionMapper.insert(sharerProductCommission);
        } else {
            sharerProductCommission.setSharerAppid(sharerAppid);
            sharerProductCommission.setCommissionRatio(commissionRatio);
            sharerProductCommission.setCommissionType(CommissionTypeEnum.PLATFORM.getCode());
            sharerProductCommission.setUnset(unsetEnum.getCode());
            sharerProductCommission.setUpdateTime(now);
            sharerProductCommissionMapper.updateById(sharerProductCommission);
        }
    }
}