package com.netease.yanxuan.wx.store.sharer.common.config;

import lombok.Data;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

@Configuration
public class RestTemplateConfig {

  @Bean
  public RestTemplateProperties getRestTemplateProperties() {
	  return new RestTemplateProperties();
  }

  @Bean
  public RestTemplate restTemplate() {
      RestTemplate restTemplate = new RestTemplate();
      restTemplate.setRequestFactory(httpRequestFactory());
      return restTemplate;
  }

  /**
   * httpclient 实现的ClientHttpRequestFactory
   */
  @Bean
  public ClientHttpRequestFactory httpRequestFactory() {
      return new HttpComponentsClientHttpRequestFactory(httpClient(getRestTemplateProperties()));
  }

  /**
   * 使用连接池的 httpclient
   */
  @Bean
  public HttpClient httpClient(RestTemplateProperties restTemplateProperties) {
      Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
              .register("http", PlainConnectionSocketFactory.getSocketFactory())
              .register("https", SSLConnectionSocketFactory.getSocketFactory())
              .build();
      PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(registry);
      // 最大链接数
      connectionManager.setMaxTotal(restTemplateProperties.getMaxTotal());
      // 同路由并发数20
      connectionManager.setDefaultMaxPerRoute(restTemplateProperties.getMaxPerRoute());
      // 增加空闲连接校验间隔‌
      connectionManager.setValidateAfterInactivity(restTemplateProperties.getValidateAfterInactivity());

      RequestConfig requestConfig = RequestConfig.custom()
              // 读超时
              .setSocketTimeout(restTemplateProperties.getReadTimeout())
              // 链接超时
              .setConnectTimeout(restTemplateProperties.getConnectTimeout())
              // 链接不够用的等待时间
              .setConnectionRequestTimeout(restTemplateProperties.getReadTimeout())
              .build();

      return HttpClientBuilder.create()
              .setDefaultRequestConfig(requestConfig)
              .setConnectionManager(connectionManager)
              .setRetryHandler(new DefaultHttpRequestRetryHandler(0, true))
              .build();
  }
  
  @Data
  @Component
  @ConfigurationProperties(prefix = "httpclient")
  public class RestTemplateProperties {
      /**
       * 最大链接数
       */
      private int maxTotal = 2000;
      /**
       * 同路由最大并发数
       */
      private int maxPerRoute = 100;
      /**
       * 读取超时时间 ms
       */
      private int readTimeout = 35000;
      /**
       * 链接超时时间 ms
       */
      private int connectTimeout = 10000;
      /**
       * 空闲连接校验间隔‌
       */
      private int validateAfterInactivity = 2000;
  }
}
