package com.netease.yanxuan.wx.store.sharer.biz.service.impl;

import com.alibaba.fastjson.JSON;
import com.netease.yanxuan.wx.store.sharer.biz.config.DrmSharerConfig;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserContextHolder;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserInfo;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.vo.ProductListVO;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.vo.ProductPromotionLinkVO;
import com.netease.yanxuan.wx.store.sharer.biz.meta.page.PageQuery;
import com.netease.yanxuan.wx.store.sharer.biz.meta.page.PageVO;
import com.netease.yanxuan.wx.store.sharer.biz.service.IProductService;
import com.netease.yanxuan.wx.store.sharer.common.constant.CoreConstant;
import com.netease.yanxuan.wx.store.sharer.common.exception.NoBindSharerException;
import com.netease.yanxuan.wx.store.sharer.common.handler.RedisClient;
import com.netease.yanxuan.wx.store.sharer.common.util.AsyncUtils;
import com.netease.yanxuan.wx.store.sharer.integration.constant.WeChatApi;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.*;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

/**
 * @Description 商品-业务实现类
 * @Author fanjiaxin
 * @Date 2025/3/10 12:31
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class ProductServiceImpl implements IProductService {
    private final RedisClient redisClient;
    private final DrmSharerConfig drmSharerConfig;
    private final Executor asyncProductListServiceExecutor;

    private final WeChatShopPromoteProductListRequest weChatShopPromoteProductListRequest;
    private final WeChatShopPromoteProductDetailRequest weChatShopPromoteProductDetailRequest;
    private final WeChatShopPromoteProductLinkRequest weChatShopPromoteProductLinkRequest;
    private final WeChatShopSharerListRequest weChatShopSharerListRequest;
    private final WeChatShopGetSharerProductCommissionRequest weChatShopGetSharerProductCommissionRequest;


    @Override
    public PageVO<ProductListVO> getProductPageList(PageQuery pageQuery, String keyword) {
        PageVO<ProductListVO> pageVO = new PageVO<>();
        WeChatPromoteProductListVO productListVO = weChatShopPromoteProductListRequest.handle(keyword,
                pageQuery.getNextKey(), pageQuery.getPageSize());
        pageVO.setHasMore(false);
        List<ProductListVO> productList = new ArrayList<>(0);
        // 商品列表
        if (!CollectionUtils.isEmpty(productListVO.getProduct_list())) {
            // 获取用户信息
            LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
            // 使用异步多线程处理
            productList = AsyncUtils.processListAsync(productListVO.getProduct_list(), item -> {
                WeChatPromoteProductDetailVO detailVO = getProductDetailCache(item.getShop_appid(), item.getProduct_id());
                if (null == detailVO) {
                    return null;
                }
                WeChatPromoteProductDetailVO.WeChatPromoteProductVO product = detailVO.getProduct();
                if (null == product) {
                    return null;
                }
                WeChatPromoteProductDetailVO.WeChatPromoteProductVO.WeChatPromoteProductInfoVO productInfo = product.getProduct_info();
                if (null == productInfo) {
                    return null;
                }
                ProductListVO listVO = new ProductListVO();
                listVO.setTitle(productInfo.getTitle());
                listVO.setHeadImg(CollectionUtils.isEmpty(productInfo.getHead_imgs()) ? null : productInfo.getHead_imgs().get(0));
                // 取参数SkuInfo中sale_price的最小
                BigDecimal price = null;
                Optional<BigDecimal> priceOpt = Optional.ofNullable(productInfo.getSkus()).orElseGet(ArrayList::new)
                        .stream()
                        .map(WeChatPromoteProductDetailVO.WeChatPromoteProductVO.WeChatPromoteProductInfoVO.WeChatPromoteProductSkuInfoVO::getSale_price)
                        .min(BigDecimal::compareTo);
                if (priceOpt.isPresent()) {
                    // 以分为单位，进行元转换
                    price = priceOpt.get().divide(new BigDecimal("100"));
                    listVO.setPrice(price.setScale(2, RoundingMode.HALF_UP));
                }
                WeChatPromoteProductDetailVO.WeChatPromoteProductVO.WeChatPromoteProductCommissionInfoVO commissionInfo
                        = product.getCommission_info();
                // 服务费率
                BigDecimal serviceRatio = null;
                if (null != commissionInfo && null != commissionInfo.getService_ratio()) {
                    serviceRatio = new BigDecimal(commissionInfo.getService_ratio())
                            .divide(new BigDecimal(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER));
                }
                // 佣金=商品价格*服务费率*推客分佣比例，四舍五入保留小数点后两位
                BigDecimal commission = BigDecimal.ZERO;
                // 推客分佣比例
                BigDecimal commissionRatio = getSharerCommissionRatio(loginUserInfo, item.getProduct_id());
                if (null != price && null != commissionRatio && null != serviceRatio) {
                    commission = price.multiply(serviceRatio).multiply(commissionRatio)
                            .setScale(2, RoundingMode.HALF_UP);
                    log.info("商品ID:{}，计算佣金，价格：{}，服务费率：{}，推客分佣比例：{}，佣金：{}",
                            item.getProduct_id(), price, serviceRatio, commissionRatio, commission);
                }
                listVO.setCommission(commission);
                listVO.setShopAppid(item.getShop_appid());
                listVO.setProductId(item.getProduct_id());
                listVO.setProductPromotionLink(productInfo.getProduct_promotion_link());
                return listVO;
            }, asyncProductListServiceExecutor);
            // 过滤空
            productList = Optional.of(productList).orElseGet(ArrayList::new)
                    .stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
        }
        pageVO.setList(productList);
        pageVO.setHasMore(StringUtils.isNotBlank(productListVO.getNext_key())
                && !CollectionUtils.isEmpty(productList) && productList.size() >= pageQuery.getPageSize());
        pageVO.setNextKey(productListVO.getNext_key());
        return pageVO;
    }

    @Override
    public ProductPromotionLinkVO getProductPromotionLink(String shopAppid, String productId) {
        // 获取用户信息
        LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
        if (StringUtils.isBlank(loginUserInfo.getSharerAppId())) {
            throw new NoBindSharerException("未绑定推客");
        }
        WeChatPromoteProductLinkVO promoteProductLinkVO = weChatShopPromoteProductLinkRequest.handle(
                loginUserInfo.getSharerAppId(),
                Long.valueOf(productId),
                shopAppid);
        if (null != promoteProductLinkVO && StringUtils.isNotBlank(promoteProductLinkVO.getShort_link())) {
            ProductPromotionLinkVO result = new ProductPromotionLinkVO();
            result.setPromotionLink(promoteProductLinkVO.getShort_link());
            return result;
        }
        return null;
    }

    @Override
    public WeChatSharerListVO getBindSharerList(String sharerOpenid) {
        return weChatShopSharerListRequest.handle(sharerOpenid, "", 1);
    }

    @Override
    public WeChatGetSharerProductCommissionVO getSharerProductCommissionInfo(String sharerAppid,
                                                                             String productId) {
        return weChatShopGetSharerProductCommissionRequest.handle(sharerAppid, Long.valueOf(productId));
    }

    /**
     * 获取商品详情，并缓存到redis中
     */
    private WeChatPromoteProductDetailVO getProductDetailCache(String shopAppid, String productId) {
        WeChatPromoteProductDetailVO detailVO;
        String productDetailKey = CoreConstant.REDIS_PRODUCT_DETAIL_KEY + shopAppid + ":" + productId;
        String productDetailJson = redisClient.getStr(productDetailKey);
        if (StringUtils.isBlank(productDetailJson)) {
            detailVO = weChatShopPromoteProductDetailRequest.handle(shopAppid, Long.valueOf(productId));
            if (null != detailVO) {
                redisClient.setStr(productDetailKey, JSON.toJSONString(detailVO), CoreConstant.REDIS_PRODUCT_DETAIL_EXPIRE_SECONDS);
            }
        } else {
            detailVO = JSON.parseObject(productDetailJson, WeChatPromoteProductDetailVO.class);
        }
        return detailVO;
    }

    /**
     * 获取推客分佣比例
     */
    private BigDecimal getSharerCommissionRatio(LoginUserInfo loginUserInfo, String productId) {
        // 默认佣金比例
        BigDecimal commissionRatio = drmSharerConfig.getDefaultCommissionRatio();
        if (null != loginUserInfo) {
            // 微信配置分佣比例
            BigDecimal weChatCommissionRatio = null;
            // 微信分佣比例倍数
            BigDecimal wechatMultiplier = new BigDecimal(WeChatApi.WECHAT_COMMISSION_RATIO_MULTIPLIER);
            // 获取商品纬度分佣比例
            if (StringUtils.isNotBlank(loginUserInfo.getSharerAppId())) {
                WeChatGetSharerProductCommissionVO sharerProductCommissionVO =
                        weChatShopGetSharerProductCommissionRequest.handle(loginUserInfo.getSharerAppId(),
                                Long.valueOf(productId));
                if (sharerProductCommissionVO.getIs_set()) {
                    weChatCommissionRatio = new BigDecimal(sharerProductCommissionVO.getCommission_ratio());
                }
            }
            // 获取推客纬度分佣比例
            if (null == weChatCommissionRatio && StringUtils.isNotBlank(loginUserInfo.getOpenId())) {
                WeChatSharerListVO sharerListVO = weChatShopSharerListRequest.handle(loginUserInfo.getOpenId(), "", 1);
                if (null != sharerListVO && !CollectionUtils.isEmpty(sharerListVO.getSharer_info_list())) {
                    WeChatSharerInfoVO sharerInfoVO = sharerListVO.getSharer_info_list().get(0);
                    if (null != sharerInfoVO && null != sharerInfoVO.getCommission_ratio()) {
                        weChatCommissionRatio = sharerInfoVO.getCommission_ratio();
                    }
                }
            }
            if (null != weChatCommissionRatio) {
                commissionRatio = weChatCommissionRatio.divide(wechatMultiplier)
                        .setScale(2, RoundingMode.HALF_UP);
            }
        }
        return commissionRatio;
    }
}
