/**
 * @(#)RedisConfiguration.java, 2022/7/8.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.yanxuan.wx.store.sharer.dal.config;

import com.netease.yanxuan.wx.store.sharer.common.json.FastJsonRedisSerializer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cache.CacheManager;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.connection.RedisClusterConfiguration;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.jedis.JedisClientConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import redis.clients.jedis.JedisPoolConfig;

import java.time.Duration;


/**
 * @author 柳敦盛 (liudunsheng@corp.netease.com)
 */
@Configuration
public class RedisConfiguration {

    private static final long CONNECT_TIMEOUT = 500L;

    private static final long READ_TIMEOUT = 500L;

    @Autowired
    private DrmDBConfig drmDBConfig;

    @Bean
    public RedisClusterConfiguration redisClusterConfiguration() {
        RedisClusterConfiguration redisClusterConfiguration = new RedisClusterConfiguration(
                drmDBConfig.getClusterNodes());
        redisClusterConfiguration.setMaxRedirects(drmDBConfig.getMaxRedirects());
        redisClusterConfiguration.setPassword(RedisPassword.of(drmDBConfig.getPassword()));
        return redisClusterConfiguration;
    }

    @Bean
    public JedisPoolConfig jedisPoolConfig() {
        JedisPoolConfig jedisPoolConfig = new JedisPoolConfig();
        // 最大连接数
        jedisPoolConfig.setMaxTotal(drmDBConfig.getMaxTotal());
        // 最大空闲连接数
        jedisPoolConfig.setMaxIdle(drmDBConfig.getMaxIdle());
        jedisPoolConfig.setMinIdle(drmDBConfig.getMinIdle());
        // 每次释放连接的最大数目
        jedisPoolConfig.setNumTestsPerEvictionRun(drmDBConfig.getNumTestsPerEvictionRun());
        // 释放连接的扫描间隔（毫秒）
        jedisPoolConfig.setTimeBetweenEvictionRunsMillis(drmDBConfig.getTimeBetweenEvictionRunsMillis());
        // 连接最小空闲时间
        jedisPoolConfig.setMinEvictableIdleTimeMillis(drmDBConfig.getMinEvictableIdleTimeMillis());
        // 连接空闲多久后释放, 当空闲时间>该值 且 空闲连接>最大空闲连接数 时直接释放
        jedisPoolConfig.setSoftMinEvictableIdleTimeMillis(drmDBConfig.getSoftMinEvictableIdleTimeMillis());
        // 获取连接时的最大等待毫秒数,小于零:阻塞不确定的时间,默认-1
        jedisPoolConfig.setMaxWaitMillis(drmDBConfig.getMaxWaitMillis());
        // 在获取连接的时候检查有效性, 默认false
        jedisPoolConfig.setTestOnBorrow(drmDBConfig.getTestOnBorrow());
        // 在空闲时检查有效性, 默认false
        jedisPoolConfig.setTestWhileIdle(drmDBConfig.getTestWhileIdle());
        // 连接耗尽时是否阻塞, false报异常,ture阻塞直到超时, 默认true
        jedisPoolConfig.setBlockWhenExhausted(drmDBConfig.getBlockWhenExhausted());
        return jedisPoolConfig;
    }

    @Bean
    public JedisClientConfiguration jedisClientConfiguration() {
        JedisClientConfiguration.JedisClientConfigurationBuilder builder = JedisClientConfiguration.builder();
        return builder.connectTimeout(Duration.ofMillis(CONNECT_TIMEOUT)).readTimeout(Duration.ofMillis(READ_TIMEOUT))
                .usePooling().poolConfig(jedisPoolConfig()).build();
    }

    @Bean(name = "connectionFactory")
    public JedisConnectionFactory jedisConnectionFactory() {
        return new JedisConnectionFactory(redisClusterConfiguration(), jedisClientConfiguration());
    }

    @Bean("redisTemplate")
    @Primary
    public RedisTemplate<String, Object> getRedisTemplate(
            @Qualifier("connectionFactory") JedisConnectionFactory connectionFactory) {
        RedisTemplate<String, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(connectionFactory);
        FastJsonRedisSerializer fastJsonRedisSerializer = new FastJsonRedisSerializer();
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.setDefaultSerializer(fastJsonRedisSerializer);
        redisTemplate.setValueSerializer(fastJsonRedisSerializer);
        redisTemplate.setHashValueSerializer(fastJsonRedisSerializer);
        return redisTemplate;
    }

    @Bean("redisStringTemplate")
    public RedisTemplate<Object, Object> getRedisStringTemplate(
            @Qualifier("connectionFactory") JedisConnectionFactory connectionFactory) {
        RedisTemplate<Object, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(connectionFactory);
        RedisSerializer<String> redisSerializer = new StringRedisSerializer();
        FastJsonRedisSerializer fastJsonRedisSerializer = new FastJsonRedisSerializer();
        redisTemplate.setDefaultSerializer(fastJsonRedisSerializer);
        redisTemplate.setValueSerializer(redisSerializer);
        redisTemplate.setHashValueSerializer(fastJsonRedisSerializer);
        return redisTemplate;
    }

    @Bean("openapiCacheManager")
    public CacheManager cacheManager(@Qualifier("connectionFactory") JedisConnectionFactory connectionFactory) {
        return RedisCacheManager.create(connectionFactory);
    }
}
