package com.netease.yanxuan.wx.store.sharer.integration.core;

import com.netease.yanxuan.wx.store.sharer.common.constant.CoreConstant;
import com.netease.yanxuan.wx.store.sharer.common.exception.BizException;
import com.netease.yanxuan.wx.store.sharer.common.handler.RedisClient;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatAccessTokenRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopAccessTokenRequest;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatAccessTokenVO;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.util.concurrent.TimeUnit;

/**
 * @Description 微信认证信息上下文持有器
 * @Author fanjiaxin
 * @Date 2025/2/24 11:53
 */
@Service
@RequiredArgsConstructor
public class WeChatContextHolder {
    private final RedisClient redisClient;
    private final WeChatAccessTokenRequest weChatAccessTokenRequest;
    private final WeChatShopAccessTokenRequest weChatShopAccessTokenRequest;

    private static final long RETRY_DELAY_MS = 200;
    private static final int EXPIRE_SECONDS = 30 * 60;

    /**
     * 获取微信认证信息
     */
    public String getAccessToken() {
        // 双重检查锁定
        String accessToken = redisClient.getStr(CoreConstant.REDIS_TOKEN_WECHAT_KEY);
        if (StringUtils.isNotBlank(accessToken)) {
            return accessToken;
        }
        try {
            return createAccessToken();
        } catch (InterruptedException e) {
            throw new BizException("微信认证获取AccessToken异常", e);
        }
    }

    /**
     * 创建微信认证信息
     */
    public String createAccessToken() throws InterruptedException {
        // 设置10秒的锁
        boolean locked = redisClient.setIfAbsent(CoreConstant.REDIS_TOKEN_WECHAT_LOCK_KEY,
                CoreConstant.REDIS_TOKEN_WECHAT_LOCK_VALUE, CoreConstant.REDIS_TOKEN_WECHAT_LOCK_TIME);
        if (!locked) {
            // 未获取到锁，等待一段时间后重试
            TimeUnit.MILLISECONDS.sleep(RETRY_DELAY_MS);
            return getAccessToken();
        }
        WeChatAccessTokenVO handle = weChatAccessTokenRequest.handle();
        redisClient.setStr(CoreConstant.REDIS_TOKEN_WECHAT_KEY, handle.getAccess_token(),
                handle.getExpires_in() - EXPIRE_SECONDS);
        return handle.getAccess_token();
    }

    /**
     * 获取微信小店认证信息
     */
    public String getShopAccessToken() {
        // 检查锁定
        String accessToken = redisClient.getStr(CoreConstant.REDIS_TOKEN_WECHAT_SHOP_KEY);
        if (StringUtils.isNotBlank(accessToken)) {
            return accessToken;
        }
        try {
            return createAccessToken();
        } catch (InterruptedException e) {
            throw new BizException("微信认证获取AccessToken异常", e);
        }
    }

    /**
     * 创建微信小店认证信息
     */
    public String createShopAccessToken() throws InterruptedException {
        // 设置10秒的锁
        boolean locked = redisClient.setIfAbsent(CoreConstant.REDIS_TOKEN_WECHAT_SHOP_LOCK_KEY,
                CoreConstant.REDIS_TOKEN_WECHAT_SHOP_LOCK_VALUE, CoreConstant.REDIS_TOKEN_WECHAT_SHOP_LOCK_TIME);
        if (!locked) {
            // 未获取到锁，等待一段时间后重试
            TimeUnit.MILLISECONDS.sleep(RETRY_DELAY_MS);
            return getShopAccessToken();
        }
        WeChatAccessTokenVO handle = weChatShopAccessTokenRequest.handle();
        redisClient.setStr(CoreConstant.REDIS_TOKEN_WECHAT_SHOP_KEY, handle.getAccess_token(),
                handle.getExpires_in() - EXPIRE_SECONDS);
        return handle.getAccess_token();
    }
}
