package com.netease.yanxuan.wx.store.sharer.biz.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.netease.yanxuan.wx.store.sharer.biz.service.ICommissionOrderService;
import com.netease.yanxuan.wx.store.sharer.dal.mapper.*;
import com.netease.yanxuan.wx.store.sharer.dal.meta.model.po.*;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatCommissionOrderDetailRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatCommissionOrderListRequest;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatCommissionOrderDetailVO;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatCommissionOrderItemVO;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatCommissionOrderListVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;


/**
 * @Description 订单-业务实现类
 * @Author fanjiaxin
 * @Date 2025/3/10 12:31
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class CommissionOrderServiceImpl implements ICommissionOrderService {

    private final SharerOrderSyncFailMapper sharerOrderSyncFailMapper;
    private final SharerOrderMapper sharerOrderMapper;
    private final SharerOrderProductMapper sharerOrderProductMapper;
    private final SharerOrderCommissionMapper sharerOrderCommissionMapper;
    private final SharerOrderCommissionSharerMapper sharerOrderCommissionSharerMapper;
    private final SharerOrderCommissionSubAgencyMapper sharerOrderCommissionSubAgencyMapper;
    private final SharerOrderCommissionPromoterAgencyMapper sharerOrderCommissionPromoterAgencyMapper;
    private final SharerOrderCommissionFinderMapper sharerOrderCommissionFinderMapper;
    private final SharerOrderCommissionTalentMapper sharerOrderCommissionTalentMapper;
    private final SharerOrderBuyerMapper sharerOrderBuyerMapper;
    private final WeChatCommissionOrderListRequest weChatCommissionOrderListRequest;
    private final WeChatCommissionOrderDetailRequest weChatCommissionOrderDetailRequest;


    private static final int COMMISSION_ORDER_SYNC_PAGE_SIZE = 30;
    private static final int COMMISSION_ORDER_SYNC_MAX_RETRY_COUNT = 5;


    @Override
    public Long getLatestCommissionSyncTimestamp() {
        // 同步成功最大时间戳记录
        LambdaQueryWrapper<SharerOrder> orderLqw = Wrappers.lambdaQuery();
        orderLqw.select(SharerOrder::getOrderCreateTime);
        orderLqw.orderByDesc(SharerOrder::getOrderCreateTime);
        orderLqw.last("LIMIT 1");
        SharerOrder sharerOrder = sharerOrderMapper.selectOne(orderLqw);
        long maxTimeStamp = 0L;
        if (null != sharerOrder && null != sharerOrder.getOrderCreateTime()) {
            maxTimeStamp = Math.max(maxTimeStamp, sharerOrder.getOrderCreateTime().getTime() / 1000);
        }
        return maxTimeStamp;
    }

    @Override
    public void doCommissionOrderSync(Long startTime, Long endTime) {
        WeChatCommissionOrderListVO listVO;
        String nextKey = null;
        while (true) {
            listVO = weChatCommissionOrderListRequest.handle(startTime, endTime, nextKey, COMMISSION_ORDER_SYNC_PAGE_SIZE);
            if (null == listVO || CollectionUtils.isEmpty(listVO.getList())) {
                return;
            }
            Boolean hasMore = listVO.getHas_more();
            nextKey = listVO.getNext_key();
            if(StringUtils.isBlank(nextKey) || (null != hasMore && !hasMore)){
                return;
            }
            for (WeChatCommissionOrderItemVO itemVO : listVO.getList()) {
                // 推客分佣单入库
                doSyncCommissionOrder(itemVO.getOrder_id(), itemVO.getSku_id());
            }
        }
    }

    @Override
    public void doCommissionOrderFailRetry() {
        int pageNum = 1;
        int pageSize = 10;
        Page<SharerOrderSyncFail> pageData;
        LambdaQueryWrapper<SharerOrderSyncFail> failLqw = Wrappers.lambdaQuery();
        failLqw.lt(SharerOrderSyncFail::getRetryCount, COMMISSION_ORDER_SYNC_MAX_RETRY_COUNT);
        while (true) {
            pageData = sharerOrderSyncFailMapper.selectPage(new Page<>(pageNum++, pageSize), failLqw);
            if (null == pageData || CollectionUtils.isEmpty(pageData.getRecords())) {
                // 如果返回值为空，直接终止循环
                break;
            }
            for (SharerOrderSyncFail fail : pageData.getRecords()) {
                LambdaQueryWrapper<SharerOrder> orderLqw = Wrappers.lambdaQuery();
                orderLqw.eq(SharerOrder::getOrderId, fail.getOrderId());
                orderLqw.eq(SharerOrder::getSkuId, fail.getSkuId());
                Long count = sharerOrderMapper.selectCount(orderLqw);
                if (null != count && count > 0) {
                    // 已经同步，溢出失败记录
                    sharerOrderSyncFailMapper.deleteById(fail.getId());
                    continue;
                }
                // 推客分佣单入库
                doSyncCommissionOrder(fail.getOrderId(), fail.getSkuId());
            }
        }
    }

    /**
     * 推客分佣单信息入库
     */
    private void doSyncCommissionOrder(String orderId, String skuId) {
        WeChatCommissionOrderDetailVO detailVO;
        try {
            detailVO = weChatCommissionOrderDetailRequest.handle(orderId, skuId);
        } catch (Exception e) {
            log.error("[op:doSyncCommissionOrder] 推客分佣单获取详情异常", e);
            String failMsg = StringUtils.substring(e.getMessage(), 0, 5000);
            doSaveCommissionOrderSyncFail(orderId, skuId, String.join("推客分佣单获取详情异常;", failMsg));
            return;
        }
        try {
            if (null == detailVO) {
                log.info("[op:doSyncCommissionOrder] 推客分佣单不存在");
                return;
            }
            WeChatCommissionOrderDetailVO.CommssionOrderDetailVO commssionOrder = detailVO.getCommssion_order();
            if (null == commssionOrder) {
                log.info("[op:doSyncCommissionOrder] 推客分佣单明细不存在");
                return;
            }
            log.info("[op:doSyncCommissionOrder] 推客分佣单获取详情 detailVO:{}", JSON.toJSONString(commssionOrder));
            doSaveCommissionOrder(commssionOrder);
        } catch (Exception e) {
            log.error("[op:doSyncCommissionOrder] 推客分佣单信息入库异常", e);
            String failMsg = StringUtils.substring(e.getMessage(), 0, 5000);
            doSaveCommissionOrderSyncFail(orderId, skuId, String.join("推客分佣单信息入库异常;", failMsg));
        }
    }

    /**
     * 保存分佣单同步失败记录
     */
    private void doSaveCommissionOrderSyncFail(String orderId, String skuId, String failMsg) {
        try {
            LambdaQueryWrapper<SharerOrderSyncFail> failLqw = Wrappers.lambdaQuery();
            failLqw.eq(SharerOrderSyncFail::getOrderId, orderId);
            failLqw.eq(SharerOrderSyncFail::getSkuId, skuId);
            failLqw.last("LIMIT 1");
            SharerOrderSyncFail sharerOrderSyncFail = sharerOrderSyncFailMapper.selectOne(failLqw);
            Date date = new Date();
            if (null == sharerOrderSyncFail) {
                sharerOrderSyncFail = new SharerOrderSyncFail();
                sharerOrderSyncFail.setOrderId(orderId);
                sharerOrderSyncFail.setSkuId(skuId);
                sharerOrderSyncFail.setRetryCount(0);
                sharerOrderSyncFail.setFailMsg(failMsg);
                sharerOrderSyncFail.setCreateTime(date);
                sharerOrderSyncFail.setUpdateTime(date);
                sharerOrderSyncFailMapper.insert(sharerOrderSyncFail);
            } else {
                LambdaUpdateWrapper<SharerOrderSyncFail> failLuw = Wrappers.lambdaUpdate();
                failLuw.setSql("retry_count = retry_count + 1");
                failLuw.set(SharerOrderSyncFail::getFailMsg, failMsg);
                failLuw.set(SharerOrderSyncFail::getUpdateTime, date);
                failLuw.eq(SharerOrderSyncFail::getId, sharerOrderSyncFail.getId());
                sharerOrderSyncFailMapper.update(null, failLuw);
            }
        } catch (Exception e) {
            log.error("[op:doSaveCommissionOrderSyncFail] 保存分佣单同步失败记录异常", e);
        }
    }

    /**
     * 保存分佣单
     */
    private void doSaveCommissionOrder(WeChatCommissionOrderDetailVO.CommssionOrderDetailVO detailVO) {
        if (null == detailVO) {
            return;
        }
        LambdaQueryWrapper<SharerOrder> orderLqw = Wrappers.lambdaQuery();
        orderLqw.eq(SharerOrder::getOrderId, detailVO.getOrder_id());
        orderLqw.eq(SharerOrder::getSkuId, String.valueOf(detailVO.getSku_id()));
        orderLqw.last("LIMIT 1");
        SharerOrder sharerOrder = sharerOrderMapper.selectOne(orderLqw);
        if(null == sharerOrder){
            sharerOrder = new SharerOrder();
        }
        sharerOrder.setOrderId(detailVO.getOrder_id());
        sharerOrder.setSkuId(String.valueOf(detailVO.getSku_id()));
        sharerOrder.setCommissionStatus(String.valueOf(detailVO.getStatus()));
        String orderStatus = null;
        try {
            orderStatus = String.valueOf(detailVO.getOrder_detail().getOrder_info().getStatus());
        } catch (Exception e) {
            log.error("[op:doSaveCommissionOrder] 分佣单状态转换异常", e);
        }
        sharerOrder.setOrderStatus(orderStatus);
        sharerOrder.setOrderCreateTime(getDate(detailVO.getCreate_time()));
        sharerOrder.setOrderUpdateTime(getDate(detailVO.getUpdate_time()));
        Date date = new Date();
        sharerOrder.setUpdateTime(date);
        if(null == sharerOrder.getId() || sharerOrder.getId() <= 0){
            sharerOrder.setCreateTime(date);
            sharerOrderMapper.insert(sharerOrder);
        }else{
            sharerOrder.setUpdateTime(date);
            sharerOrderMapper.updateById(sharerOrder);
        }
        WeChatCommissionOrderDetailVO.OrderDetailVO orderDetail = detailVO.getOrder_detail();
        if (null == orderDetail) {
            log.info("[op:doSaveCommissionOrder] orderDetail is null");
            return;
        }
        // 商品信息
        doSaveCommissionOrderProduct(orderDetail, sharerOrder.getId());
        // 分佣信息
        doSaveCommissionOrderCommission(orderDetail.getCommission_info(), sharerOrder.getId());
        // 买家信息
        doSaveCommissionOrderBuyer(orderDetail.getBuyer_info(), sharerOrder.getId());
    }

    /**
     * 商品信息
     */
    private void doSaveCommissionOrderProduct(WeChatCommissionOrderDetailVO.OrderDetailVO detailVO,
                                              Long sharerOrderId) {
        // 先删后增
        LambdaQueryWrapper<SharerOrderProduct> delLqw = Wrappers.lambdaQuery();
        delLqw.eq(SharerOrderProduct::getSharerOrderId, sharerOrderId);
        sharerOrderProductMapper.delete(delLqw);
        WeChatCommissionOrderDetailVO.ProductInfoVO productInfo = detailVO.getProduct_info();
        if (null == productInfo) {
            log.info("[op:doSaveCommissionOrderProduct] productInfo is null");
            return;
        }
        SharerOrderProduct productInfoPO = new SharerOrderProduct();
        productInfoPO.setSharerOrderId(sharerOrderId);
        productInfoPO.setShopAppid(null != detailVO.getShop_info() ? detailVO.getShop_info().getAppid() : null);
        productInfoPO.setProductId(productInfo.getProduct_id());
        productInfoPO.setThumbImg(productInfo.getThumb_img());
        productInfoPO.setActualPayment(productInfo.getActual_payment());
        productInfoPO.setTitle(productInfo.getTitle());
        Date date = new Date();
        productInfoPO.setCreateTime(date);
        productInfoPO.setUpdateTime(date);
        sharerOrderProductMapper.insert(productInfoPO);
    }

    /**
     * 分佣信息
     */
    private void doSaveCommissionOrderCommission(WeChatCommissionOrderDetailVO.CommissionInfoVO commissionInfo,
                                                 Long sharerOrderId) {
        // 先删后增
        LambdaQueryWrapper<SharerOrderCommission> delLqw = Wrappers.lambdaQuery();
        delLqw.eq(SharerOrderCommission::getSharerOrderId, sharerOrderId);
        sharerOrderCommissionMapper.delete(delLqw);
        if (null == commissionInfo) {
            log.info("[op:doSaveCommissionOrderCommission] commissionInfo is null");
            return;
        }
        SharerOrderCommission commissionInfoPO = new SharerOrderCommission();
        commissionInfoPO.setSharerOrderId(sharerOrderId);
        commissionInfoPO.setServiceRatio(commissionInfo.getService_ratio());
        commissionInfoPO.setServiceAmount(commissionInfo.getService_amount());
        commissionInfoPO.setProfitShardingSucTime(getDate(commissionInfo.getProfit_sharding_suc_time()));
        commissionInfoPO.setPromotionChannel(commissionInfo.getPromotion_channel());
        commissionInfoPO.setPromotionInfo(JSON.toJSONString(commissionInfo.getPromotion_info()));
        Date date = new Date();
        commissionInfoPO.setCreateTime(date);
        commissionInfoPO.setUpdateTime(date);
        sharerOrderCommissionMapper.insert(commissionInfoPO);
        // 分佣推客
        doSaveCommissionOrderCommissionSharer(commissionInfo.getSharer_info(), sharerOrderId, commissionInfoPO.getId());
        // 分佣带货机构
        doSaveCommissionOrderCommissionSubAgency(commissionInfo.getSub_agency_info(), sharerOrderId, commissionInfoPO.getId());
        // 分佣推客机构
        doSaveCommissionOrderCommissionPromoterAgency(commissionInfo.getPromoter_agency_info(), sharerOrderId, commissionInfoPO.getId());
        // 分佣视频号达人信息
        doSaveCommissionOrderCommissionFinder(commissionInfo.getFinder_info(), sharerOrderId, commissionInfoPO.getId());
        // 分佣新带货达人信息
        doSaveCommissionOrderCommissionTalent(commissionInfo.getTalent_info(), sharerOrderId, commissionInfoPO.getId());
    }

    /**
     * 分佣推客
     */
    private void doSaveCommissionOrderCommissionSharer(WeChatCommissionOrderDetailVO.SharerInfoVO sharerInfo,
                                                       Long sharerOrderId, Long sharerOrderCommissionId) {
        // 先删后增
        LambdaQueryWrapper<SharerOrderCommissionSharer> delLqw = Wrappers.lambdaQuery();
        delLqw.eq(SharerOrderCommissionSharer::getSharerOrderId, sharerOrderId);
        delLqw.eq(SharerOrderCommissionSharer::getSharerOrderCommissionId, sharerOrderCommissionId);
        sharerOrderCommissionSharerMapper.delete(delLqw);
        if (null == sharerInfo) {
            log.info("[op:doSaveCommissionOrderCommissionSharer] sharerInfo is null");
            return;
        }
        SharerOrderCommissionSharer sharerInfoPO = new SharerOrderCommissionSharer();
        sharerInfoPO.setSharerOrderId(sharerOrderId);
        sharerInfoPO.setSharerOrderCommissionId(sharerOrderCommissionId);
        sharerInfoPO.setSharerAppid(sharerInfo.getSharer_appid());
        sharerInfoPO.setNickname(sharerInfo.getNickname());
        sharerInfoPO.setRatio(sharerInfo.getRatio());
        sharerInfoPO.setAmount(sharerInfo.getAmount());
        sharerInfoPO.setOpensharerid(sharerInfo.getOpensharerid());
        Date date = new Date();
        sharerInfoPO.setCreateTime(date);
        sharerInfoPO.setUpdateTime(date);
        sharerOrderCommissionSharerMapper.insert(sharerInfoPO);
    }

    /**
     * 分佣带货机构
     */
    private void doSaveCommissionOrderCommissionSubAgency(WeChatCommissionOrderDetailVO.AgencyInfoVO agencyInfo,
                                                          Long sharerOrderId, Long sharerOrderCommissionId) {
        // 先删后增
        LambdaQueryWrapper<SharerOrderCommissionSubAgency> delLqw = Wrappers.lambdaQuery();
        delLqw.eq(SharerOrderCommissionSubAgency::getSharerOrderId, sharerOrderId);
        delLqw.eq(SharerOrderCommissionSubAgency::getSharerOrderCommissionId, sharerOrderCommissionId);
        sharerOrderCommissionSubAgencyMapper.delete(delLqw);
        if (null == agencyInfo) {
            log.info("[op:doSaveCommissionOrderCommissionSubAgency] agencyInfo is null");
            return;
        }
        SharerOrderCommissionSubAgency agencyInfoPO = new SharerOrderCommissionSubAgency();
        agencyInfoPO.setSharerOrderId(sharerOrderId);
        agencyInfoPO.setSharerOrderCommissionId(sharerOrderCommissionId);
        agencyInfoPO.setAppid(agencyInfo.getAppid());
        agencyInfoPO.setNickname(agencyInfo.getNickname());
        agencyInfoPO.setRatio(agencyInfo.getRatio());
        agencyInfoPO.setAmount(agencyInfo.getAmount());
        agencyInfoPO.setProfitShardingSucTime(getDate(agencyInfo.getProfit_sharding_suc_time()));
        Date date = new Date();
        agencyInfoPO.setCreateTime(date);
        agencyInfoPO.setUpdateTime(date);
        sharerOrderCommissionSubAgencyMapper.insert(agencyInfoPO);
    }

    /**
     * 分佣推客机构
     */
    private void doSaveCommissionOrderCommissionPromoterAgency(WeChatCommissionOrderDetailVO.AgencyInfoVO agencyInfo,
                                                               Long sharerOrderId, Long sharerOrderCommissionId) {
        // 先删后增
        LambdaQueryWrapper<SharerOrderCommissionPromoterAgency> delLqw = Wrappers.lambdaQuery();
        delLqw.eq(SharerOrderCommissionPromoterAgency::getSharerOrderId, sharerOrderId);
        delLqw.eq(SharerOrderCommissionPromoterAgency::getSharerOrderCommissionId, sharerOrderCommissionId);
        sharerOrderCommissionPromoterAgencyMapper.delete(delLqw);
        if (null == agencyInfo) {
            log.info("[op:doSaveCommissionOrderCommissionPromoterAgency] agencyInfo is null");
            return;
        }
        SharerOrderCommissionPromoterAgency agencyInfoPO = new SharerOrderCommissionPromoterAgency();
        agencyInfoPO.setSharerOrderId(sharerOrderId);
        agencyInfoPO.setSharerOrderCommissionId(sharerOrderCommissionId);
        agencyInfoPO.setAppid(agencyInfo.getAppid());
        agencyInfoPO.setNickname(agencyInfo.getNickname());
        agencyInfoPO.setRatio(agencyInfo.getRatio());
        agencyInfoPO.setAmount(agencyInfo.getAmount());
        agencyInfoPO.setProfitShardingSucTime(getDate(agencyInfo.getProfit_sharding_suc_time()));
        Date date = new Date();
        agencyInfoPO.setCreateTime(date);
        agencyInfoPO.setUpdateTime(date);
        sharerOrderCommissionPromoterAgencyMapper.insert(agencyInfoPO);
    }

    /**
     * 分佣视频号达人信息
     */
    private void doSaveCommissionOrderCommissionFinder(WeChatCommissionOrderDetailVO.FinderInfoVO finderInfo,
                                                       Long sharerOrderId, Long sharerOrderCommissionId) {
        // 先删后增
        LambdaQueryWrapper<SharerOrderCommissionFinder> delLqw = Wrappers.lambdaQuery();
        delLqw.eq(SharerOrderCommissionFinder::getSharerOrderId, sharerOrderId);
        delLqw.eq(SharerOrderCommissionFinder::getSharerOrderCommissionId, sharerOrderCommissionId);
        sharerOrderCommissionFinderMapper.delete(delLqw);
        if (null == finderInfo) {
            log.info("[op:doSaveCommissionOrderCommissionFinder] finderInfo is null");
            return;
        }
        SharerOrderCommissionFinder finderInfoPO = new SharerOrderCommissionFinder();
        finderInfoPO.setSharerOrderId(sharerOrderId);
        finderInfoPO.setSharerOrderCommissionId(sharerOrderCommissionId);
        finderInfoPO.setNickname(finderInfo.getNickname());
        finderInfoPO.setRatio(finderInfo.getRatio());
        finderInfoPO.setAmount(finderInfo.getAmount());
        finderInfoPO.setOpenfinderid(finderInfo.getOpenfinderid());
        Date date = new Date();
        finderInfoPO.setCreateTime(date);
        finderInfoPO.setUpdateTime(date);
        sharerOrderCommissionFinderMapper.insert(finderInfoPO);
    }

    /**
     * 分佣新带货达人信息
     */
    private void doSaveCommissionOrderCommissionTalent(WeChatCommissionOrderDetailVO.TalentInfoVO talentInfo,
                                                       Long sharerOrderId, Long sharerOrderCommissionId) {
        // 先删后增
        LambdaQueryWrapper<SharerOrderCommissionTalent> delLqw = Wrappers.lambdaQuery();
        delLqw.eq(SharerOrderCommissionTalent::getSharerOrderId, sharerOrderId);
        delLqw.eq(SharerOrderCommissionTalent::getSharerOrderCommissionId, sharerOrderCommissionId);
        sharerOrderCommissionTalentMapper.delete(delLqw);
        if (null == talentInfo) {
            log.info("[op:doSaveCommissionOrderCommissionTalent] talentInfo is null");
            return;
        }
        SharerOrderCommissionTalent sharerOrderCommissionTalent = new SharerOrderCommissionTalent();
        sharerOrderCommissionTalent.setSharerOrderId(sharerOrderId);
        sharerOrderCommissionTalent.setSharerOrderCommissionId(sharerOrderCommissionId);
        sharerOrderCommissionTalent.setTalent_appid(talentInfo.getTalent_appid());
        sharerOrderCommissionTalent.setNickname(talentInfo.getNickname());
        sharerOrderCommissionTalent.setRatio(talentInfo.getRatio());
        sharerOrderCommissionTalent.setAmount(talentInfo.getAmount());
        sharerOrderCommissionTalent.setOpentalentid(talentInfo.getOpentalentid());
        Date date = new Date();
        sharerOrderCommissionTalent.setCreateTime(date);
        sharerOrderCommissionTalent.setUpdateTime(date);
        sharerOrderCommissionTalentMapper.insert(sharerOrderCommissionTalent);
    }

    /**
     * 买家信息
     */
    private void doSaveCommissionOrderBuyer(WeChatCommissionOrderDetailVO.BuyerInfoVO buyerInfo,
                                            Long sharerOrderId) {
        // 先删后增
        LambdaQueryWrapper<SharerOrderBuyer> delLqw = Wrappers.lambdaQuery();
        delLqw.eq(SharerOrderBuyer::getSharerOrderId, sharerOrderId);
        sharerOrderBuyerMapper.delete(delLqw);
        if (null == buyerInfo) {
            log.info("[op:doSaveCommissionOrderBuyer] buyerInfo is null");
            return;
        }
        SharerOrderBuyer buyerInfoPO = new SharerOrderBuyer();
        buyerInfoPO.setSharerOrderId(sharerOrderId);
        buyerInfoPO.setOpenId(buyerInfo.getOpen_id());
        buyerInfoPO.setUnionId(buyerInfo.getUnion_id());
        Date date = new Date();
        buyerInfoPO.setCreateTime(date);
        buyerInfoPO.setUpdateTime(date);
        sharerOrderBuyerMapper.insert(buyerInfoPO);
    }

    private Date getDate(Long wechatTime) {
        return new Date(wechatTime * 1000);
    }
}
