/**
 * @(#)DrmSharerConfig.java, 2025/3/7.
 * <p/>
 * Copyright 2025 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.yanxuan.wx.store.sharer.biz.config;

import com.alibaba.fastjson.JSON;
import com.ctrip.framework.apollo.model.ConfigChange;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.ctrip.framework.apollo.spring.annotation.EnableAutoUpdateApolloConfig;
import com.ctrip.framework.apollo.spring.annotation.ValueMapping;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserContextHolder;
import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserInfo;
import com.netease.yanxuan.wx.store.sharer.biz.meta.enums.CommissionChangeOptTypeEnum;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.bo.CommissionSharerBO;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.bo.CommissionSharerProductBO;
import com.netease.yanxuan.wx.store.sharer.dal.mapper.SharerProductCommissionRecordMapper;
import com.netease.yanxuan.wx.store.sharer.dal.meta.model.po.SharerProductCommissionRecord;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatSetSharerCommissionRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatSetSharerProductCommissionRequest;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatSharerListRequest;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatSharerInfoVO;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatSharerListVO;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 推客配置apo
 *
 * @author 刘运星 (liuyunxing01@corp.netease.com)
 */
@Slf4j
@Data
@Component
@RequiredArgsConstructor
@EnableAutoUpdateApolloConfig("sharer-config")
public class DrmSharerConfig {
    private final SharerProductCommissionRecordMapper sharerProdutcCommissionRecordMapper;
    private final WeChatSharerListRequest weChatSharerListRequest;
    private final WeChatSetSharerCommissionRequest weChatSetSharerCommissionRequest;
    private final WeChatSetSharerProductCommissionRequest weChatSetSharerProductCommissionRequest;


    /**
     * 默认分佣比例
     */
    @Value("${commissionratio.default:10}")
    private BigDecimal defaultCommissionRatio;
    /**
     * 用户分佣比例
     */
    @ValueMapping("${commissionratio.sharer:}")
    private List<CommissionSharerBO> commissionSharerList;
    /**
     * 用户商品分佣比例
     */
    @ValueMapping("${commissionratio.sharerproduct:}")
    private List<CommissionSharerProductBO> commissionSharerProductList;

    /**
     * 监听
     */
    @ApolloConfigChangeListener("sharer-config")
    public void onChange(ConfigChangeEvent changeEvent) {
        log.info("ProductCommissionConfig:onChange, changeEvent = {}", JSON.toJSONString(changeEvent));

        // 默认分佣比例
        String defaultCommissionratio = "application.commissionratio.default";
        if (changeEvent.isChanged(defaultCommissionratio)) {
            ConfigChange change = changeEvent.getChange(defaultCommissionratio);
            defaultCommissionratioChange(change);
        }

        // 用户分佣比例
        String sharerCommissionratio = "application.commissionratio.sharer";
        if (changeEvent.isChanged(sharerCommissionratio)) {
            ConfigChange change = changeEvent.getChange(sharerCommissionratio);
            sharerCommissionRatioChange(change);
        }

        // 用户商品分佣比例
        String sharerproductCommissionratio = "application.commissionratio.sharerproduct";
        if (changeEvent.isChanged(sharerproductCommissionratio)) {
            ConfigChange change = changeEvent.getChange(sharerproductCommissionratio);
            sharerProductCommissionRatioChange(change);
        }
    }

    /**
     * 默认分佣比例变更
     */
    private void defaultCommissionratioChange(ConfigChange change) {
        if (StringUtils.isBlank(change.getNewValue()) || change.getNewValue().equals(change.getOldValue())) {
            return;
        }
        // 获取推客绑定的所有用户
        String nextKey = null;
        int pageSize = 10;
        while (true) {
            WeChatSharerListVO sharerListVO = weChatSharerListRequest.handle(null, nextKey, pageSize);
            if (sharerListVO == null) {
                break; // 如果返回值为空，直接终止循环
            }
            nextKey = sharerListVO.getNext_key();
            if (!CollectionUtils.isEmpty(sharerListVO.getSharer_info_list())) {
                for (WeChatSharerInfoVO sharerInfoVO : sharerListVO.getSharer_info_list()) {
                    // 处理每个 sharerInfoVO
                    setSharerDefaultCommissionInfo(change, sharerInfoVO);
                }
            }
            if (CollectionUtils.isEmpty(sharerListVO.getSharer_info_list()) || sharerListVO.getSharer_info_list().size() < pageSize) {
                break; // 如果列表为空或小于 pageSize，终止循环
            }
        }
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerDefaultCommissionInfo(ConfigChange change, WeChatSharerInfoVO sharerInfoVO) {
        // 分佣比例为小数,  微信接口参数应当分佣比例*1000 为整数
        BigDecimal commissionRatio = new BigDecimal(change.getNewValue()).divide(BigDecimal.valueOf(1000));
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatSetSharerCommissionRequest.handle(sharerInfoVO.getSharer_appid(), commissionRatio.toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
            if (null != loginUserInfo) {
                Date now = new Date();
                SharerProductCommissionRecord record = new SharerProductCommissionRecord();
                record.setOpenId(loginUserInfo.getOpenId());
                record.setSharerAppid(sharerInfoVO.getSharer_appid());
                record.setProductId("");
                record.setOptType(CommissionChangeOptTypeEnum.DEFAULT_CHANGE.getCode());
                record.setOptInfo(change.getOldValue() + "&" + change.getNewValue());
                record.setOptTime(now);
                record.setOptStatus(isSuccess ? "1" : "2");
                record.setExtInfo("");
                record.setCreateTime(now);
                record.setUpdateTime(now);
                sharerProdutcCommissionRecordMapper.insert(record);
            } else {
                log.error("用户登录状态过期，保存记录失败");
            }
        }
    }

    /**
     * 用户分佣比例变更
     */
    private void sharerCommissionRatioChange(ConfigChange change) {
        List<CommissionSharerBO> oldCommissionSharerList = JSON.parseArray(change.getOldValue(), CommissionSharerBO.class);
        if (null == oldCommissionSharerList) {
            oldCommissionSharerList = new ArrayList<>();
        }
        List<CommissionSharerBO> newCommissionSharerList = JSON.parseArray(change.getNewValue(), CommissionSharerBO.class);
        if (null == newCommissionSharerList) {
            newCommissionSharerList = Collections.emptyList();
        }
        // 将 newCommissionSharerList 转换为 Map，以减少流操作的开销
        Map<String, CommissionSharerBO> newCommissionSharerMap = newCommissionSharerList.stream()
                .collect(Collectors.toMap(CommissionSharerBO::getOpenId, Function.identity()));
        for (CommissionSharerBO oldInfo : oldCommissionSharerList) {
            CommissionSharerBO newInfo = newCommissionSharerMap.get(oldInfo.getOpenId());
            if (newInfo != null) {
                // 修改
                if (newInfo.getCommissionRatio().compareTo(oldInfo.getCommissionRatio()) != 0) {
                    setSharerCommissionInfo(newInfo, oldInfo);
                }
            } else {
                // 删除
                deleteSharerCommissionInfo(oldInfo);
            }
        }
        for (CommissionSharerBO newInfo : newCommissionSharerList) {
            // 新增
            if (oldCommissionSharerList.stream().noneMatch(item -> item.getOpenId().equals(newInfo.getOpenId()))) {
                setSharerCommissionInfo(newInfo, null);
            }
        }
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerCommissionInfo(CommissionSharerBO newCommissionSharer,
                                         CommissionSharerBO oldCommissionSharer) {
        // 分佣比例为小数,  微信接口参数应当分佣比例*1000 为整数
        BigDecimal commissionRatio = newCommissionSharer.getCommissionRatio().divide(BigDecimal.valueOf(1000));
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatSetSharerCommissionRequest.handle(newCommissionSharer.getSharerId(), commissionRatio.toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
            if (null != loginUserInfo) {
                Date now = new Date();
                SharerProductCommissionRecord record = new SharerProductCommissionRecord();
                record.setOpenId(loginUserInfo.getOpenId());
                record.setSharerAppid(newCommissionSharer.getSharerId());
                record.setProductId("");
                record.setOptType(CommissionChangeOptTypeEnum.SET.getCode());
                record.setOptInfo(oldCommissionSharer.getCommissionRatio().toPlainString()
                        + "&" + newCommissionSharer.getCommissionRatio().toPlainString());
                record.setOptTime(now);
                record.setOptStatus(isSuccess ? "1" : "2");
                record.setExtInfo("");
                record.setCreateTime(now);
                record.setUpdateTime(now);
                sharerProdutcCommissionRecordMapper.insert(record);
            } else {
                log.error("用户登录状态过期，保存记录失败");
            }
        }
    }

    /**
     * 删除推客的的分佣类型和比例信息
     */
    private void deleteSharerCommissionInfo(CommissionSharerBO oldCommissionSharer) {
        // 分佣比例为小数,  微信接口参数应当分佣比例*1000 为整数
        BigDecimal commissionRatio = defaultCommissionRatio.divide(BigDecimal.valueOf(1000));
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatSetSharerCommissionRequest.handle(oldCommissionSharer.getSharerId(), commissionRatio.toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
            if (null != loginUserInfo) {
                Date now = new Date();
                SharerProductCommissionRecord record = new SharerProductCommissionRecord();
                record.setOpenId(loginUserInfo.getOpenId());
                record.setSharerAppid(oldCommissionSharer.getSharerId());
                record.setProductId("");
                record.setOptType(CommissionChangeOptTypeEnum.UNSET.getCode());
                record.setOptInfo(oldCommissionSharer.getCommissionRatio().toPlainString() + "&" + defaultCommissionRatio.toPlainString());
                record.setOptTime(now);
                record.setOptStatus(isSuccess ? "1" : "2");
                record.setExtInfo("");
                record.setCreateTime(now);
                record.setUpdateTime(now);
                sharerProdutcCommissionRecordMapper.insert(record);
            } else {
                log.error("用户登录状态过期，保存记录失败");
            }
        }
    }

    /**
     * 用户商品分佣比例变更
     */
    private void sharerProductCommissionRatioChange(ConfigChange change) {
        List<CommissionSharerProductBO> oldCommissionSharerList = JSON.parseArray(change.getOldValue(), CommissionSharerProductBO.class);
        if (null == oldCommissionSharerList) {
            oldCommissionSharerList = new ArrayList<>();
        }
        List<CommissionSharerProductBO> newCommissionSharerList = JSON.parseArray(change.getNewValue(), CommissionSharerProductBO.class);
        if (null == newCommissionSharerList) {
            newCommissionSharerList = Collections.emptyList();
        }
        // 将 newCommissionSharerList 转换为 Map，以减少流操作的开销
        Map<String, CommissionSharerProductBO> newCommissionSharerMap = newCommissionSharerList.stream()
                .collect(Collectors.toMap(this::getSharerProductCommissionMapKey, Function.identity()));
        for (CommissionSharerProductBO oldInfo : oldCommissionSharerList) {
            CommissionSharerProductBO newInfo = newCommissionSharerMap.get(getSharerProductCommissionMapKey(oldInfo));
            if (newInfo != null) {
                // 修改
                if (newInfo.getCommissionRatio().compareTo(oldInfo.getCommissionRatio()) != 0) {
                    setSharerProductCommissionInfo(newInfo, oldInfo);
                }
            } else {
                // 删除
                deleteSharerProductCommissionInfo(oldInfo);
            }
        }
        for (CommissionSharerProductBO newInfo : newCommissionSharerList) {
            // 新增
            if (oldCommissionSharerList.stream().noneMatch(item -> item.getOpenId().equals(newInfo.getOpenId())
                    && item.getProductId().equals(newInfo.getProductId()))) {
                setSharerProductCommissionInfo(newInfo, null);
            }
        }
    }

    private String getSharerProductCommissionMapKey(CommissionSharerProductBO bo) {
        return bo.getOpenId() + "&" + bo.getProductId();
    }

    /**
     * 设置推客的的分佣类型和比例信息
     */
    private void setSharerProductCommissionInfo(CommissionSharerProductBO newBo,
                                                CommissionSharerProductBO oldBo) {
        // 分佣比例为小数,  微信接口参数应当分佣比例*1000 为整数
        BigDecimal commissionRatio = newBo.getCommissionRatio().divide(BigDecimal.valueOf(1000));
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatSetSharerProductCommissionRequest.handle(newBo.getSharerId(), newBo.getProductId(), commissionRatio.toPlainString());
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
            if (null != loginUserInfo) {
                Date now = new Date();
                SharerProductCommissionRecord record = new SharerProductCommissionRecord();
                record.setOpenId(loginUserInfo.getOpenId());
                record.setSharerAppid(newBo.getSharerId());
                record.setProductId(newBo.getProductId().toString());
                record.setOptType(CommissionChangeOptTypeEnum.SET.getCode());
                record.setOptInfo(oldBo.getCommissionRatio().toPlainString()
                        + "&" + newBo.getCommissionRatio().toPlainString());
                record.setOptTime(now);
                record.setOptStatus(isSuccess ? "1" : "2");
                record.setExtInfo("");
                record.setCreateTime(now);
                record.setUpdateTime(now);
                sharerProdutcCommissionRecordMapper.insert(record);
            } else {
                log.error("用户登录状态过期，保存记录失败");
            }
        }
    }

    /**
     * 删除推客的的分佣类型和比例信息
     */
    private void deleteSharerProductCommissionInfo(CommissionSharerProductBO oldBo) {
        // 设置推客分佣比例
        boolean isSuccess = false;
        try {
            // 设置推客分佣比例
            weChatSetSharerProductCommissionRequest.handle(oldBo.getSharerId(), oldBo.getProductId(), null);
            isSuccess = true;
        } catch (Exception e) {
            log.error("设置推客的的分佣类型和比例信息失败", e);
        } finally {
            LoginUserInfo loginUserInfo = LoginUserContextHolder.get();
            if (null != loginUserInfo) {
                Date now = new Date();
                SharerProductCommissionRecord record = new SharerProductCommissionRecord();
                record.setOpenId(loginUserInfo.getOpenId());
                record.setSharerAppid(oldBo.getSharerId());
                record.setProductId(oldBo.getProductId().toString());
                record.setOptType(CommissionChangeOptTypeEnum.UNSET.getCode());
                record.setOptInfo(oldBo.getCommissionRatio().toPlainString() + "&");
                record.setOptTime(now);
                record.setOptStatus(isSuccess ? "1" : "2");
                record.setExtInfo("");
                record.setCreateTime(now);
                record.setUpdateTime(now);
                sharerProdutcCommissionRecordMapper.insert(record);
            } else {
                log.error("用户登录状态过期，保存记录失败");
            }
        }
    }
}