package com.netease.yanxuan.wx.store.sharer.integration.core;

import com.netease.yanxuan.wx.store.sharer.common.constant.CoreConstant;
import com.netease.yanxuan.wx.store.sharer.common.exception.WeChatException;
import com.netease.yanxuan.wx.store.sharer.common.handler.RedisClient;
import com.netease.yanxuan.wx.store.sharer.integration.handler.impl.WeChatShopAccessTokenRequest;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.vo.WeChatAccessTokenVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.util.concurrent.TimeUnit;

/**
 * @Description 微信认证信息上下文持有器
 * @Author fanjiaxin
 * @Date 2025/2/24 11:53
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class WeChatContextHolder {
    private final RedisClient redisClient;
    private final WeChatShopAccessTokenRequest weChatShopAccessTokenRequest;

    private static final long RETRY_DELAY_MS = 200;
    private static final int EXPIRE_SECONDS = 30 * 60;


    /**
     * 获取微信小店认证信息
     */
    public String getShopAccessToken() {
        // 检查锁定
        String accessToken = redisClient.getStr(CoreConstant.REDIS_TOKEN_WECHAT_SHOP_KEY);
        if (StringUtils.isNotBlank(accessToken)) {
            return accessToken;
        }
        return createShopAccessToken();
    }

    /**
     * 创建微信小店认证信息
     */
    public String createShopAccessToken() {
        try {
            // 设置10秒的锁
            boolean locked = redisClient.setIfAbsent(CoreConstant.REDIS_TOKEN_WECHAT_SHOP_LOCK_KEY,
                    CoreConstant.REDIS_TOKEN_WECHAT_SHOP_LOCK_VALUE, CoreConstant.REDIS_TOKEN_WECHAT_SHOP_LOCK_TIME);
            if (!locked) {
                // 未获取到锁，等待一段时间后重试
                TimeUnit.MILLISECONDS.sleep(RETRY_DELAY_MS);
                return getShopAccessToken();
            }
            WeChatAccessTokenVO handle = weChatShopAccessTokenRequest.handle();
            redisClient.setStr(CoreConstant.REDIS_TOKEN_WECHAT_SHOP_KEY, handle.getAccess_token(),
                    handle.getExpires_in() - EXPIRE_SECONDS);
            return handle.getAccess_token();
        } catch (InterruptedException e) {
            throw new WeChatException("微信小店认证AccessToken生成失败", e);
        }
    }
}
