package com.netease.yanxuan.wx.store.sharer.biz.service.impl;

import com.netease.yanxuan.wx.store.sharer.biz.core.LoginUserContextHolder;
import com.netease.yanxuan.wx.store.sharer.biz.meta.enums.SmsScenesEnum;
import com.netease.yanxuan.wx.store.sharer.biz.meta.model.bo.SmsCodeBO;
import com.netease.yanxuan.wx.store.sharer.biz.service.ISmsService;
import com.netease.yanxuan.wx.store.sharer.common.exception.BizException;
import com.netease.yanxuan.wx.store.sharer.common.handler.RedisClient;
import com.netease.yanxuan.wx.store.sharer.common.util.PhoneCheckUtils;
import com.netease.yanxuan.wx.store.sharer.common.util.RandomUtils;
import com.netease.yanxuan.wx.store.sharer.integration.client.IUasClient;
import com.netease.yanxuan.wx.store.sharer.integration.config.SmsConfig;
import com.netease.yanxuan.wx.store.sharer.integration.constant.SmsConstant;
import com.netease.yanxuan.wx.store.sharer.integration.meta.model.bo.SmsCodeScenesConfigBO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.TimeUnit;


/**
 * @Description 短信-业务实现类
 * @Author fanjiaxin
 * @Date 2025/3/10 12:31
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class SmsServiceImpl implements ISmsService {

    private final SmsConfig smsConfig;
    private final IUasClient iUasClient;
    private final RedisClient redisClient;


    @Override
    public void sendSmsCode(SmsCodeBO bo) {
        log.error("[op:sendSmsCode] begin...");
        // 前置校验
        validateSendCode(bo.getScenes(), bo.getMobilePhone());
        boolean success = sendCode(bo.getScenes(), bo.getMobilePhone());
        if (!success) {
            log.error("[op:sendSmsCode] is error...");
            throw new BizException("短信验证码发送失败");
        }
    }

    @Override
    public void verifySmsCode(String scenes, String phone, String code) {
        // 检查每日校验次数上限
        if (!validateVerifyCodeVerifyCount(scenes, phone)) {
            throw new BizException("验证码校验达到次数上限");
        }
        // 对比验证码
        String codeKey = String.format(SmsConstant.SMS_CODE_KEY, scenes, phone);
        String originCode = redisClient.getStr(codeKey);
        if (StringUtils.isBlank(originCode) || !originCode.equals(code)) {
            throw new BizException("验证码有误");
        }
        // 验证成功后清除验证码
        redisClient.delete(codeKey);
    }

    /**
     * 发送短信
     */
    private Boolean sendCode(String scenes, String mobilePhone) {
        log.error("[op:sendSmsCode] sendCode...");
        SmsCodeScenesConfigBO scenesConfigBO = smsConfig.getScenesConfig(scenes);
        if(null == scenesConfigBO){
            throw new BizException("短信配置错误");
        }
        // 生成6位随机数字
        String code = RandomUtils.randomNumber(6);
        // uas发送验证码
        boolean success = iUasClient.sendSmsCode(LoginUserContextHolder.get().getOpenId(),
                mobilePhone, code, scenes, scenesConfigBO);
        if (!success) {
            return false;
        }
        int sendIntervalSeconds = scenesConfigBO.getSendIntervalSeconds();
        int codeValidMinutes = scenesConfigBO.getCodeValidMinutes();
        // 标记验证码已发送
        String sendCodeKey = String.format(SmsConstant.SMS_SEND_INTERVAL_KEY, scenes, mobilePhone);
        redisClient.setStr(sendCodeKey, String.valueOf(System.currentTimeMillis()), sendIntervalSeconds);
        // 验证码放进缓存
        String codeKey = String.format(SmsConstant.SMS_CODE_KEY, scenes, mobilePhone);
        redisClient.setStr(codeKey, code, codeValidMinutes * 60);
        return true;
    }

    /**
     * 手机格式校验以及验证码相关校验
     *
     * @param scenes      场景
     * @param mobilePhone 手机号
     */
    private void validateSendCode(String scenes, String mobilePhone) {
        log.error("[op:sendSmsCode] validate...");
        // 场景标识检查
        if (!SmsScenesEnum.checkScenes(scenes)) {
            throw new BizException("场景标识错误");
        }
        // 手机格式校验
        if (!PhoneCheckUtils.checkPhoneFormat(mobilePhone)) {
            throw new BizException("手机号码格式错误");
        }
        //  验证码发送标记缓存
        if (!validateVerifyCodeSend(scenes, mobilePhone)) {
            throw new BizException("验证码发送过于频繁，请稍后重试");
        }
        //  验证码每日发送次数上限缓存
        if (!validateVerifyCodeSendCount(scenes, mobilePhone)) {
            throw new BizException("验证码发送达到次数上限");
        }
    }

    /**
     * 验证码发送标记
     *
     * @param scenes      场景
     * @param mobilePhone 手机号
     * @return true: 标记不存在，false：标记存在
     */
    private Boolean validateVerifyCodeSend(String scenes, String mobilePhone) {
        String key = String.format(SmsConstant.SMS_SEND_INTERVAL_KEY, scenes, mobilePhone);
        return Boolean.FALSE.equals(redisClient.hasKey(key));
    }

    /**
     * 验证码每日发送次数上限
     *
     * @param scenes      场景
     * @param mobilePhone 手机号
     * @return true: 未达上限，false: 到了上限
     */
    private Boolean validateVerifyCodeSendCount(String scenes, String mobilePhone) {
        String key = String.format(SmsConstant.SMS_SEND_COUNT_KEY, scenes, mobilePhone, getToday());
        BoundValueOperations<String, Object> operations = redisClient.boundValueOps(key);
        Long sendCount = operations.increment();
        operations.expire(1, TimeUnit.DAYS);
        return smsConfig.getScenesConfig(scenes).getSendMaxCount() >= (sendCount == null ? 1 : sendCount);
    }

    /**
     * 验证码每日校验次数上限
     *
     * @param scenes      场景
     * @param mobilePhone 手机号
     * @return true: 未达上限，false: 到了上限
     */
    public Boolean validateVerifyCodeVerifyCount(String scenes, String mobilePhone) {
        String key = String.format(SmsConstant.SMS_VERIFY_COUNT_KEY, scenes, mobilePhone, getToday());
        BoundValueOperations<String, Object> operations = redisClient.boundValueOps(key);
        Long verifyCount = operations.increment();
        operations.expire(1, TimeUnit.DAYS);
        return smsConfig.getScenesConfig(scenes).getVerifyCodeMaxCount() >= (verifyCount == null ? 1 : verifyCount);
    }

    private String getToday() {
        // 获取当前日期
        LocalDate today = LocalDate.now();
        // 定义日期格式
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
        // 格式化日期
        return today.format(formatter);
    }
}
