/**
 * @(#)ChangeFlowBiz.java, 2022/11/15.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.mail.yanxuan.change.biz.biz;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.*;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import com.netease.mail.yanxuan.change.common.enums.*;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import com.alibaba.fastjson.JSON;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.netease.mail.yanxuan.change.biz.config.AppConfig;
import com.netease.mail.yanxuan.change.biz.meta.exception.ExceptionFactory;
import com.netease.mail.yanxuan.change.biz.service.BuildAndSendEmail;
import com.netease.mail.yanxuan.change.biz.service.ChangeFileService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowExecService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowService;
import com.netease.mail.yanxuan.change.biz.service.ChangeSubFlowRecordService;
import com.netease.mail.yanxuan.change.biz.service.InteriorChangeConfigService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeConfigService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeTypeService;
import com.netease.mail.yanxuan.change.biz.service.rpc.FlowService;
import com.netease.mail.yanxuan.change.biz.service.rpc.ItemService;
import com.netease.mail.yanxuan.change.biz.service.rpc.IusService;
import com.netease.mail.yanxuan.change.biz.service.rpc.QCService;
import com.netease.mail.yanxuan.change.biz.service.rpc.SupplierSendService;
import com.netease.mail.yanxuan.change.biz.service.rpc.SupplierService;
import com.netease.mail.yanxuan.change.biz.util.PageUtils;
import com.netease.mail.yanxuan.change.common.bean.CommonConstants;
import com.netease.mail.yanxuan.change.common.bean.RequestLocalBean;
import com.netease.mail.yanxuan.change.common.bean.ResponseCode;
import com.netease.mail.yanxuan.change.common.util.DateUtils;
import com.netease.mail.yanxuan.change.dal.entity.ChangeConfig;
import com.netease.mail.yanxuan.change.dal.entity.ChangeExecRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeFile;
import com.netease.mail.yanxuan.change.dal.entity.ChangeRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeSubFlowRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeType;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeRecordMapper;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeSubFlowRecordMapper;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeCommanderPO;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeConfigPo;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeGoodsPrincipalPO;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeExecConfigReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCancelReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCreateReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowDeliverReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowFile;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowSubmitReq;
import com.netease.mail.yanxuan.change.dal.meta.model.rpc.GoodsResponseRpc;
import com.netease.mail.yanxuan.change.integration.email.enums.EmailTemplateEnum;
import com.netease.mail.yanxuan.change.integration.email.service.IEmailService;
import com.netease.mail.yanxuan.change.integration.excel.ChangeFlowExcelDTO;
import com.netease.mail.yanxuan.change.integration.flow.UserQueryDTO;
import com.netease.mail.yanxuan.change.integration.flow.ius.IusRpcService;
import com.netease.mail.yanxuan.change.integration.flow.ius.req.IusDepartmentReq;
import com.netease.mail.yanxuan.change.integration.flow.ius.rsp.IusUserInfoRsp;
import com.netease.mail.yanxuan.change.integration.flow.ius.rsp.SecondaryDepartments;
import com.netease.mail.yanxuan.change.integration.flow.supplier.rsp.SupplierSimpleRsp;
import com.netease.mail.yanxuan.change.integration.item.SimplePhyCateGoryResultCo;
import com.netease.mail.yanxuan.change.integration.item.meta.SpuTO;
import com.netease.mail.yanxuan.change.integration.item.param.BatchQuerySpuInfoParam;
import com.netease.mail.yanxuan.change.integration.item.param.CommonIdsParamQuery;
import com.netease.mail.yanxuan.change.integration.qc.meta.QcCategoryVO;
import com.netease.yanxuan.flowx.sdk.meta.controller.communal.AjaxResponse;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.FlowDataDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.UserBaseDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.UserReachDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.InterfaceInputDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.UserBaseContainerDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.flow.FlowCreateReqDTO;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import lombok.extern.slf4j.Slf4j;

/**
 * @Author zcwang
 * @Date 2022/11/15
 */
@Component
@Slf4j
public class ChangeFlowBiz {

    @Autowired
    private ChangeConfigService changeConfigService;

    @Autowired
    private ChangeTypeService changeTypeService;

    @Autowired
    private ChangeFlowService changeFlowService;

    @Autowired
    private AppConfig appConfig;

    @Autowired
    private FlowService flowService;

    @Autowired
    private ChangeFileService changeFileService;

    @Autowired
    private ChangeFlowExecService changeFlowExecService;

    @Autowired
    private ChangeSubFlowRecordService changeSubFlowRecordService;

    @Autowired
    private ChangeRecordMapper changeRecordMapper;

    @Autowired
    private ChangeSubFlowRecordMapper changeSubFlowRecordMapper;

    @Autowired
    private ItemService itemService;

    @Autowired
    private InteriorChangeConfigService interiorChangeConfigService;

    @Autowired
    private IEmailService iEmailService;

    @Autowired
    private IusService iusService;

    @Autowired
    private IusRpcService iusRpcService;

    @Autowired
    private DepartmentLeaderBiz departmentLeaderBiz;

    @Autowired
    private SupplierSendService sendSupplierEmail;

    @Autowired
    private SupplierService supplierService;

    @Autowired
    private QCService qcService;

    @Autowired
    private BuildAndSendEmail buildAndSendEmail;

    @Autowired
    private ChangeExecRecordBiz changeExecRecordBiz;

    @Transactional
    public String createAndSubmit(ChangeFlowCreateReq changeFlowCreateReq) {
        String uid = RequestLocalBean.getUid();
        String name = RequestLocalBean.getName();
        log.info("[create] createReq={}, uid:{}, name:{}", JSON.toJSONString(changeFlowCreateReq), uid, name);
        
        // 校验变更主体类型
        ChangeSubjectEnum subjectEnum = ChangeSubjectEnum.getByType(changeFlowCreateReq.getChangeSubject());
        Assert.notNull(subjectEnum, "变更主体类型不能为空或不存在");
        
        // 根据变更类型分别处理
        switch (subjectEnum) {
            case PRODUCT:
                return createProductChangeFlow(changeFlowCreateReq, uid, name);
            case SUPPLIER:
                return createSupplierChangeFlow(changeFlowCreateReq, uid, name);
            case OTHER:
                return createOtherChangeFlow(changeFlowCreateReq, uid, name);
            default:
                throw ExceptionFactory.createBiz(ResponseCode.CHANGE_SUBJECT_ERROR, "不支持的变更主体类型");
        }
    }

    /**
     * 创建商品变更工单
     */
    private String createProductChangeFlow(ChangeFlowCreateReq changeFlowCreateReq, String uid, String name) {
        // 商品变更：校验商品不能为空
        Assert.isTrue(StringUtils.isNotBlank(changeFlowCreateReq.getChangeItems()), "变更商品不可为空");
        // 商品变更需要关联供应商
        if (CreateSourceEnum.TONG_ZHOU.getType().equals(changeFlowCreateReq.getCreateSource())) {
            // 供应商发起：取当前人为供应商
            changeFlowCreateReq.setChangeSupplier(uid);
        } else {
            Assert.isTrue(StringUtils.isNotBlank(changeFlowCreateReq.getChangeSupplier()), "变更供应商不可为空");
        }
        // 解析商品ID列表
        List<Long> itemIds = parseItemIds(changeFlowCreateReq.getChangeItems());
        // 执行公共创建流程（内部已处理邮件逻辑）
        return executeCommonCreateFlow(changeFlowCreateReq, uid, name, itemIds);
    }

    /**
     * 创建供应商变更工单
     */
    private String createSupplierChangeFlow(ChangeFlowCreateReq changeFlowCreateReq, String uid, String name) {
        Assert.isTrue(StringUtils.isNotBlank(changeFlowCreateReq.getChangeSupplier()), "变更供应商不可为空");
        // 执行公共创建流程（内部已处理邮件逻辑）
        return executeCommonCreateFlow(changeFlowCreateReq, uid, name, null);
    }

    /**
     * 创建其他变更工单
     */
    private String createOtherChangeFlow(ChangeFlowCreateReq changeFlowCreateReq, String uid, String name) {
        // 执行公共创建流程（内部已处理邮件逻辑）
        return executeCommonCreateFlow(changeFlowCreateReq, uid, name, null);
    }

    /**
     * 解析商品ID列表
     */
    private List<Long> parseItemIds(String changeItems) {
        List<ItemVO> itemVOS = JSON.parseArray(changeItems, ItemVO.class);
        return itemVOS.stream().map(ItemVO::getItemId).collect(Collectors.toList());
    }

    /**
     * 执行公共的创建流程逻辑
     */
    private String executeCommonCreateFlow(ChangeFlowCreateReq changeFlowCreateReq, String uid, String name,
                                          List<Long> itemIds) {
        Long parentChangeClassId = changeFlowCreateReq.getParentChangeClassId();
        Long sonChangeClassId = changeFlowCreateReq.getSonChangeClassId();
        // 获取工单负责人
        String changeCommander = getChangeCommander(changeFlowCreateReq, uid, parentChangeClassId, sonChangeClassId, itemIds);
        Assert.notNull(changeCommander, "未查询到负责人信息");
        // 检验是否需要资料
        ChangeConfig changeConfig = changeConfigService.getSonChange(sonChangeClassId);
        Assert.notNull(changeConfig, "二级变更类型不存在");
        Integer needFile = changeConfig.getNeedFile();
        if (NeedFileEnum.NEED.getStatus().equals(needFile)) {
            Assert.notEmpty(changeFlowCreateReq.getUploadFiles(), "必须上传资料");
        }
        changeFlowCreateReq.setChangeDepartment(changeConfig.getChangeDepartment());
        // 变更行动项不可为空，最多20项
        List<ChangeExecConfigReq> changeExecProject = changeFlowCreateReq.getChangeExecProject();
        if (CollectionUtils.isEmpty(changeExecProject)) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "行动项列表不能为空");
        }
        Assert.isTrue(changeExecProject.size() <= appConfig.getChangeExecLimit(),
                "变更行动方案配置数超限");
        // 校验每个行动项的必填字段
        changeExecRecordBiz.validateChangeExecProjectRequiredFields(changeExecProject);
        // 校验变更行动方案中是否有重复的变更行动人
        validateDuplicateChangeExecUser(changeExecProject);
        // 结束时间不可晚于第二天定时任务执行时间
        Long tomorrowSpecificTime = DateUtils.getTomorrowSpecificTime("00:00:00");
        Assert.isTrue(changeFlowCreateReq.getChangeConfirmResultTime() >= tomorrowSpecificTime, "时间不可晚于下次执行时间");
        // 构建工单内容
        Map<String, Object> content = buildFlowContent(uid);
        // 构建变更类型
        StringBuilder changeType = buildChangeType(changeFlowCreateReq);
        // 变更工单命名：变更主体+变更类型+变更商品（SPU+商品名称）/变更供应商（供应商ID+供应商名称）
        String subject = ChangeSubjectEnum.getDescByType(changeFlowCreateReq.getChangeSubject());
        String flowName = subject + changeType;
        flowName += buildFlowNameBySubject(changeFlowCreateReq);

        // 组装topo工单创建数据
        FlowCreateReqDTO flowCreateReqDTO = buildFlowCreateReqDTO(ChangeFlowEnum.NEW_CHANGE_FLOW_START.getTopoId(), uid,
                JSON.toJSONString(content), FlowxOperationEnum.CREATE.getName(), name, flowName);
        // 创建工单
        String flowId = flowService.createFlow(flowCreateReqDTO);
        // 查询工单详情
        FlowDataDTO flowDataDTO = flowService.flowDetail(flowId);
        String nodeId = flowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
        // 保存工单数据
        ChangeRecord changeRecord = buildChangeRecord(flowId, nodeId, changeFlowCreateReq, changeCommander, uid);
        changeFlowService.saveRecord(changeRecord);
        // 保存变更行动方案记录（不绑定子流程，待触发时再创建）
        List<ChangeExecRecord> changeExecRecords = changeExecRecordBiz.buildChangeExecRecord(changeRecord.getId(), changeExecProject, null, null);
        changeExecRecords.forEach(exec -> changeFlowExecService.saveRecord(exec));
        
        // 保存附件
        saveChangeFiles(changeRecord.getId(), changeFlowCreateReq);
        
        // 如果发起人=变更负责人，直接提交到执行节点，创建子流程并发送邮件
        if (changeCommander.equals(uid)) {
            submitToExecutionNode(flowId, flowDataDTO, uid, content, changeRecord);
            // 创建子流程并绑定
            createAndBindSubFlows(changeRecord, flowName);
            buildAndSendEmail.buildAndSendEmailCreate(changeRecord, changeExecRecords, changeType, changeCommander, changeFlowCreateReq);
        } else {
            // 如果发起人≠变更负责人，只发送邮件（不提交节点，不创建子流程）
            sendCreateFlowEmail(changeRecord, changeCommander, changeType, changeFlowCreateReq);
        }
        return flowId;
    }

    /**
     * 保存变更文件
     */
    private void saveChangeFiles(Long changeRecordId, ChangeFlowCreateReq changeFlowCreateReq) {
        List<ChangeFile> allFiles = new ArrayList<>();
        // 变更前后图片/视频，非必填
        List<ChangeFlowFile> changeFiles = changeFlowCreateReq.getChangeFiles();
        if (CollectionUtils.isNotEmpty(changeFiles)) {
            allFiles.addAll(buildChangeFileRecord(changeRecordId, changeFiles, FileTypeEnum.CHANGE.getType()));
        }
        // 保存附件，根据配置看是否必传
        List<ChangeFlowFile> uploadFiles = changeFlowCreateReq.getUploadFiles();
        if (CollectionUtils.isNotEmpty(uploadFiles)) {
            allFiles.addAll(buildChangeFileRecord(changeRecordId, uploadFiles, FileTypeEnum.UPLOAD.getType()));
        }
        if (CollectionUtils.isNotEmpty(allFiles)) {
            allFiles.forEach(file -> changeFileService.saveRecord(file));
        }
    }

    /**
     * 创建变更行动工单并绑定行动项
     * 说明：按照 changeExecDepartment + changeExecUserEmail 维度分组，每个分组创建一个变更行动工单
     * 前端传值是拆分的（每个行动项一条记录），但需要汇总到同一个变更行动工单
     *
     * @param changeRecord 主流程记录
     * @param flowName 主流程名称
     */
    private void createAndBindSubFlows(ChangeRecord changeRecord, String flowName) {
        // 从数据库查询执行记录列表
        List<ChangeExecRecord> execRecords = changeFlowExecService.getChangeExecRecordList(changeRecord.getId());
        if (CollectionUtils.isEmpty(execRecords)) {
            log.warn("[createAndBindSubFlows] 未查询到执行记录, changeRecordId:{}", changeRecord.getId());
            return;
        }
        
        // 按照 changeExecDepartment + changeExecUserEmail 维度分组
        Map<String, List<ChangeExecRecord>> groupedRecords = execRecords.stream()
                .filter(record -> StringUtils.isNotBlank(record.getChangeExecUserEmail()))
                .collect(Collectors.groupingBy(record -> 
                    (StringUtils.isNotBlank(record.getChangeExecDepartment()) ? record.getChangeExecDepartment() : "") 
                    + "_" + record.getChangeExecUserEmail()));
        
        if (groupedRecords.isEmpty()) {
            log.warn("[createAndBindSubFlows] 未找到有效的执行记录（行动人邮箱为空）, changeRecordId:{}", changeRecord.getId());
            return;
        }
        
        // 为每个分组创建一个变更行动工单
        for (Map.Entry<String, List<ChangeExecRecord>> entry : groupedRecords.entrySet()) {
            List<ChangeExecRecord> groupRecords = entry.getValue();
            
            if (CollectionUtils.isEmpty(groupRecords)) {
                continue;
            }
            
            // 取第一个记录作为代表（同一分组下的记录，changeExecDepartment 和 changeExecUserEmail 相同）
            ChangeExecRecord firstRecord = groupRecords.get(0);
            String execUserEmail = firstRecord.getChangeExecUserEmail();
            String execDepartment = firstRecord.getChangeExecDepartment();
            
            // 查询行动人名称
            IusUserInfoRsp execUser = iusService.queryUserInfo(execUserEmail);
            String execUserName = execUser == null || StringUtils.isBlank(execUser.getName()) 
                    ? execUserEmail : execUser.getName();
            
            // 构建变更行动工单内容
            Map<String, Object> subFlowContent = buildFlowContent(execUserEmail);
            
            // 构建变更行动工单名称：主流程名称 + 变更行动部门 + 行动人（截取前100字符）
            String subFlowName = flowName + "-" + execDepartment + "-" + execUserName;
            if (subFlowName.length() > 100) {
                subFlowName = subFlowName.substring(0, 100) + "...";
            }
            
            // 创建变更行动工单
            FlowCreateReqDTO subFlowCreateReqDTO = buildFlowCreateReqDTO(ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), 
                    execUserEmail, JSON.toJSONString(subFlowContent), FlowxOperationEnum.CREATE.getName(), 
                    execUserName, subFlowName);
            String subFlowId = flowService.createFlow(subFlowCreateReqDTO);
            // 查询工单详情
            FlowDataDTO subFlowDataDTO = flowService.flowDetail(subFlowId);
            // 获取流程节点ID
            String subNodeId = subFlowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
            
            // 创建变更行动工单记录
            // 变更行动工单审批人：第一次新建时就是行动人
            String approverJson = JSON.toJSONString(Collections.singletonList(execUserEmail));
            // 初始状态：待确认行动方案
            ChangeSubFlowRecord subFlowRecord = ChangeSubFlowRecord.builder()
                    .changeRecordId(changeRecord.getId())
                    .subFlowId(subFlowId)
                    .subFlowNode(subNodeId)
                    .status(ChangeSubFlowStatusEnum.WAIT_CONFIRM_ACTION_PLAN.getStatus())
                    .approver(approverJson)
                    .changeExecUserEmail(execUserEmail)
                    .changeExecDepartment(execDepartment)
                    .changeCommander(changeRecord.getChangeCommander())
                    .changeConfirmResultTime(changeRecord.getChangeConfirmResultTime())
                    .createTime(DateUtils.getCurrentTime())
                    .updateTime(DateUtils.getCurrentTime())
                    .build();
            changeSubFlowRecordService.saveRecord(subFlowRecord);
            
            log.info("[createAndBindSubFlows] changeRecordId:{}, subFlowRecordId:{}, subFlowId:{}, nodeId:{}, execDepartment:{}, execUserEmail:{}, execCount:{}",
                    changeRecord.getId(), subFlowRecord.getId(), subFlowId, subNodeId, execDepartment, execUserEmail, groupRecords.size());
            
            // 将该分组下的所有行动项关联到变更行动工单
            for (ChangeExecRecord execRecord : groupRecords) {
                execRecord.setSubFlowRecordId(subFlowRecord.getId());
                execRecord.setSubFlowId(subFlowId);
                execRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeFlowExecService.update(execRecord);
            }
        }
    }

    /**
     * 提交到执行节点
     */
    private void submitToExecutionNode(String flowId, FlowDataDTO flowDataDTO, String uid, Map<String, Object> content, ChangeRecord changeRecord) {
        log.debug("[submitToExecutionNode] flowId:{}, uid:{}", flowId, uid);
        String nextNodeId = flowService.submitFlow(flowId, flowDataDTO, uid,
                ChangeFlowEnum.NEW_CHANGE_FLOW_START.getTopoId(), JSON.toJSONString(content), true,
                FlowxOperationEnum.SUBMIT.getName(), "提交工单", changeRecord.getCreateTime());
        if (null == nextNodeId) {
            throw ExceptionFactory.createBiz(ResponseCode.SUBMIT_FLOW_ERROR, "首次提交节点失败");
        }
        // 更新节点id，使用返回的节点ID
        changeRecord.setFlowNode(nextNodeId);
        changeRecord.setState(ChangeStatusEnum.WAIT_SUBMIT_CHANGE_APPLY.getStatus());
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
        changeFlowService.updateRecord(changeRecord);
    }

    /**
     * 发送创建流程邮件
     *
     * @param changeRecord 变更记录
     * @param changeCommander 变更负责人
     * @param changeType 变更类型
     * @param changeFlowCreateReq 创建请求
     */
    private void sendCreateFlowEmail(ChangeRecord changeRecord, String changeCommander, StringBuilder changeType, ChangeFlowCreateReq changeFlowCreateReq) {
        String uid = RequestLocalBean.getUid();
        
        // 如果发起人=变更负责人，已经在 executeCommonCreateFlow 中发送了邮件
        if (changeCommander.equals(uid)) {
            return;
        }
        
        // 如果发起人≠变更负责人，停留在变更申请提交节点，发送邮件
        HashMap<String, Object> param = new HashMap<>();
        param.put("changeId", changeRecord.getFlowId());
        param.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
        param.put("changeType", changeType.toString());
        param.put("flowUrl", changeRecord.getFlowId());
        String subjectParam = changeRecord.getFlowId().toString();
        
        // 发起变更,收件人：变更负责人
        List<String> receiver = Collections.singletonList(changeCommander);
        List<String> ccList = departmentLeaderBiz.getDepartmentLeaders(receiver);
        // 发起变更，抄送：变更负责人上一级主管、变更管理QM
        ccList.add(appConfig.getChangeManageQM());
        qcSendEmail(receiver, ccList, subjectParam, EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, param);
        
        // 如果是同舟端，发送供应商邮件
        if (changeFlowCreateReq.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())) {
            sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), subjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, param);
        } else if (ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
            // 如果是其他端发起但是是供应商变更，发送供应商邮件
            sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), subjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, param);
        }
    }


    /**
     * 严选QC端发送邮件
     *
     * @param subjectParam      主体参数
     * @param receiver          收件人
     * @param emailTemplateEnum 邮件模板
     * @param param             正文参数
     */
    public void qcSendEmail(List<String> receiver, List<String> ccList, String subjectParam, EmailTemplateEnum emailTemplateEnum,
                            Map<String, Object> param) {

        try {
            iEmailService.sendEmail(receiver, ccList, param, emailTemplateEnum,
                    subjectParam);
        } catch (Exception e) {
            log.error("[op:qcSendEmail] error,receiver:{}, ccList:{}, subjectParam:{}, e:{}", receiver, ccList, subjectParam,
                    e);
        }
    }

    private List<ChangeFile> buildChangeFileRecord(Long changeRecordId, List<ChangeFlowFile> files, Integer type) {
        return files.stream().map(f -> {
            ChangeFile changeFile = new ChangeFile();
            changeFile.setChangeRecordId(changeRecordId);
            changeFile.setFileType(type);
            changeFile.setFileName(f.getFileName());
            changeFile.setFileUrl(f.getFileUrl());
            changeFile.setCreateTime(DateUtils.getCurrentTime());
            changeFile.setUpdateTime(DateUtils.getCurrentTime());
            return changeFile;
        }).collect(Collectors.toList());
    }

    /**
     * 构建工单内容
     *
     * @param uid 创建人
     * @return 工单内容
     */
    private Map<String, Object> buildFlowContent(String uid) {
        Map<String, Object> content = new HashMap<>(10);
        content.put("createUserName", uid);
        content.put("createUser", uid);
        content.put("createTime", System.currentTimeMillis());
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.PASS.getValue());
        return content;
    }

    private FlowCreateReqDTO buildFlowCreateReqDTO(String topoId, String uid, String content, String operateResult,
        String name, String flowName) {
        FlowCreateReqDTO flowCreateReqDTO = new FlowCreateReqDTO();
        flowCreateReqDTO.setTopoId(topoId);
        flowCreateReqDTO.setUid(uid);
        flowCreateReqDTO.setUserName(name);
        flowCreateReqDTO.setOperateResult(operateResult);
        flowCreateReqDTO.setWorkOrderId(StringUtils.joinWith("-", topoId, UUID.randomUUID().toString()));
        flowCreateReqDTO.setContent(content);
        flowCreateReqDTO.setFlowName(flowName);
        return flowCreateReqDTO;
    }


    private ChangeRecord buildChangeRecord(String flowId, String nodeId, ChangeFlowCreateReq changeFlowCreateReq,
                                           String changeCommander, String uid) {
        ChangeRecord changeRecord = new ChangeRecord();
        changeRecord.setFlowId(Long.parseLong(flowId));
        changeRecord.setFlowNode(nodeId);
        changeRecord.setChangeSubject(changeFlowCreateReq.getChangeSubject());
        changeRecord.setParentChangeClassId(changeFlowCreateReq.getParentChangeClassId());
        changeRecord.setSonChangeClassId(changeFlowCreateReq.getSonChangeClassId());
        changeRecord.setChangeLevel(changeFlowCreateReq.getChangeLevel());
        changeRecord.setChangeCommander(changeCommander);
        changeRecord.setChangeDepartment(changeFlowCreateReq.getChangeDepartment());
        List<ChangeExecConfigReq> changeExecProject = changeFlowCreateReq.getChangeExecProject();
        List<String> execDepartmentList = changeExecProject.stream().map(ChangeExecConfigReq::getChangeExecDepartment)
                .collect(Collectors.toList());
        changeRecord.setParticipateChangeExecDepartment(JSON.toJSONString(execDepartmentList));
        if (ChangeSubjectEnum.PRODUCT.getType().equals(changeFlowCreateReq.getChangeSubject())) {
            // 当变更类型是商品时有值
            changeRecord.setChangeItem(changeFlowCreateReq.getChangeItems());
            changeRecord.setChangeSku(changeFlowCreateReq.getChangeSkus());
            // 商品变更需要关联供应商
            changeRecord.setChangeSupplier(changeFlowCreateReq.getChangeSupplier());
        }
        if (ChangeSubjectEnum.SUPPLIER.getType().equals(changeFlowCreateReq.getChangeSubject())) {
            changeRecord.setChangeSupplier(changeFlowCreateReq.getChangeSupplier());
        }
        changeRecord.setChangeReason(changeFlowCreateReq.getChangeReason());
        changeRecord.setChangeContent(changeFlowCreateReq.getChangeContent());
        changeRecord.setChangeRiskDesc(changeFlowCreateReq.getChangeRiskDesc());
        changeRecord.setChangeProfit(changeFlowCreateReq.getChangeProfit());
        changeRecord.setChangeProfitDesc(changeFlowCreateReq.getChangeProfitDesc());
        changeRecord.setChangeConfirmResultTime(changeFlowCreateReq.getChangeConfirmResultTime());
        changeRecord.setState(ChangeStatusEnum.WAIT_SUBMIT_CHANGE_APPLY.getStatus());
        // 变更结论
        changeRecord.setCreateSource(changeFlowCreateReq.getCreateSource());
        changeRecord.setCreateSupplier(changeFlowCreateReq.getSupplier());
        changeRecord.setCreateTime(DateUtils.getCurrentTime());
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
        changeRecord.setCreator(uid);
        changeRecord.setChangeProfitAmount(changeFlowCreateReq.getChangeProfitAmount());
        // 批次说明，非必传，有的话需要填充
        if (StringUtils.isNotBlank(changeFlowCreateReq.getBatchDescription())) {
            changeRecord.setBatchDescription(changeFlowCreateReq.getBatchDescription());
        }
        return changeRecord;
    }

    public String submit(ChangeFlowSubmitReq changeFlowSubmitReq) {
        log.info("[submitFlow] changeFlowReq:{}", JSON.toJSONString(changeFlowSubmitReq));
        Long flowId = changeFlowSubmitReq.getFlowId();
        // 查询工单有效性
        ChangeRecord changeRecord = this.getFlowInfo(flowId);
        String uid = RequestLocalBean.getUid();
        // 历史工单负责人建单时固定，新流程有不同的节点处理人
        //String changeCommander = changeRecord.getChangeCommander();
        //if (!uid.equals(changeCommander)) {
        //    throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, ResponseCode.NO_AUTH.getMsg());
        //}
        // 获取工单详情（topo实际节点信息）
        FlowDataDTO flowDataDTO = flowService.flowDetail(flowId.toString());
        if (flowDataDTO == null) {
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "工单查询错误，不存在");
        }
        // 使用topo实际节点信息进行节点检查
        String actualNodeId = flowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
        if (!actualNodeId.equals(changeFlowSubmitReq.getCurrentNodeId())) {
            throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, "工单已流转至其他节点");
        }

        return checkUpdateAndSubmit(flowId, flowDataDTO, uid, changeRecord, actualNodeId, changeFlowSubmitReq);
    }

    private String checkUpdateAndSubmit(Long flowId, FlowDataDTO flowDataDTO, String uid, ChangeRecord changeRecord,
                                        String currentNode, ChangeFlowSubmitReq changeFlowSubmitReq) {
        ChangeFlowEnum node = ChangeFlowEnum.getByNodeId(currentNode);
        Assert.notNull(node, "节点配置不存在");
        log.debug("[checkUpdateAndSubmit] flowId:{}, nodeEnum:{}, changeFlowSubmitReq:{}", flowId, node,
                JSON.toJSONString(changeFlowSubmitReq));
        // 工单流传
        Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.PASS.getValue());
        switch (node) {
            // 提交变更申请节点，可以修改数据，以新的数据为准
            case NEW_CHANGE_FLOW_START:
                // 检验是否需要资料
                ChangeConfig changeConfig = changeConfigService.getSonChange(changeFlowSubmitReq.getSonChangeClassId());
                Assert.notNull(changeConfig, "二级变更类型不存在");
                Integer needFile = changeConfig.getNeedFile();
                if (NeedFileEnum.NEED.getStatus().equals(needFile)) {
                    List<ChangeFlowFile> uploadFiles = changeFlowSubmitReq.getUploadFiles();
                    if (CollectionUtils.isEmpty(uploadFiles)) {
                        throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "必须上传资料");
                    }
                }
                // 变更行动项不可为空，最多20项
                List<ChangeExecConfigReq> changeExecProjectList = changeFlowSubmitReq.getChangeExecProjectList();
                if (CollectionUtils.isEmpty(changeExecProjectList)) {
                    throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "变更行动方案不能为空");
                }
                if (changeExecProjectList.size() > appConfig.getChangeExecLimit()) {
                    throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "变更行动方案配置数超限");
                }
                // 校验每个行动项的必填字段
                changeExecRecordBiz.validateChangeExecProjectRequiredFields(changeExecProjectList);
                // 校验变更行动方案中是否有重复的变更行动人
                validateDuplicateChangeExecUser(changeExecProjectList);
                changeRecord.setParentChangeClassId(changeFlowSubmitReq.getParentChangeClassId());
                changeRecord.setSonChangeClassId(changeFlowSubmitReq.getSonChangeClassId());
                List<String> execDepartmentList = changeExecProjectList.stream().map(ChangeExecConfigReq::getChangeExecDepartment)
                        .collect(Collectors.toList());
                changeRecord.setParticipateChangeExecDepartment(JSON.toJSONString(execDepartmentList));
                changeRecord.setChangeItem(changeFlowSubmitReq.getChangeItems());
                changeRecord.setChangeSku(changeFlowSubmitReq.getChangeSkus());
                changeRecord.setChangeSupplier(changeFlowSubmitReq.getChangeSupplier());
                changeRecord.setChangeReason(changeFlowSubmitReq.getChangeReason());
                changeRecord.setChangeContent(changeFlowSubmitReq.getChangeContent());
                changeRecord.setChangeRiskDesc(changeFlowSubmitReq.getChangeRiskDesc());
                changeRecord.setChangeChecking(changeFlowSubmitReq.getChangeChecking());
                changeRecord.setChangeProfit(changeFlowSubmitReq.getChangeProfit());
                changeRecord.setChangeProfitDesc(changeFlowSubmitReq.getChangeProfitDesc());
                changeRecord.setChangeConfirmResultTime(changeFlowSubmitReq.getChangeConfirmResultTime());
                changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeRecord.setChangeProfitAmount(changeFlowSubmitReq.getChangeProfitAmount());
                // 提交工单并更新节点
                submitToExecutionNode(flowId.toString(), flowDataDTO, uid, content, changeRecord);
                String submitNode = changeRecord.getFlowNode();
                // 更新行动执行方案，覆盖操作，先删除，后插入
                Integer changeExecCount = changeFlowExecService.deleteByChangeRecordId(changeRecord.getId());
                log.debug("[NEW_CHANGE_FLOW_START] delete id:{},  changeExecCount:{}", changeRecord.getId(), changeExecCount);
                // 保存变更行动方案记录
                List<ChangeExecRecord> changeExecRecords = changeExecRecordBiz.buildChangeExecRecord(changeRecord.getId(),
                        changeExecProjectList, null, null);
                changeExecRecords.forEach(exec -> changeFlowExecService.saveRecord(exec));
                // 更新附件，覆盖操作，先删除，后插入
                Integer fileCount = changeFileService.deleteByChangeRecordId(changeRecord.getId());
                log.debug("[NEW_CHANGE_FLOW_START] delete fileCount:{}", fileCount);
                List<ChangeFile> allFiles = new ArrayList<>();
                List<ChangeFlowFile> changeFiles = changeFlowSubmitReq.getChangeFiles();
                if (CollectionUtils.isNotEmpty(changeFiles)) {
                    allFiles.addAll(buildChangeFileRecord(changeRecord.getId(), changeFiles, FileTypeEnum.CHANGE.getType()));
                }
                // 保存附件，根据配置看是否必传
                List<ChangeFlowFile> uploadFiles = changeFlowSubmitReq.getUploadFiles();
                if (CollectionUtils.isNotEmpty(uploadFiles)) {
                    allFiles.addAll(buildChangeFileRecord(changeRecord.getId(), uploadFiles, FileTypeEnum.UPLOAD.getType()));
                }
                if (CollectionUtils.isNotEmpty(allFiles)) {
                    allFiles.forEach(file -> changeFileService.saveRecord(file));
                }
                log.debug("[NEW_CHANGE_FLOW_START] changeFiles:{}, uploadFiles:{}, allFiles:{}", JSON.toJSONString(changeFiles),
                    JSON.toJSONString(uploadFiles), JSON.toJSONString(allFiles));
                // 创建子流程并绑定（使用行动人作为创建人和审批人）
                String flowName = flowDataDTO.getFlowMeta() != null && StringUtils.isNotBlank(flowDataDTO.getFlowMeta().getFlowName()) 
                        ? flowDataDTO.getFlowMeta().getFlowName() : "";
                createAndBindSubFlows(changeRecord, flowName);
                buildAndSendEmail.buildAndSendEmailSubmit(changeRecord, changeExecRecords);
                return submitNode;
            case NEW_CHANGE_FLOW_CONFIRM_EXEC_PLAN:
                // 确认变更方案节点：由子单审批通过后自动触发，不应该由前端主动调用
                throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, 
                        "该节点不允许主动提交，需等待所有子单审批通过后自动流转");
            case NEW_CHANGE_FLOW_OWNER_APPROVE:
            case NEW_CHANGE_FLOW_ADMIN_APPROVE:
            case NEW_CHANGE_FLOW_QUALITY_APPROVE:
                // 部门负责人、质量部负责人、变更管理员审批节点：检查所有子单是否都已审批通过
                return null;
            case NEW_CHANGE_FLOW_EXE:
                // todo:
                // 这些节点检查所有子单是否都已完成（FINISHED或CANCELLED）
                if (!checkAllSubFlowsFinishedForMainFlow(changeRecord.getId())) {
                    throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST,
                            "存在子单未完成，无法提交主工单");
                }

                // 所有子单都已完成，提交主工单到下一节点
                String mainFlowNextNodeId = flowService.submitFlow(flowId.toString(), flowDataDTO, uid,
                        ChangeFlowEnum.NEW_CHANGE_FLOW.getTopoId(), JSON.toJSONString(content), true,
                        FlowxOperationEnum.SUBMIT.getName(), "所有子单已完成，主工单流转", changeRecord.getCreateTime());
                changeRecord.setFlowNode(mainFlowNextNodeId);
                changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeFlowService.updateRecord(changeRecord);
                log.info("[NEW_CHANGE_FLOW] 主工单流转成功，flowId:{}, node:{}, nextNodeId:{}",
                        flowId, node, mainFlowNextNodeId);
                return mainFlowNextNodeId;
            case NEW_CHANGE_FLOW_CONFIRM:
                // todo:
                return null;
            case CHANGE_FLOW_CONFIRM:
                Integer changeResult = changeFlowSubmitReq.getChangeResult();
                ChangeResultEnum changeResultStatus = ChangeResultEnum.getByStatus(changeResult);
                Assert.notNull(changeResultStatus, "变更结论类型错误");
                // 填写变更结论
                changeRecord.setChangeResult(changeResultStatus.getStatus());
                // 填写备注
                changeRecord.setRemark(changeFlowSubmitReq.getRemark());
                List<ChangeFlowFile> finialFiles = changeFlowSubmitReq.getChangeResultFiles();
                if (CollectionUtils.isNotEmpty(finialFiles)) {
                    changeFileService.deleteByChangeRecordIdAndType(changeRecord.getId(), FileTypeEnum.CHANGE_RESULT_INFO);
                    List<ChangeFile> finialFileList = buildChangeFileRecord(changeRecord.getId(), finialFiles, FileTypeEnum.CHANGE_RESULT_INFO.getType());
                    finialFileList.forEach(file -> changeFileService.saveRecord(file));
                }
                switch (changeResultStatus) {
                    case FINISH_ALL:
                    case FINISH_PART:
                        String reason = changeFlowSubmitReq.getCancelReason();
                        changeRecord.setCancelReason(reason);
                        List<ChangeExecConfigReq> execProjectList = changeFlowSubmitReq.getChangeExecProjectList();
                        List<ChangeExecRecord> changeExecRecordList = execProjectList.stream().map(exec -> {
                            Long changeExecFinishTime = exec.getChangeExecFinishTime();
                            Assert.notNull(changeExecFinishTime, "行动完成时间不可为空" + exec.getChangeTemplateId());
                            ChangeExecRecord changeExecRecord = new ChangeExecRecord();
                            changeExecRecord.setId(exec.getChangeExecId());
                            changeExecRecord.setChangeExecFinishTime(exec.getChangeExecFinishTime());
                            changeExecRecord.setChangeExecFinishDesc(exec.getChangeExecFinishDesc());
                            return changeExecRecord;
                        }).collect(Collectors.toList());
                        // 完成直接完结
                        String confirmNode = flowService.submitFlow(flowId.toString(), flowDataDTO, uid,
                                ChangeFlowEnum.CHANGE_FLOW_CONFIRM.getTopoId(), JSON.toJSONString(content), true,
                                FlowxOperationEnum.SUBMIT.getName(), "提交工单", changeRecord.getCreateTime());
                        changeRecord.setState(ChangeStatusEnum.END.getStatus());
                        changeRecord.setFlowNode(confirmNode);
                        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                        changeFlowService.updateRecord(changeRecord);
                        changeExecRecordList.forEach(exec -> changeFlowExecService.update(exec));
                        // 发送邮件
                        HashMap<String, Object> finishPartMap = new HashMap<>();
                        finishPartMap.put("changeId", changeRecord.getFlowId());
                        finishPartMap.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
                        finishPartMap.put("flowUrl", changeRecord.getFlowId());
                        ChangeConfigPo changeConfigPo = changeConfigService
                                .queryInfoPo(changeRecord.getParentChangeClassId(), changeRecord.getSonChangeClassId());
                        StringBuilder changeType = new StringBuilder();
                        try {
                            changeType.append(changeConfigPo.getChangeTypes().get(0).getTypeName());
                        } catch (Exception e) {
                            log.error("queryInfoPo error:{}", JSON.toJSONString(changeConfigPo));
                        }
                        try {
                            changeType.append(changeConfigPo.getChangeTypes().get(1).getTypeName());
                        } catch (Exception e) {
                            log.error("queryInfoPo error:{}", JSON.toJSONString(changeConfigPo));
                        }
                        //【已完结】变更主体+变更内容+工单ID 变更风险行动项已完结，请开始执行变更。
                        String finishSubjectParam = ChangeSubjectEnum
                            .getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc()
                            + changeRecord.getChangeContent() + changeRecord.getFlowId();
                        // 变更确认，收件人：变更负责人+变更供应商+所有变更行动人+变更管理QM（cuiyixian@corp.netease.com）
                        List<ChangeFlowExecVO> execRecord = changeFlowExecService
                                .getChangeFlowExecRecord(changeRecord.getId());
                        List<String> userEmailList = execRecord.stream().map(ChangeFlowExecVO::getChangeExecUserEmail)
                                .collect(Collectors.toList());
                        // 变更确认，抄送人：所有变更行动人上一级LEADER+变更管理委员会（yx_change_SC@qun.mail.163.com）
                        List<String> list = departmentLeaderBiz.getDepartmentLeaders(userEmailList);
                        list.add(appConfig.getChangeCommittee());
                        userEmailList.add(changeRecord.getChangeCommander());
                        userEmailList.add(appConfig.getChangeManageQM());
                        qcSendEmail(userEmailList, list, finishSubjectParam,
                                EmailTemplateEnum.YX_QC_CHANGE_FINISH, finishPartMap);
                        // 如果是供应商，再次发送供应商邮件
                        if (changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())) {
                            sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), finishSubjectParam,
                                    EmailTemplateEnum.YX_QC_CHANGE_FINISH, finishPartMap);
                        }
                        // 如果是其他端发起但是是供应商变更，再次发送供应商邮件
                        if (!changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())
                                && ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
                            sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), finishSubjectParam,
                                    EmailTemplateEnum.YX_QC_CHANGE_FINISH, finishPartMap);
                        }
                        return confirmNode;
                    case CANCEL:
                        // 取消填写取消原因
                        String cancelReason = changeFlowSubmitReq.getCancelReason();
                        Assert.isTrue(StringUtils.isNotBlank(cancelReason), "变更取消原因不可为空");
                        changeRecord.setCancelReason(cancelReason);
                        List<ChangeExecRecord> cancelChangeExecRecordList = changeFlowSubmitReq
                            .getChangeExecProjectList().stream().map(exec -> {
                                Long changeExecFinishTime = exec.getChangeExecFinishTime();
                            Assert.notNull(changeExecFinishTime, "行动完成时间不可为空" + exec.getChangeTemplateId());
                            ChangeExecRecord changeExecRecord = new ChangeExecRecord();
                            changeExecRecord.setId(exec.getChangeExecId());
                            changeExecRecord.setChangeExecFinishTime(exec.getChangeExecFinishTime());
                            changeExecRecord.setChangeExecFinishDesc(exec.getChangeExecFinishDesc());
                            return changeExecRecord;
                        }).collect(Collectors.toList());
                        String cancelNode = flowService.submitFlow(flowId.toString(), flowDataDTO, uid,
                                ChangeFlowEnum.CHANGE_FLOW_CONFIRM.getTopoId(), JSON.toJSONString(content), true,
                                FlowxOperationEnum.SUBMIT.getName(), "提交工单", changeRecord.getCreateTime());
                        changeRecord.setState(ChangeStatusEnum.END.getStatus());
                        changeRecord.setFlowNode(cancelNode);
                        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                        changeFlowService.updateRecord(changeRecord);
                        cancelChangeExecRecordList.forEach(exec -> changeFlowExecService.update(exec));
                        // 发送邮件
                        HashMap<String, Object> cancelMap = new HashMap<>();
                        cancelMap.put("changeId", changeRecord.getFlowId());
                        cancelMap.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
                        cancelMap.put("flowUrl", changeRecord.getFlowId());
                        ChangeConfigPo changeConfigPo1 = changeConfigService
                                .queryInfoPo(changeRecord.getParentChangeClassId(), changeRecord.getSonChangeClassId());
                        StringBuilder changeType1 = new StringBuilder();
                        try {
                            changeType1.append(changeConfigPo1.getChangeTypes().get(0).getTypeName());
                        } catch (Exception e) {
                            log.error("queryInfoPo error:{}", JSON.toJSONString(changeConfigPo1));
                        }
                        try {
                            changeType1.append(changeConfigPo1.getChangeTypes().get(1).getTypeName());
                        } catch (Exception e) {
                            log.error("queryInfoPo error:{}", JSON.toJSONString(changeConfigPo1));
                        }
                        //【已完结】变更主体+变更内容+工单ID 变更风险行动项已完结，请开始执行变更。
                        String cancelFinishSubjectParam = ChangeSubjectEnum
                            .getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc()
                            + changeRecord.getChangeContent() + changeRecord.getFlowId();
                        // 变更确认，收件人：变更负责人+变更供应商+所有变更行动人+变更管理QM（cuiyixian@corp.netease.com）
                        List<ChangeFlowExecVO> execRecordList = changeFlowExecService
                                .getChangeFlowExecRecord(changeRecord.getId());
                        List<String> userEmails = execRecordList.stream().map(ChangeFlowExecVO::getChangeExecUserEmail)
                                .collect(Collectors.toList());
                        // 变更确认，抄送人：所有变更行动人上一级LEADER+变更管理委员会（yx_change_SC@qun.mail.163.com）
                        List<String> departmentInfoList = departmentLeaderBiz.getDepartmentLeaders(userEmails);
                        departmentInfoList.add(appConfig.getChangeCommittee());
                        userEmails.add(changeRecord.getChangeCommander());
                        userEmails.add(appConfig.getChangeManageQM());
                        qcSendEmail(userEmails, departmentInfoList, cancelFinishSubjectParam,
                                EmailTemplateEnum.YX_QC_CHANGE_FINISH, cancelMap);
                        // 如果是供应商，再次发送供应商邮件
                        if (changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())) {
                            sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), cancelFinishSubjectParam,
                                    EmailTemplateEnum.YX_QC_CHANGE_FINISH, cancelMap);
                        }
                        // 如果是其他端发起但是是供应商变更，再次发送供应商邮件
                        if (!changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())
                                && ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
                            sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), cancelFinishSubjectParam,
                                    EmailTemplateEnum.YX_QC_CHANGE_FINISH, cancelMap);
                        }
                        return cancelNode;
                    case DELAY:
                        // 延期不流转工单
                        Long changeConfirmResultTime = changeFlowSubmitReq.getChangeConfirmResultTime();
                        Long tomorrowSpecificTime = DateUtils.getTomorrowSpecificTime("00:00:00");
                        Assert.isTrue(
                            changeConfirmResultTime != null && changeConfirmResultTime >= tomorrowSpecificTime,
                            "时间不可晚于下次执行时间");
                        // 更改变更结果确认时间
                        changeRecord.setChangeConfirmResultTime(changeConfirmResultTime);
                        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                        // 延期需要重新发邮件，改变执行邮件状态
                        changeRecord.setAutoSubmitEmail(0);
                        changeFlowService.updateRecord(changeRecord);
                        return null;
                    default:
                        return null;
                }
                // 其他几点，不可提交
            case CHANGE_FLOW_EXE:
            case CHANGE_FLOW_START:
            case END:
            default:
                throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, "不可提交节点" + currentNode);
        }
    }

    private ChangeRecord getFlowInfo(Long flowId) {
        ChangeRecord changeRecord = changeFlowService.getByFlowId(flowId);
        if (changeRecord == null) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "工单id不存在");
        }
        return changeRecord;
    }

    private void checkNode(String recordNode, List<String> checkNode) {
        Optional<String> nodeOptional = checkNode.stream().filter(check -> check.equals(recordNode)).findAny();
        if (!nodeOptional.isPresent()) {
            throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, "工单已流转至其他节点");
        }
    }

    public Boolean cancel(ChangeFlowCancelReq req) {
        log.info("[cancel] req:{}", JSON.toJSONString(req));
        Long flowId = req.getFlowId();
        // 查询工单有效性
        ChangeRecord changeRecord = getFlowInfo(flowId);
        log.info("[cancel] nodeId:{}", changeRecord.getFlowNode());
        // 检查工单节点
        this.checkNode(changeRecord.getFlowNode(), Collections.singletonList(ChangeFlowEnum.CHANGE_FLOW_SUBMIT.getNodeId()));
        String uid = RequestLocalBean.getUid();
        String changeCommander = changeRecord.getChangeCommander();
        if (!uid.equals(changeCommander)) {
            throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, ResponseCode.NO_AUTH.getMsg());
        }
        // 获取工单详情
        FlowDataDTO flowDataDTO = flowService.flowDetail(flowId.toString());
        if (flowDataDTO == null) {
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "工单查询错误，不存在");
        }
        // 工单流转
        Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.REFUSE.getValue());
        String nextNodeId = flowService.submitFlow(String.valueOf(flowId), flowDataDTO, uid,
                ChangeFlowEnum.CHANGE_FLOW_SUBMIT.getTopoId(), JSON.toJSONString(content), false,
                FlowxOperationEnum.APPROVE_FAIL.getName(), "取消工单", changeRecord.getCreateTime());
        log.info("[cancel] flowId:{}, nextNodeId:{}", flowId, nextNodeId);
        // 填充更新数据
        changeRecord.setFlowNode(nextNodeId);
        changeRecord.setState(ChangeStatusEnum.CANCEL.getStatus());
        changeRecord.setCancelReason(req.getCancelReason());
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
        Map<String, Object> cancelMap = new HashMap<>();
        cancelMap.put("changeId", changeRecord.getFlowId());
        cancelMap.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
        cancelMap.put("changeContent", changeRecord.getChangeContent());
        IusUserInfoRsp user = iusService.queryUserInfo(changeRecord.getChangeCommander());
        cancelMap.put("changeCommander", user == null ? changeRecord.getChangeCommander() : user.getName());
        cancelMap.put("cancelReason", changeRecord.getCancelReason());
        String cancelSubjectParam = changeRecord.getFlowId().toString();
        List<String> receiver = new ArrayList<>(Collections.singletonList(changeCommander));
        List<ChangeFlowExecVO> execRecord = changeFlowExecService
                .getChangeFlowExecRecord(changeRecord.getId());
        List<String> userEmailList = execRecord.stream().map(ChangeFlowExecVO::getChangeExecUserEmail)
                .collect(Collectors.toList());
        receiver.addAll(userEmailList);
        // 取消变更，抄送人：变更负责人+变更行动人上一级主管、变更管理QM（cuiyixian@corp.netease.com）
        userEmailList.addAll(receiver);
        List<String> ccList = departmentLeaderBiz.getDepartmentLeaders(userEmailList);
        ccList.add(appConfig.getChangeManageQM());
        // 取消变更，收件人：变更发起人（供应商邮箱号或严选发起人）、变更负责人、变更行动人
        String creator = changeRecord.getCreator();
        if (!creator.equals(changeCommander)) {
            receiver.add(creator);
        }
        qcSendEmail(receiver, ccList, cancelSubjectParam, EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_CANCEL, cancelMap);
        // 如果是供应商，再次发送供应商邮件
        if (changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())) {
            sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), cancelSubjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_CANCEL, cancelMap);
        }
        // 如果是其他端发起但是是供应商变更，再次发送供应商邮件
        if (!changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())
                && ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
            sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), cancelSubjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_CANCEL, cancelMap);
        }
        return changeFlowService.updateRecord(changeRecord);
    }

    public BasicChangeFlowVO quote(Long flowId, String supplier, Integer createSource) {
        // 获取工单详情
        ChangeRecord changeRecord = this.getFlowInfo(flowId);
        // 同舟端需校验只能引用同一供应商下的工单
        if (CreateSourceEnum.TONG_ZHOU.getType().equals(createSource)) {
            Assert.isTrue(changeRecord.getCreateSupplier().equals(supplier), "不能引用非本供应商工单");
        }
        String changeSupplier = changeRecord.getChangeSupplier();
        String changeSupplierName = "";
        if (StringUtils.isNotBlank(changeSupplier)) {
            List<SupplierSimpleRsp> supplierSimple = supplierService.getSupplierName(changeSupplier);
            if (CollectionUtils.isNotEmpty(supplierSimple)) {
                SupplierSimpleRsp supplierInfo = supplierSimple.get(0);
                changeSupplierName = supplierInfo.getSupplierName();
            }
        }
        // 获取附件
        List<ChangeFlowFile> changeFileList = changeFileService.getChangeFileList(changeRecord.getId());
        return BasicChangeFlowVO.builder().parentChangeClassId(changeRecord.getParentChangeClassId())
            .sonChangeClassId(changeRecord.getSonChangeClassId()).changeSubject(changeRecord.getChangeSubject())
            .changeItem(changeRecord.getChangeItem()).changeSupplier(changeSupplier)
            .changeSupplierName(changeSupplierName).changeReason(changeRecord.getChangeReason())
            .changeContent(changeRecord.getChangeContent()).changeRiskDesc(changeRecord.getChangeRiskDesc())
            .changeProfit(changeRecord.getChangeProfit()).changeProfitDesc(changeRecord.getChangeProfitDesc())
            .files(changeFileList).build();
    }

    public ChangeFlowVO detail(Long flowId) {
        ChangeRecord changeRecord = changeFlowService.getByFlowId(flowId);
        if (changeRecord == null) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "变更工单id不存在");
        }
        log.debug("[detail] detail:{}", JSON.toJSONString(changeRecord));
        ChangeFlowVO changeFlowVO = new ChangeFlowVO();
        BeanUtils.copyProperties(changeRecord, changeFlowVO);
        // 设置变更类型，一级变更类型>二级变更类型
        String changeType = this.buildChangeType(changeRecord.getParentChangeClassId(), changeRecord.getSonChangeClassId());
        changeFlowVO.setChangeType(changeType);
        // 设置供应商名称
        String changeSupplierName = this.buildChangeSupplierName(changeRecord.getChangeSupplier());
        changeFlowVO.setChangeSupplierName(changeSupplierName);
        // 获取附件
        List<ChangeFlowFile> changeFileList = changeFileService.getChangeFileList(changeRecord.getId());
        changeFlowVO.setFiles(changeFileList);
        // 组装商品信息
        this.buildItemBasicInfoList(changeRecord, changeFlowVO);
        changeFlowVO.setChangeState(changeRecord.getState());
        changeFlowVO.setChangeStateDesc(ChangeStatusEnum.getDescByStatus(changeRecord.getState()));
        changeFlowVO.setChangeCreator(changeRecord.getCreator());
        List<ChangeFlowExecVO> changeFlowExecRecord = changeFlowExecService.getChangeFlowExecRecord(changeRecord.getId());
        // 填充行动人姓名
        this.fillChangeExecUserName(changeFlowExecRecord);
        changeFlowVO.setChangeExecProjectList(changeFlowExecRecord);
        changeFlowVO.setTopoId(ChangeFlowEnum.NEW_CHANGE_FLOW.getTopoId());
        // 设置负责人和创建人视图
        this.buildCommanderAndCreatorView(changeRecord, changeFlowVO);
        return changeFlowVO;
    }

    private void buildItemBasicInfoList(ChangeRecord changeRecord, ChangeFlowVO changeFlowVO) {
        String itemJsonStr = changeRecord.getChangeItem();
        if (StringUtils.isBlank(itemJsonStr)) {
            changeFlowVO.setChangeItems(null);
            changeFlowVO.setItemBasicInfoList(null);
            return;
        }
        
        String changeSkuJson = changeRecord.getChangeSku();
        List<ItemVO> itemList = JSON.parseArray(itemJsonStr, ItemVO.class);
        List<ItemSkuVO> itemSkuVOS = new ArrayList<>();
        if (StringUtils.isNotBlank(changeSkuJson)) {
            itemSkuVOS.addAll(JSON.parseArray(changeSkuJson, ItemSkuVO.class));
        }
        itemList.forEach(i -> {
            Optional<ItemSkuVO> anySkuInfo = itemSkuVOS.stream().filter(s -> s.getItemId().equals(i.getItemId()))
                .findAny();
            if (anySkuInfo.isPresent()) {
                i.setSkuIds(anySkuInfo.get().getSkuId());
            }
        });
        changeFlowVO.setChangeItems(itemList);
        List<Long> itemIds = itemList.stream().map(ItemVO::getItemId).collect(Collectors.toList());
        // 批量查询spu信息
        List<SpuTO> spuTOS = itemService.batchQuerySpuInfo(BatchQuerySpuInfoParam.builder().ids(itemIds)
            .commonProps(new ArrayList<>()).spuProps(Arrays.asList("itemSetupType", "businessForm")).build());
        log.debug("[detail] spuTOS:{}", JSON.toJSONString(spuTOS));
        // 批量查询物理类目
        Map<Long, List<SimplePhyCateGoryResultCo>> categoryChain = itemService
            .queryBySpuIds(CommonIdsParamQuery.builder().ids(itemIds).build());
        // 查询商品对应负责人信息
        Map<Long, ChangeGoodsPrincipalPO> longChangeGoodsPrincipalPOMap = interiorChangeConfigService
            .queryGoodsPrincipalInfo(itemIds);
        List<ItemBasicInfoVO> itemBasicInfoVOS = itemList.stream().map(itemVO -> {
            ItemBasicInfoVO itemBasicInfoVO = new ItemBasicInfoVO();
            Long itemId = itemVO.getItemId();
            itemBasicInfoVO.setItemId(itemId);
            itemBasicInfoVO.setItemName(itemVO.getItemName());
            List<QcCategoryVO> qcCategoryList = qcService.getQcCategoryListByItemId(itemId);
            if (CollectionUtils.isNotEmpty(qcCategoryList)) {
                String collect = qcCategoryList.stream().map(QcCategoryVO::getCategoryName)
                    .collect(Collectors.joining("/"));
                itemBasicInfoVO.setQcCategory(collect);
            } else {
                itemBasicInfoVO.setQcCategory("/");
            }
            List<SimplePhyCateGoryResultCo> simplePhyCateGoryResultCos = categoryChain.get(itemId);
            log.debug("[detail] itemId:{}, simplePhyCateGoryResultCos:{}", itemId,
                JSON.toJSONString(simplePhyCateGoryResultCos));
            if (CollectionUtils.isEmpty(simplePhyCateGoryResultCos)) {
                itemBasicInfoVO.setPhyCategory(null);
            } else {
                List<CategoryInfoVO> categoryInfoVOS = simplePhyCateGoryResultCos.stream().map(phy -> {
                    CategoryInfoVO categoryInfoVO = new CategoryInfoVO();
                    categoryInfoVO.setId(phy.getId());
                    categoryInfoVO.setName(phy.getName());
                    return categoryInfoVO;
                }).collect(Collectors.toList());
                itemBasicInfoVO.setPhyCategory(categoryInfoVOS);
            }
            Optional<SpuTO> optionalSpuTO = spuTOS.stream().filter(spu -> spu.getId() == itemId).findAny();
            log.debug("[detail] itemId:{}, optionalSpuTO:{}", itemId, JSON.toJSONString(optionalSpuTO));
            if (!optionalSpuTO.isPresent()) {
                itemBasicInfoVO.setItemSetupType(null);
                itemBasicInfoVO.setBusinessForm(null);
                itemBasicInfoVO.setStatus(null);
            } else {
                SpuTO spuTO = optionalSpuTO.get();
                Map<String, String> propertyMap = spuTO.getPropertyMap();
                itemBasicInfoVO.setItemSetupType(Integer.valueOf(propertyMap.get("itemSetupType")));
                String businessForm = propertyMap.get("businessForm");
                int business = StringUtils.isBlank(businessForm) ? 0 : Integer.parseInt(businessForm);
                itemBasicInfoVO.setBusinessForm(business);
                itemBasicInfoVO.setStatus(spuTO.getStatus());
            }
            ChangeGoodsPrincipalPO changeGoodsPrincipalPO = longChangeGoodsPrincipalPOMap.get(itemId);
            log.debug("[detail] itemId:{}, changeGoodsPrincipalPO:{}", itemId,
                JSON.toJSONString(changeGoodsPrincipalPO));
            itemBasicInfoVO.setChangeGoodsPrincipal(changeGoodsPrincipalPO);
            Optional<ItemVO> anyHasSku = itemList.stream().filter(i -> i.getItemId().equals(itemId)).findAny();
            if (anyHasSku.isPresent()) {
                itemBasicInfoVO.setSkuIds(anyHasSku.get().getSkuIds());
            }
            return itemBasicInfoVO;
        }).collect(Collectors.toList());
        changeFlowVO.setItemBasicInfoList(itemBasicInfoVOS);
    }

    /**
     * 构建变更类型字符串
     * 
     * @param parentChangeClassId 一级变更类型ID
     * @param sonChangeClassId 二级变更类型ID
     * @return 变更类型字符串，格式：一级类型>二级类型
     */
    private String buildChangeType(Long parentChangeClassId, Long sonChangeClassId) {
        ChangeType parentChangeType = changeTypeService.getChangeTypeById(parentChangeClassId);
        if (parentChangeType == null) {
            throw ExceptionFactory.createBiz(ResponseCode.CHANGE_TYPE_NOT_EXIST, "变更类型不存在");
        }
        ChangeType sonChangeType = changeTypeService.getChangeTypeById(sonChangeClassId);
        if (sonChangeType == null) {
            throw ExceptionFactory.createBiz(ResponseCode.CHANGE_TYPE_NOT_EXIST, "变更类型配置不存在");
        }
        return parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName();
    }

    /**
     * 构建供应商名称
     * 
     * @param changeSupplier 供应商ID
     * @return 供应商名称，如果查询不到则返回空字符串
     */
    private String buildChangeSupplierName(String changeSupplier) {
        if (StringUtils.isBlank(changeSupplier)) {
            return "";
        }
        List<SupplierSimpleRsp> supplierSimple = supplierService.getSupplierName(changeSupplier);
        if (CollectionUtils.isEmpty(supplierSimple)) {
            return "";
        }
        return supplierSimple.get(0).getSupplierName();
    }

    /**
     * 构建负责人和创建人视图
     * 
     * @param changeRecord 变更记录
     * @param changeFlowVO 变更工单视图对象
     */
    private void buildCommanderAndCreatorView(ChangeRecord changeRecord, ChangeFlowVO changeFlowVO) {
        try {
            Set<String> commanderList = new HashSet<>();
            commanderList.add(changeRecord.getChangeCommander());
            commanderList.add(changeRecord.getCreator());
            
            IusDepartmentReq iusDepartmentReq = new IusDepartmentReq();
            iusDepartmentReq.setUids(new ArrayList<>(commanderList));
            iusDepartmentReq.setIcac(true);
            HashMap<String, List<SecondaryDepartments>> orgMap = iusService.queryDepartment(iusDepartmentReq);
            
            List<SecondaryDepartments> commander = orgMap.get(changeRecord.getChangeCommander());
            List<SecondaryDepartments> creator = orgMap.get(changeRecord.getCreator());
            
            // 设置负责人视图：姓名(三级部门名称)
            if (!CollectionUtils.isEmpty(commander)) {
                Optional<SecondaryDepartments> anyOrg = commander.stream()
                    .filter(o -> o.getOrgPosLevel() == 97)
                    .findAny();
                String orgName = anyOrg.isPresent() ? anyOrg.get().getOrgPosName() : "无三级部门";
                changeFlowVO.setChangeCommanderView(commander.get(0).getUserName() + "(" + orgName + ")");
            }
            
            // 设置创建人视图
            if (CreateSourceEnum.TONG_ZHOU.getType().equals(changeRecord.getCreateSource())) {
                // 同舟端发起变更，发起人展示供应商id
                changeFlowVO.setChangeCreatorView(changeRecord.getCreateSupplier());
            } else if (!CollectionUtils.isEmpty(creator)) {
                // 非同舟端展示人名及三级部门名称
                Optional<SecondaryDepartments> anyOrg = creator.stream()
                    .filter(o -> o.getOrgPosLevel() == 97)
                    .findAny();
                String orgName = anyOrg.isPresent() ? anyOrg.get().getOrgPosName() : "无三级部门";
                changeFlowVO.setChangeCreatorView(creator.get(0).getUserName() + "(" + orgName + ")");
            }
        } catch (Exception ex) {
            log.error("add org has ex", ex);
        }
    }

    /**
     * 填充变更行动人姓名
     * 
     * @param changeFlowExecRecord 变更行动方案列表
     */
    private void fillChangeExecUserName(List<ChangeFlowExecVO> changeFlowExecRecord) {
        if (CollectionUtils.isEmpty(changeFlowExecRecord)) {
            return;
        }
        
        try {
            Set<String> userEmail = changeFlowExecRecord.stream()
                .map(ChangeFlowExecVO::getChangeExecUserEmail)
                .collect(Collectors.toSet());
            
            // 批量查询用户信息
            AjaxResponse<List<IusUserInfoRsp>> userListInfo = iusRpcService.queryUserListInfo(
                UserQueryDTO.builder().uids(new ArrayList<>(userEmail)).build());
            List<IusUserInfoRsp> data = userListInfo.getData();
            
            // 填充用户姓名
            changeFlowExecRecord.forEach(i -> {
                Optional<IusUserInfoRsp> anyExeUser = data.stream()
                    .filter(u -> u.getUid().equals(i.getChangeExecUserEmail()))
                    .findAny();
                
                if (anyExeUser.isPresent()) {
                    i.setChangeExecUserName(anyExeUser.get().getName());
                } else {
                    // 批量查询没查到的人名，通过全量查询单独再查一次
                    IusUserInfoRsp user = iusService.queryUserInfo(i.getChangeExecUserEmail());
                    if (user != null && StringUtils.isNotBlank(user.getName())) {
                        i.setChangeExecUserName(user.getName());
                    }
                }
            });
        } catch (Exception ex) {
            log.error("query user info has ex", ex);
        }
    }

    public ChangeFlowListVO query(Integer page, Integer pageSize, ChangeFlowListQueryReq changeFlowListQueryReq) {
        log.info("[query] page:{}, pageSize:{}, changeFlowListQueryReq:{}", page, pageSize,
                JSON.toJSONString(changeFlowListQueryReq));
        
        if (StringUtils.isNotBlank(changeFlowListQueryReq.getChangeExecUser())) {
            List<Long> recordIds = changeFlowExecService.queryByExecUser(changeFlowListQueryReq.getChangeExecUser());
            changeFlowListQueryReq.setChangeRecordIds(recordIds);
            log.info("recordIds: {}", recordIds);
            if (CollectionUtils.isEmpty(recordIds)) {
                ChangeFlowListVO changeFlowListVO = new ChangeFlowListVO();
                PageVO pageVO = PageUtils.buildPageVo(0L, pageSize, page);
                changeFlowListVO.setPageVo(pageVO);
                changeFlowListVO.setChangeFlowList(new ArrayList<>());
                return changeFlowListVO;
            }
        }

        // 数据可见范围：仅查看我跟进的工单（默认true）
        if (changeFlowListQueryReq.getOnlyMyFollowed()) {
            String currentUser = RequestLocalBean.getUid();
            if (StringUtils.isNotBlank(currentUser)) {
                changeFlowListQueryReq.setCurrentUser(currentUser);
                log.info("[query] 数据可见范围：仅查看我跟进的工单, currentUser:{}", currentUser);
            }
        }

        //进行分页
        PageHelper.startPage(page, pageSize);
        PageInfo<ChangeRecord> changeRecordPageInfo = new PageInfo<>(
                changeRecordMapper.selectByCondition(changeFlowListQueryReq));
        List<ChangeRecord> changeRecords = changeRecordPageInfo.getList();
        List<ChangeFlowVO> list = new ArrayList<>();
        // 处理数据
        if (CollectionUtils.isNotEmpty(changeRecords)) {
            // 批量查询所有变更行动工单，构建 Map<changeRecordId, List<subFlowId>>，提升性能
            final Map<Long, List<String>> subFlowIdsMap;
            List<Long> changeRecordIds = changeRecords.stream()
                    .map(ChangeRecord::getId)
                    .collect(Collectors.toList());
            List<ChangeSubFlowRecord> subFlowRecords = changeSubFlowRecordService.getByChangeRecordIds(changeRecordIds);
            if (CollectionUtils.isNotEmpty(subFlowRecords)) {
                subFlowIdsMap = subFlowRecords.stream()
                        .filter(record -> StringUtils.isNotBlank(record.getSubFlowId()))
                        .collect(Collectors.groupingBy(
                                ChangeSubFlowRecord::getChangeRecordId,
                                Collectors.mapping(ChangeSubFlowRecord::getSubFlowId, Collectors.toList())
                        ));
            } else {
                subFlowIdsMap = new HashMap<>();
            }
            list = changeRecords.stream().map(c -> {
                ChangeFlowVO changeFlowVO = new ChangeFlowVO();
                changeFlowVO.setId(c.getId());
                changeFlowVO.setFlowId(c.getFlowId());
                changeFlowVO.setChangeSubject(c.getChangeSubject());
                ChangeType parentChangeType = changeTypeService.getChangeTypeById(c.getParentChangeClassId());
                ChangeType sonChangeType = changeTypeService.getChangeTypeById(c.getSonChangeClassId());
                changeFlowVO.setChangeType(parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
                changeFlowVO.setChangeDepartment(c.getChangeDepartment());
                changeFlowVO.setChangeContent(c.getChangeContent());
                changeFlowVO.setChangeCommander(c.getChangeCommander());
                String itemJsonStr = c.getChangeItem();
                if (StringUtils.isNotBlank(itemJsonStr)) {
                    List<ItemVO> itemVOS = JSON.parseArray(itemJsonStr, ItemVO.class);
                    String changeSkuStr = c.getChangeSku();
                    if (StringUtils.isNotBlank(changeSkuStr)) {
                        List<ItemSkuVO> itemSkuVOS = JSON.parseArray(changeSkuStr, ItemSkuVO.class);
                        itemVOS.forEach(i -> {
                            Optional<ItemSkuVO> anySkuInfo = itemSkuVOS.stream()
                                    .filter(s -> s.getItemId().equals(i.getItemId())).findAny();
                            if (anySkuInfo.isPresent()) {
                                i.setSkuIds(anySkuInfo.get().getSkuId());
                            }
                        });
                    }
                    changeFlowVO.setChangeItems(itemVOS);
                }
                changeFlowVO.setChangeSupplier(c.getChangeSupplier());
                Integer changeState = c.getState() == 3 ? ChangeStatusEnum.IN.getStatus() : c.getState();
                changeFlowVO.setChangeState(changeState);
                changeFlowVO.setChangeStateDesc(ChangeStatusEnum.getDescByStatus(changeState));
                // 同舟端展示供应商id
                changeFlowVO.setChangeCreator(
                    CreateSourceEnum.TONG_ZHOU.getType().equals(c.getCreateSource()) ? c.getCreateSupplier()
                        : c.getCreator());
                String participateChangeExecDepartment = c.getParticipateChangeExecDepartment();
                if (StringUtils.isNotBlank(participateChangeExecDepartment)) {
                    changeFlowVO.setExecDepartmentStrList(JSON.parseArray(participateChangeExecDepartment, String.class));
                    HashSet<String> validDep = new HashSet<>(changeFlowVO.getExecDepartmentStrList());
                    changeFlowVO.setExecDepartmentStrList(new ArrayList<>(validDep));
                }
                changeFlowVO.setCreateTime(c.getCreateTime());
                changeFlowVO.setChangeConfirmResultTime(c.getChangeConfirmResultTime());
                changeFlowVO.setCancelReason(c.getCancelReason());
                changeFlowVO.setRemark(c.getRemark());
                
                // 从 Map 中获取变更行动工单 flowId 列表（已批量查询，提升性能）
                List<String> subFlowIds = subFlowIdsMap.getOrDefault(c.getId(), new ArrayList<>());
                changeFlowVO.setSubFlowIds(subFlowIds);
                
                return changeFlowVO;
            }).collect(Collectors.toList());
        }
        PageVO pageVO = PageUtils.buildPageVo(changeRecordPageInfo.getTotal(), pageSize, page);
        ChangeFlowListVO changeFlowListVO = new ChangeFlowListVO();
        changeFlowListVO.setPageVo(pageVO);
        try {
            Set<String> commanderList = list.stream().map(ChangeFlowVO::getChangeCommander).collect(Collectors.toSet());
            Set<String> creatorList = list.stream().map(ChangeFlowVO::getChangeCreator).collect(Collectors.toSet());
            commanderList.addAll(creatorList);
            IusDepartmentReq iusDepartmentReq = new IusDepartmentReq();
            iusDepartmentReq.setUids(new ArrayList<>(commanderList));
            iusDepartmentReq.setIcac(true);
            HashMap<String, List<SecondaryDepartments>> orgMap = iusService.queryDepartment(iusDepartmentReq);
            //AjaxResponse<List<IusUserInfoRsp>> userListInfo = iusRpcService.queryUserListInfo(UserQueryDTO.builder().uids(new ArrayList<>(commanderList)).build());
            log.debug("query user: {} info: {}", commanderList, JSON.toJSONString(orgMap));
            //List<IusUserInfoRsp> data = userListInfo.getData();
            list.forEach(i -> {
                List<SecondaryDepartments> commander = orgMap.get(i.getChangeCommander());
                List<SecondaryDepartments> creator = orgMap.get(i.getChangeCreator());
                //Optional<IusUserInfoRsp> commander = data.stream().filter(u -> u.getUid().equals(i.getChangeCommander())).findAny();
                //Optional<IusUserInfoRsp> creator = data.stream().filter(u -> u.getUid().equals(i.getChangeCreator())).findAny();
                if (!CollectionUtils.isEmpty(commander)) {
                    Optional<SecondaryDepartments> anyOrg = commander.stream().filter(o -> o.getOrgPosLevel() == 97).findAny();
                    String orgName = "无三级部门";
                    if (anyOrg.isPresent()) {
                        orgName = anyOrg.get().getOrgPosName();
                    }
                    i.setChangeCommander(commander.get(0).getUserName() + "(" + orgName + ")");
                }
                // 供应商查询不到数据，不会覆盖，展示供应商id
                if (!CollectionUtils.isEmpty(creator)) {
                    Optional<SecondaryDepartments> anyOrg = creator.stream().filter(o -> o.getOrgPosLevel() == 97).findAny();
                    String orgName = "无三级部门";
                    if (anyOrg.isPresent()) {
                        orgName = anyOrg.get().getOrgPosName();
                    }
                    i.setChangeCreator(creator.get(0).getUserName() + "(" + orgName + ")");
                }
            });
        } catch (Exception ex) {
            log.error("add org has ex", ex);
        }
        changeFlowListVO.setChangeFlowList(list);
        return changeFlowListVO;
    }


    public UserBaseContainerDTO getOperator(InterfaceInputDTO interfaceInput) {
        log.info("[getOperator] interfaceInput:{}", JSON.toJSONString(interfaceInput));
        // 根据工单flowId获取审批人，创建工单时设置，唯一
        String flowId = interfaceInput.getPublicFieldDTO().getFlowMeta().getFlowId();
        String nodeId = interfaceInput.getPublicFieldDTO().getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
        log.info("[getOperator] flowId:{}, nodeId:{}", flowId, nodeId);
        ChangeRecord changeRecord;
        changeRecord = changeFlowService.getByFlowId(Long.valueOf(flowId));
        log.info("[getOperator] first time get changeCommander:{}", changeRecord);
        // 创建工单后落库，此时审批人还未落库，工单平台无法查询到审批人，方法休眠500ms
        if (changeRecord == null) {
            try {
                Thread.sleep(500);
                changeRecord = changeFlowService.getByFlowId(Long.valueOf(flowId));
                log.info("[getOperator] second time get changeCommander:{}", changeRecord);
            } catch (Exception e) {
                log.info("[getOperator] flowId:{}, nodeId:{}, e={}", flowId, nodeId, e);
            }
        }
        String changeCommander = changeRecord.getChangeCommander();
        UserReachDTO userReachDTO = new UserReachDTO();
        // 查询邮箱对应的名字
        IusUserInfoRsp user = iusService.queryUserInfo(changeRecord.getChangeCommander());
        // 流程拓扑图，如果没查询到名字，以邮箱兜底
        userReachDTO.setUserName(
            user == null || StringUtils.isBlank(user.getName()) ? changeRecord.getChangeCommander() : user.getName());
        userReachDTO.setUid(changeCommander);
        List<UserReachDTO> totalUserList = Collections.singletonList(userReachDTO);
        UserBaseContainerDTO userBaseContainer = new UserBaseContainerDTO();
        userBaseContainer.setUserList(totalUserList);
        return userBaseContainer;
    }

    @Deprecated
    public void export(ChangeFlowListQueryReq changeFlowListQueryReq, HttpServletResponse response) {
        List<ChangeRecord> changeRecords = changeRecordMapper.selectByCondition(changeFlowListQueryReq);
        if (CollectionUtils.isEmpty(changeRecords)) {
            throw ExceptionFactory.createBiz(ResponseCode.EMPTY, "无导出数据");
        }
        List<ChangeFlowExcelDTO> changeFlowExcelDTOList = changeRecords.stream().map(record -> {
            ChangeFlowExcelDTO changeFlowExcelDTO = new ChangeFlowExcelDTO();
            /*changeFlowExcelDTO.setId(record.getId());
            changeFlowExcelDTO.setChangeSubject(ChangeSubjectEnum.getChangeSubjectEnum(record.getChangeSubject()).getDesc());
            ChangeType parentChangeType = changeTypeService.getChangeTypeById(record.getParentChangeClassId());
            ChangeType sonChangeType = changeTypeService.getChangeTypeById(record.getSonChangeClassId());
            changeFlowExcelDTO.setChangeType(parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
            changeFlowExcelDTO.setChangeDepartment(record.getChangeDepartment());
            changeFlowExcelDTO.setChangeContent(record.getChangeContent());
            changeFlowExcelDTO.setChangeCommander(record.getChangeCommander());
            String changeItemJsonStr = record.getChangeItem();
            if (StringUtils.isNotBlank(changeItemJsonStr)) {
                List<ItemVO> itemVOS = JSON.parseArray(changeItemJsonStr, ItemVO.class);
                changeFlowExcelDTO.setChangeItems(itemVOS.stream().map(item -> item.getItemId() + item.getItemName())
                        .collect(Collectors.joining(",")));
            }
            if (StringUtils.isNotBlank(record.getChangeSupplier())) {
                changeFlowExcelDTO.setSupplier(record.getChangeSupplier());
            }
            changeFlowExcelDTO.setChangeState(record.getState());
            // todo:变更发起人组织查询：姓名（组织架构三级部门）/供应商ID/system+发起时间
            changeFlowExcelDTO.setChangeCreator(record.getCreator());
            changeFlowExcelDTO.setExecDepartmentStrList(record.getParticipateChangeExecDepartment());
            changeFlowExcelDTO.setCreateTime(record.getCreateTime());*/
            return changeFlowExcelDTO;
        }).collect(Collectors.toList());
        Workbook workbook = ExcelExportUtil.exportExcel(new ExportParams(), ChangeFlowExcelDTO.class, changeFlowExcelDTOList);
        downLoadExcel("变更管理列表", response, workbook);
    }

    private void downLoadExcel(String fileName, HttpServletResponse response, Workbook workbook) {
        try {
            response.setCharacterEncoding("UTF-8");
            response.setHeader("content-Type", "application/vnd.ms-excel");
            response.setHeader("Content-Disposition",
                    "attachment;filename=" + URLEncoder.encode(fileName, "UTF-8") + ".xls");
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            workbook.write(baos);
            response.setHeader("Content-Length", String.valueOf(baos.size()));
            OutputStream out = response.getOutputStream();
            out.write(baos.toByteArray());

        } catch (IOException ex) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST);
        }
    }

    public void deliver(@Valid ChangeFlowDeliverReq req) {
        log.info("[deliver] req:{}", JSON.toJSONString(req));
        Long flowId = req.getFlowId();
        // 查询工单有效性
        ChangeRecord changeRecord = getFlowInfo(flowId);
        log.info("[deliver] nodeId:{}", changeRecord.getFlowNode());
        // 检查工单节点
        List<String> nodeList = Arrays.asList(ChangeFlowEnum.CHANGE_FLOW_SUBMIT.getNodeId(),
                ChangeFlowEnum.CHANGE_FLOW_CONFIRM.getNodeId());
        this.checkNode(changeRecord.getFlowNode(), nodeList);
        String uid = RequestLocalBean.getUid();
        String name = RequestLocalBean.getName();
        String changeCommander = changeRecord.getChangeCommander();
        if (!uid.equals(changeCommander)) {
            throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, ResponseCode.NO_AUTH.getMsg());
        }
        if (changeCommander.equals(req.getDeliverUser())) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "不能转交给自己！");
        }
        // 工单审核人转交
        String deliverUser = req.getDeliverUser();
        UserBaseDTO userBaseDTO = new UserBaseDTO();
        userBaseDTO.setUserName(deliverUser);
        userBaseDTO.setUid(deliverUser);
        String remark = req.getRemark();
        Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
        content.put("updateTime", System.currentTimeMillis());
        flowService.updateApprovers(ChangeFlowEnum.CHANGE_FLOW_EXE.getTopoId(), flowId.toString(),
                changeRecord.getFlowNode(), Collections.singletonList(userBaseDTO), uid,
                name, remark, JSON.toJSONString(content));
        // 更新工单负责人
        changeRecord.setChangeCommander(deliverUser);
        changeRecord.setRemark(remark);
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());

        Map<String, Object> deliverMap = new HashMap<>();
        deliverMap.put("changeId", changeRecord.getFlowId());
        deliverMap.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
        deliverMap.put("changeContent", changeRecord.getChangeContent());
        // 转交人名称
        IusUserInfoRsp oldUser = iusService.queryUserInfo(changeCommander);
        deliverMap.put("changeCommander", oldUser == null ? changeCommander : oldUser.getName());
        deliverMap.put("changeCommanderEmail", changeCommander);
        // 被转交人名称
        IusUserInfoRsp newUser = iusService.queryUserInfo(deliverUser);
        deliverMap.put("restsChangeCommander", newUser == null ? deliverUser : newUser.getName());
        deliverMap.put("restsChangeCommanderEmail", deliverUser);
        String deliverSubjectParam = changeRecord.getFlowId().toString();
        // 转交，收件人：变更转交人（工单接收人）、变更负责人（工单转交人）
        List<String> receiver = Arrays.asList(changeCommander, deliverUser);
        // 转交，抄送人：变更发起人+原变更负责人上一级主管、变更转交人（工单接收人）上一级主管+变更行动项负责人
        List<String> ccList = departmentLeaderBiz.getDepartmentLeaders(receiver);
        ccList.add(changeRecord.getCreator());
        List<ChangeFlowExecVO> execRecord = changeFlowExecService
                .getChangeFlowExecRecord(changeRecord.getId());
        List<String> userEmailList = execRecord.stream().map(ChangeFlowExecVO::getChangeExecUserEmail)
                .collect(Collectors.toList());
        ccList.addAll(userEmailList);
        qcSendEmail(receiver, ccList, deliverSubjectParam, EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_FORWARD, deliverMap);
        // 如果是供应商，再次发送供应商邮件
        if (changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())) {
            sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), deliverSubjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_FORWARD, deliverMap);
        }
        // 如果是其他端发起但是是供应商变更，再次发送供应商邮件
        if (!changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())
                && ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
            sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), deliverSubjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_FORWARD, deliverMap);
        }
        changeFlowService.updateRecord(changeRecord);
    }

    /**
     * 获取工单负责人
     *
     * @param changeFlowCreateReq 创建请求
     * @param uid 当前用户ID
     * @param parentChangeClassId 一级变更类型ID
     * @param sonChangeClassId 二级变更类型ID
     * @param itemIds 商品ID列表
     * @return 工单负责人邮箱
     */
    private String getChangeCommander(ChangeFlowCreateReq changeFlowCreateReq, String uid,
                                     Long parentChangeClassId, Long sonChangeClassId, List<Long> itemIds) {
        // 非同舟端：负责人=发起人
        if (!CreateSourceEnum.TONG_ZHOU.getType().equals(changeFlowCreateReq.getCreateSource())) {
            return uid;
        }
        
        // 同舟端：通过RPC查询负责人信息
        GoodsResponseRpc goodsResponseRpc = null;
        try {
            goodsResponseRpc = interiorChangeConfigService
                    .queryCommanderInfo(ChangeCommanderPO.builder()
                            .parentChangeClassId(parentChangeClassId)
                            .sonChangeClassId(sonChangeClassId)
                            .changeSupplierId(changeFlowCreateReq.getChangeSupplier())
                            .goodsInfos(itemIds)
                            .flowCreator(uid)
                            .createSource(changeFlowCreateReq.getCreateSource())
                            .supplier(changeFlowCreateReq.getSupplier())
                            .build(),
                            changeFlowCreateReq.getChangeCommander());
            return goodsResponseRpc != null ? goodsResponseRpc.getEmail() : null;
        } catch (Exception e) {
            log.error("[getChangeCommander] queryCommanderInfo error, goodsResponseRpc:{}, e:{}",
                    JSON.toJSONString(goodsResponseRpc), e);
            return null;
        }
    }

    /**
     * 构建变更类型
     *
     * @param changeFlowCreateReq 创建请求
     * @return 变更类型
     */
    private StringBuilder buildChangeType(ChangeFlowCreateReq changeFlowCreateReq) {
        Long parentChangeClassId = changeFlowCreateReq.getParentChangeClassId();
        Long sonChangeClassId = changeFlowCreateReq.getSonChangeClassId();
        
        // 变更类型， 一级变更类型>二级变更类型
        ChangeConfigPo changeConfigPo = changeConfigService.queryInfoPo(parentChangeClassId, sonChangeClassId);
        StringBuilder changeType = new StringBuilder();
        if (changeConfigPo != null && changeConfigPo.getChangeTypes() != null && changeConfigPo.getChangeTypes().size() >= 2) {
            try {
                changeType.append(changeConfigPo.getChangeTypes().get(0).getTypeName());
                changeType.append(changeConfigPo.getChangeTypes().get(1).getTypeName());
            } catch (Exception e) {
                log.error("queryInfoPo error:{}", JSON.toJSONString(changeConfigPo), e);
            }
        }
        return changeType;
    }

    /**
     * 根据变更主体类型构建工单名称后缀（辅助方法，由 buildFlowName 调用）
     *
     * @param changeFlowCreateReq 创建请求
     * @return 工单名称后缀
     */
    private String buildFlowNameBySubject(ChangeFlowCreateReq changeFlowCreateReq) {
        ChangeSubjectEnum subjectEnum = ChangeSubjectEnum.getByType(changeFlowCreateReq.getChangeSubject());
        switch (Objects.requireNonNull(subjectEnum)) {
            case PRODUCT:
                // 商品变更：解析商品ID列表，查询SPU信息，拼接商品ID和名称
                List<Long> itemIds = parseItemIds(changeFlowCreateReq.getChangeItems());
                List<SpuTO> spuTOS = itemService.batchQuerySpuInfo(BatchQuerySpuInfoParam.builder().ids(itemIds)
                    .commonProps(new ArrayList<>()).spuProps(new ArrayList<>()).build());
                log.debug("[buildFlowNameBySubject] spuTOS:{}", JSON.toJSONString(spuTOS));
                return spuTOS.stream().map(spu -> "(" + spu.getId() + spu.getName() + ")")
                    .collect(Collectors.joining(","));

            case SUPPLIER:
                // 供应商变更：查询供应商信息，拼接供应商ID和名称
                List<SupplierSimpleRsp> supplierSimple = supplierService.getSupplierName(changeFlowCreateReq.getChangeSupplier());
                if (CollectionUtils.isNotEmpty(supplierSimple)) {
                    SupplierSimpleRsp supplier = supplierSimple.get(0);
                    return supplier.getSupplierId() + supplier.getSupplierName();
                }
                return "";

            case OTHER:
            default:
                return "";
        }
    }

    /**
     * 查询topo实际节点信息
     * @param flowId 工单ID
     * @return topo实际节点信息
     */
    public FlowDataDTO getFlowNodeInfo(Long flowId) {
        return flowService.flowDetail(flowId.toString());
    }

    /**
     * 校验变更行动方案中是否有重复的变更行动人
     * 
     * 规则：
     * 1. 供应商关联角色/商品关联角色：如果 changeExecUserType 和 changeExecUser 都相同，则重复
     * 2. 固定人：如果 changeExecUserType 相同且 changeExecUser（邮箱）相同，则重复
     * 3. 发起人：如果 changeExecUserType 相同，则重复
     * 
     * @param changeExecProjectList 变更行动方案列表
     */
    private void validateDuplicateChangeExecUser(List<ChangeExecConfigReq> changeExecProjectList) {
        if (CollectionUtils.isEmpty(changeExecProjectList)) {
            return;
        }
        
        // 用于记录邮箱对应的部门：邮箱 -> 部门
        Map<String, String> userEmailToDepartment = new HashMap<>();
        
        for (ChangeExecConfigReq execConfig : changeExecProjectList) {
            // 校验部门+人员，需要邮箱和部门都不为空
            if (StringUtils.isBlank(execConfig.getChangeExecUserEmail()) 
                    || StringUtils.isBlank(execConfig.getChangeExecDepartment())) {
                continue;
            }
            
            String changeExecUserEmail = execConfig.getChangeExecUserEmail();
            String changeExecDepartment = execConfig.getChangeExecDepartment();
            
            // 检查该邮箱是否已经在其他部门出现过
            String existingDepartment = userEmailToDepartment.get(changeExecUserEmail);
            if (existingDepartment != null && !existingDepartment.equals(changeExecDepartment)) {
                String errorMsg = String.format("变更行动方案中，同一人员不能出现在多个部门：%s 已在部门 %s 中，不能同时出现在部门 %s", 
                    changeExecUserEmail, existingDepartment, changeExecDepartment);
                log.warn("[validateDuplicateChangeExecUser] 检测到同一人员出现在多个部门: email={}, existingDept={}, newDept={}", 
                    changeExecUserEmail, existingDepartment, changeExecDepartment);
                throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, errorMsg);
            }
            
            // 记录邮箱和部门的对应关系（如果已存在相同部门，覆盖不影响）
            userEmailToDepartment.put(changeExecUserEmail, changeExecDepartment);
            log.debug("[validateDuplicateChangeExecUser] 添加变更行动人: email={}, department={}", 
                changeExecUserEmail, changeExecDepartment);
        }
    }

    /**
     * 所有子单审批通过后，流转主工单到部门负责人审批节点
     * 从 NEW_CHANGE_FLOW_CONFIRM_EXEC_PLAN 流转到 NEW_CHANGE_FLOW_OWNER_APPROVE
     * 
     * @param changeRecordId 主工单ID
     */
    void submitMainFlowAfterAllSubFlowsApproved(Long changeRecordId) {
        // 查询主工单
        ChangeRecord changeRecord = changeRecordMapper.selectByPrimaryKey(changeRecordId);
        if (changeRecord == null) {
            log.error("[submitMainFlowAfterAllSubFlowsApproved] 主工单不存在，changeRecordId:{}", changeRecordId);
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "主工单不存在");
        }
        
        // 验证主工单当前节点
        String currentNode = changeRecord.getFlowNode();
        if (!ChangeFlowEnum.NEW_CHANGE_FLOW_CONFIRM_EXEC_PLAN.getNodeId().equals(currentNode)) {
            throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, 
                    String.format("主工单当前节点不是确认变更方案节点，无法流转。当前节点：%s", currentNode));
        }
        
        try {
            Long flowId = changeRecord.getFlowId();
            String uid = RequestLocalBean.getUid();
            
            // 获取工单详情
            FlowDataDTO flowDataDTO = flowService.flowDetail(flowId.toString());
            if (flowDataDTO == null) {
                throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "工单查询错误，不存在");
            }
            
            // 构建提交内容
            Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
            content.put("updateTime", System.currentTimeMillis());
            content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.PASS.getValue());
            
            // 流转主工单到部门负责人审批节点
            String ownerApproveNodeId = flowService.submitFlow(flowId.toString(), flowDataDTO, uid,
                    ChangeFlowEnum.NEW_CHANGE_FLOW.getTopoId(), JSON.toJSONString(content), true,
                    FlowxOperationEnum.SUBMIT.getName(), "所有子单审批通过，主工单流转", changeRecord.getCreateTime());
            
            changeRecord.setFlowNode(ownerApproveNodeId);
            changeRecord.setState(ChangeStatusEnum.WAIT_DEPT_LEADER_APPROVE.getStatus());
            changeRecord.setUpdateTime(DateUtils.getCurrentTime());
            changeFlowService.updateRecord(changeRecord);
            
            log.info("[submitMainFlowAfterAllSubFlowsApproved] 主工单流转成功，flowId:{}, nextNodeId:{}", 
                    flowId, ownerApproveNodeId);
        } catch (Exception e) {
            log.error("[submitMainFlowAfterAllSubFlowsApproved] 主工单流转失败，changeRecordId:{}", changeRecordId, e);
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, 
                    String.format("主工单流转失败：%s", e.getMessage()));
        }
    }

    /**
     * 检查主工单下所有子单是否都已完成（供主工单流转时调用）
     * 条件：status = FINISHED 或 CANCELLED
     * 
     * @param changeRecordId 主工单ID
     * @return true 表示所有子单都已完成
     */
    public boolean checkAllSubFlowsFinishedForMainFlow(Long changeRecordId) {
        // 查询该主工单下所有子单
        List<ChangeSubFlowRecord> allSubFlows = changeSubFlowRecordMapper.selectByChangeRecordId(changeRecordId);
        if (CollectionUtils.isEmpty(allSubFlows)) {
            return false;
        }
        
        // 检查是否所有子单都满足条件：状态是 FINISHED 或 CANCELLED
        return allSubFlows.stream().allMatch(subFlow -> 
            ChangeSubFlowStatusEnum.FINISHED.getStatus().equals(subFlow.getStatus())
            || ChangeSubFlowStatusEnum.CANCELLED.getStatus().equals(subFlow.getStatus())
        );
    }

}
