package com.netease.mail.yanxuan.change.biz.callback;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import com.netease.mail.dp.des.process.api.entity.CallbackQueryReq;
import com.netease.mail.dp.des.process.api.entity.ExcelCell;
import com.netease.mail.dp.des.process.api.entity.ExcelTitle;
import com.netease.mail.dp.des.starter.callback.DesCallbackService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeExecConfigService;
import com.netease.mail.yanxuan.change.common.enums.ChangeCommanderEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeLevelEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeSubjectEnum;
import com.netease.mail.yanxuan.change.common.enums.NeedFileEnum;
import com.netease.mail.yanxuan.change.common.util.DateUtils;
import com.netease.mail.yanxuan.change.common.util.ExcelUtil;
import com.netease.mail.yanxuan.change.dal.entity.ChangeExecConfig;
import com.netease.mail.yanxuan.change.dal.entity.ChangeType;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeConfigMapper;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeTypeMapper;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeConfigPo;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeConfigQueryReq;
import com.netease.mail.yanxuan.change.integration.excel.ChangeConfigExcelDTO;
import com.netease.mail.yanxuan.change.integration.excel.ChangeConfigExcelVo;

import lombok.extern.slf4j.Slf4j;

/**
 * 变更配置导出回调。一层主表 + 二层行动项：无行动项时输出一行主数据，有多条行动项时主数据复制多行，总行数按展开后的行数。
 */
@Slf4j
@Component
public class ChangeConfigExportCallback implements DesCallbackService {

    @Autowired
    private ChangeConfigMapper changeConfigMapper;

    @Autowired
    private ChangeTypeMapper changeTypeMapper;

    @Autowired
    private ChangeExecConfigService changeExecConfigService;

    @Override
    public ExcelTitle queryExcelTitle(CallbackQueryReq req) {
        log.info("[op: change config export callback queryExcelTitle] req={}", JSON.toJSONString(req));
        ChangeConfigQueryReq query = JSON.parseObject(req.getParamJson(), ChangeConfigQueryReq.class);
        
        List<ChangeConfigPo> list = changeConfigMapper.selectChangeConfigPo(query);
        ExcelTitle excelTitle = ExcelUtil.fetchExceltile(ChangeConfigExcelVo.class);
        excelTitle.setTotal((long) list.size());
        log.info("[op: change config export callback queryExcelTitle] result={}", JSON.toJSONString(excelTitle));
        return excelTitle;
    }

    @Override
    public List<List<ExcelCell>> queryExcelData(CallbackQueryReq req) {
        log.info("[op: change config export callback queryExcelData] req={}", JSON.toJSONString(req));
        ChangeConfigQueryReq query = JSON.parseObject(req.getParamJson(), ChangeConfigQueryReq.class);
        query.setOffset(req.getOffset());
        query.setLimit(req.getLimit());
        List<ChangeConfigExcelDTO> list = getExportList(query);
        log.debug("[op: change config export callback queryExcelData] size:{}", list.size());
        return list.stream().map(ChangeConfigExcelVo::init).collect(Collectors.toList());
    }

    @Override
    public com.netease.mail.dp.des.process.api.entity.PivotTable queryPivotTable(CallbackQueryReq req) {
        // 不需要透视表，返回null
        return null;
    }

    private List<ChangeConfigExcelDTO> getExportList(ChangeConfigQueryReq query) {
        List<ChangeConfigPo> changeConfigPos = changeConfigMapper.selectChangeConfigPoWithLimit(query);
        if (changeConfigPos.isEmpty()) {
            return new ArrayList<>();
        }
        // 处理执行部门格式
        changeConfigPos.forEach(i -> {
            if (StringUtils.isNotBlank(i.getParticipateChangeExecDepartment())) {
                String replaceDep = i.getParticipateChangeExecDepartment().replace("[", "").replace("]", "").replace(" ", "");
                List<String> depList = Arrays.stream(replaceDep.split(","))
                        .filter(StringUtils::isNotBlank)
                        .distinct()
                        .collect(Collectors.toList());
                i.setParticipateChangeExecDepartment(Joiner.on(",").join(depList));
            }
        });
        
        // 批量查询变更类型：收集所有的 parentChangeClassId 和 sonChangeClassId
        Set<Long> changeTypeIds = new HashSet<>();
        for (ChangeConfigPo po : changeConfigPos) {
            if (po.getParentChangeClassId() != null) {
                changeTypeIds.add(po.getParentChangeClassId().longValue());
            }
            if (po.getSonChangeClassId() != null) {
                changeTypeIds.add(po.getSonChangeClassId().longValue());
            }
        }
        
        Map<Long, ChangeType> changeTypeMap = new HashMap<>();
        if (!changeTypeIds.isEmpty()) {
            List<ChangeType> changeTypes = changeTypeMapper.selectByIds(new ArrayList<>(changeTypeIds));
            if (changeTypes != null) {
                for (ChangeType ct : changeTypes) {
                    changeTypeMap.put(ct.getId(), ct);
                }
            }
        }
        final Map<Long, ChangeType> finalChangeTypeMap = changeTypeMap;

        List<ChangeConfigExcelDTO> expanded = new ArrayList<>();
        for (ChangeConfigPo po : changeConfigPos) {
            List<ChangeExecConfig> execs = changeExecConfigService.queryChangeExecConfig(po.getId());
            if (CollectionUtils.isEmpty(execs)) {
                expanded.add(buildRow(po, null, finalChangeTypeMap));
            } else {
                for (ChangeExecConfig exec : execs) {
                    expanded.add(buildRow(po, exec, finalChangeTypeMap));
                }
            }
        }
        return expanded;
    }

    private ChangeConfigExcelDTO buildRow(ChangeConfigPo po, ChangeExecConfig exec, Map<Long, ChangeType> changeTypeMap) {
        ChangeConfigExcelDTO dto = new ChangeConfigExcelDTO();
        // 一层：主表字段
        dto.setId(po.getId());
        if (po.getChangeSubject() != null) {
            dto.setChangeSubject(ChangeSubjectEnum.getChangeSubjectEnum(po.getChangeSubject()) != null
                    ? ChangeSubjectEnum.getChangeSubjectEnum(po.getChangeSubject()).getDesc() : "");
        }
        String changeType = "";
        if (po.getParentChangeClassId() != null && po.getSonChangeClassId() != null) {
            ChangeType parentType = changeTypeMap.get(po.getParentChangeClassId().longValue());
            ChangeType sonType = changeTypeMap.get(po.getSonChangeClassId().longValue());
            if (parentType != null && sonType != null) {
                changeType = parentType.getTypeName() + ">" + sonType.getTypeName();
            } else if (parentType != null) {
                changeType = parentType.getTypeName();
            } else if (sonType != null) {
                changeType = sonType.getTypeName();
            }
        }
        dto.setChangeType(changeType);
        if (po.getChangeLevel() != null) {
            dto.setChangeLevel(ChangeLevelEnum.getByType(po.getChangeLevel()));
        }
        dto.setChangeDepartment(po.getChangeDepartment());
        dto.setParticipateChangeExecDepartment(po.getParticipateChangeExecDepartment());
        if (po.getChangeCommanderType() != null) {
            ChangeCommanderEnum commanderEnum = ChangeCommanderEnum.getChangeCommanderEnum(po.getChangeCommanderType());
            dto.setChangeCommanderType(commanderEnum != null ? commanderEnum.getName() : "");
        }
        dto.setChangeCommander(po.getChangeCommander());
        if (po.getNeedFile() != null) {
            for (NeedFileEnum e : NeedFileEnum.values()) {
                if (e.getStatus().equals(po.getNeedFile())) {
                    dto.setNeedFile(e.getDesc());
                    break;
                }
            }
        }
        dto.setCreator(po.getCreator());
        if (po.getCreateTime() != null) {
            dto.setCreateTime(DateUtils.parseLongToString(po.getCreateTime(), DateUtils.DATE_TIME_FORMAT));
        }
        if (po.getUpdateTime() != null) {
            dto.setUpdateTime(DateUtils.parseLongToString(po.getUpdateTime(), DateUtils.DATE_TIME_FORMAT));
        }
        // 二层：行动项
        if (exec != null) {
            dto.setChangeExecDepartment(exec.getChangeExecDepartment());
            dto.setChangeExecUser(exec.getChangeExecUser());
            dto.setChangeRiskDesc(exec.getChangeRiskDesc());
            dto.setChangeExecProject(exec.getChangeExecProject());
            dto.setChangeChecking(exec.getChangeChecking());
        }
        return dto;
    }
}

