package com.netease.mail.yanxuan.change.integration.email.conig;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.parser.ParserConfig;
import com.netease.mail.yanxuan.change.integration.email.exception.RpcException;
import com.netease.mail.yanxuan.change.integration.email.exception.RpcStatusException;
import com.netease.mail.yanxuan.change.integration.email.exception.RpcTimeoutException;
import com.netease.mail.yanxuan.change.integration.email.util.EncodeUtil;
import com.netease.mail.yanxuan.change.integration.email.util.HttpClientUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import java.io.IOException;
import java.util.Map;

/**
 * rpc 请求封装,在 HttpClientUtil 基础上加入了返回值处理和日志
 *
 * @author lwtang
 * @date 2019-02-19
 */
@Component
public class RpcTemplate {


    private Logger logger = LoggerFactory.getLogger(RpcTemplate.class);

    @Value("${act.rpc.get.retry:1}")
    private int rpcGetRetry;

    @Value("${act.rpc.post.retry:0}")
    private int rpcPostRetry;

    public RpcTemplate() {
        ParserConfig.getGlobalInstance().setAutoTypeSupport(true);
    }

    private String convertParam(Object param) {
        if (param instanceof String) {
            return (String) param;
        }
        String reqParam = "";
        if (param instanceof Map) {
            reqParam = EncodeUtil.encodeMap((Map) param);
        } else if (param != null) {
            reqParam = JSON.toJSONString(param);
        }
        return reqParam;
    }

    private <T> T execute(String url, Object params, CallHandler<T> handler,
                          CallTemplate callTemplate, int retryTime, Method method) {
        int tryTimes = 0;
        RpcException rpcException = null;
        String respBody = null;
        int statusCode = 200;
        while (tryTimes <= retryTime) {
            long startTime = System.currentTimeMillis();
            try {
                respBody = callTemplate.executeCall();
                if (StringUtils.isEmpty(respBody)) {
                    throw new RpcException("response empty");
                }

                T t = handler.handle(respBody);
                return t;
            } catch (RpcTimeoutException e) {
                rpcException = e;
            } catch (RpcStatusException e) {
                rpcException = e;
                statusCode = e.getCode();
            } catch (RpcException e) {
                rpcException = e;
            } catch (Exception e) {
                rpcException = new RpcException(e);
            } finally {
                if (logger.isDebugEnabled()) {
                    long costTime = System.currentTimeMillis() - startTime;
                    String reqParam = convertParam(params);
                    logger.debug(
                            "process request, url={}, method={}, params={}, statusCode = {}, response={}, cost={}ms",
                            url, method.name(), reqParam, statusCode, respBody, costTime);
                }
            }
            tryTimes++;
        }

        logger.error("request error, url={}, params={}, statusCode = {}, respBody={}", url,
                convertParam(params), statusCode, respBody, rpcException);

        if (rpcException == null) {
            rpcException = new RpcException("服务异常");
        }
        throw rpcException;
    }

    /**
     * Get 请求
     *
     * @param url
     * @param params
     * @param header
     * @param timeout
     *            -1 取默认的超时时间
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T get(String url, Map<String, Object> params, Map<String, String> header,
        int timeout, CallHandler<T> handler, Integer retryTimes) {
        CallTemplate template = () -> HttpClientUtil.get(url, params, header, timeout);
        return this.execute(url, params, handler, template, retryTimes, Method.GET);
    }

    public <T> T get(String url, Map<String, Object> params, CallHandler<T> handler) {
        return get(url, params, null, -1, handler, rpcGetRetry);
    }

    /**
     * Post请求(url encoded)
     *
     * @param url
     * @param params
     * @param timeout
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T post(String url, Map<String, Object> params, int timeout, CallHandler<T> handler,
        Integer retryTime) {
        CallTemplate template = () -> HttpClientUtil.post(url, params, timeout);
        return this.execute(url, params, handler, template, retryTime, Method.POST);
    }

    /**
     * Post请求(url encoded)
     *
     * @param url
     * @param params
     * @param timeout
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T post(String url, Map<String, Object> params, int timeout, CallHandler<T> handler) {
        CallTemplate template = () -> HttpClientUtil.post(url, params, timeout);
        return this.execute(url, params, handler, template, rpcPostRetry, Method.POST);
    }

    /**
     * Post Json请求
     *
     * @param url
     * @param params
     * @param timeout
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T postJson(String url, String params, int timeout, CallHandler<T> handler,
        Integer retryTime) {
        CallTemplate template = () -> HttpClientUtil.postJson(url, params, timeout);
        return this.execute(url, params, handler, template, retryTime, Method.POST);
    }

    /**
     * Post Json请求
     *
     * @param url
     * @param params
     * @param timeout
     * @param handler
     * @param <T>
     * @return
     */
    public <T> T postJson(String url, String params, int timeout, CallHandler<T> handler) {
        CallTemplate template = () -> HttpClientUtil.postJson(url, params, timeout);
        return this.execute(url, params, handler, template, rpcPostRetry, Method.POST);
    }

    enum RpcStatus {

        /**
         * 成功
         */
        SUCCESS(1),

        /**
         * 告警（请求成功，但非理想值）
         */
        WARNING(2),

        /**
         * 异常
         */
        EXCEPTION(3),

        /**
         * 超时
         */
        TIMEOUT(4);

        private int value;

        RpcStatus(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }

    enum Method {

        /**
         * Get请求
         */
        GET,

        /**
         * Post表单请求
         */
        POST,
    }

    /**
     * 模板调用
     */
    @FunctionalInterface
    public interface CallTemplate {

        /**
         * 模板调用类
         */
        String executeCall();
    }

    /**
     * 报文解析
     */
    @FunctionalInterface
    public interface CallHandler<T> {

        /**
         * 报文解析
         *
         * @param resp
         *            response body
         */
        T handle(String resp) throws IOException;

        /**
         * 校验数据体
         *
         * @param t
         * @return
         */
        default boolean check(T t) {
            return true;
        }
    }

}
