/**
 * @(#)ChangeFlowBiz.java, 2022/11/15.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.mail.yanxuan.change.biz.biz;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import com.alibaba.fastjson.JSON;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.netease.mail.yanxuan.change.biz.config.AppConfig;
import com.netease.mail.yanxuan.change.biz.meta.exception.ExceptionFactory;
import com.netease.mail.yanxuan.change.biz.service.ChangeFileService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowExecService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeConfigService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeTypeService;
import com.netease.mail.yanxuan.change.common.bean.CommonConstants;
import com.netease.mail.yanxuan.change.common.bean.RequestLocalBean;
import com.netease.mail.yanxuan.change.common.bean.ResponseCode;
import com.netease.mail.yanxuan.change.common.bean.ResponseCodeEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeFlowEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeResultEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeStatusEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeSubjectEnum;
import com.netease.mail.yanxuan.change.common.enums.FlowOperationTypeEnum;
import com.netease.mail.yanxuan.change.common.enums.FlowxOperationEnum;
import com.netease.mail.yanxuan.change.common.enums.NeedFileEnum;
import com.netease.mail.yanxuan.change.common.util.DateUtils;
import com.netease.mail.yanxuan.change.dal.entity.ChangeConfig;
import com.netease.mail.yanxuan.change.dal.entity.ChangeExecRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeFile;
import com.netease.mail.yanxuan.change.dal.entity.ChangeRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeType;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeRecordMapper;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeExecConfigReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeExecProject;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCancelReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCreateReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowFile;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowSubmitReq;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.BasicChangeFlowVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeFlowExecVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeFlowVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ItemVO;
import com.netease.mail.yanxuan.change.integration.flow.FlowRpcService;
import com.netease.yanxuan.flowx.sdk.meta.controller.communal.AjaxResponse;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.FlowDataDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.UserReachDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.InterfaceInputDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.UserBaseContainerDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.flow.FlowCreateReqDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.flow.NodeSubmitReqDTO;

import lombok.extern.slf4j.Slf4j;

/**
 * @Author zcwang
 * @Date 2022/11/15
 */
@Component
@Slf4j
public class ChangeFlowBiz {

    @Autowired
    private ChangeConfigService changeConfigService;

    @Autowired
    private ChangeTypeService changeTypeService;

    @Autowired
    private ChangeFlowService changeFlowService;

    @Autowired
    private AppConfig appConfig;

    @Autowired
    private FlowRpcService flowRpcService;

    @Autowired
    private ChangeFileService changeFileService;

    @Autowired
    private ChangeFlowExecService changeFlowExecService;

    @Autowired
    private ChangeRecordMapper changeRecordMapper;

    public String createAndSubmit(ChangeFlowCreateReq changeFlowCreateReq) {
        String uid = RequestLocalBean.getUid();
        log.info("[create] createReq={}, uid:{}", JSON.toJSONString(changeFlowCreateReq), uid);
        // 校验变更主体
        checkSubject(changeFlowCreateReq);
        // 检验是否需要资料
        ChangeConfig changeConfig = changeConfigService.getSonChange(changeFlowCreateReq.getSonChangeClassId());
        Assert.notNull(changeConfig, "二级变更类型不存在");
        Integer needFile = changeConfig.getNeedFile();
        if (NeedFileEnum.NEED.getStatus().equals(needFile)) {
            Assert.notEmpty(changeFlowCreateReq.getFiles(), "必须上传资料");
        }
        // 变更行动项不可为空，最多20项
        List<ChangeExecConfigReq> changeExecProject = changeFlowCreateReq.getChangeExecProject();
        Assert.isTrue(changeExecProject.size() <= appConfig.getChangeExecLimit(),
            "变更行动方案配置数超限");
        // 结束时间不可晚于第二天定时任务执行时间
        Long tomorrowSpecificTime = DateUtils.getTomorrowSpecificTime("09:00:00");
        Assert.isTrue(changeFlowCreateReq.getChangeConfirmResultTime() >= tomorrowSpecificTime, "时间不可晚于下次执行时间");
        Map<String, Object> content = new HashMap<>(10);
        content.put("createUserName", uid);
        content.put("createUser", uid);
        content.put("createTime", System.currentTimeMillis());
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.PASS.getValue());
        // 组装工单创建数据
        FlowCreateReqDTO flowCreateReqDTO = buildFlowCreateReqDTO(ChangeFlowEnum.CHANGE_FLOW_START.getTopoId(), uid,
            JSON.toJSONString(content), FlowxOperationEnum.CREATE.getName());
        // 创建工单
        String flowId = createFlow(flowCreateReqDTO);
        // 查询工单详情
        FlowDataDTO flowDataDTO = flowDetail(flowId);
        String nodeId = flowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
        // todo: 根据类型查询负责人详情，格式邮箱
        String changeCommander = appConfig.getTestEmail();
        // 保存工单数据
        ChangeRecord changeRecord = buildChangeRecord(flowId, nodeId, changeFlowCreateReq, changeCommander, uid);
        changeFlowService.saveRecord(changeRecord);
        // 保存变更行动方案记录
        List<ChangeExecRecord> changeExecRecords = buildChangeExecRecord(changeRecord.getId(), changeExecProject);
        changeExecRecords.forEach(exec->changeFlowExecService.saveRecord(exec));
        // 保存附件
        List<ChangeFlowFile> files = changeFlowCreateReq.getFiles();
        if (CollectionUtils.isNotEmpty(files)) {
            List<ChangeFile> changeFiles = buildChangeFileRecord(changeRecord.getId(), files);
            changeFiles.forEach(file->changeFileService.saveRecord(file));
        }
        // 如果发起人=变更负责人
        if (changeCommander.equals(uid)) {
            log.debug("[createAndSubmit] changeCommander:{}, uid:{}", changeCommander, uid);
            String nextNodeId = submitFlow(flowId, flowDataDTO, uid,
                    ChangeFlowEnum.CHANGE_FLOW_START.getTopoId(), JSON.toJSONString(content));
            if (null == nextNodeId) {
                throw ExceptionFactory.createBiz(ResponseCode.SUBMIT_FLOW_ERROR, "首次提交节点失败");
            }
            // 更新节点id
            changeRecord.setFlowNode(nodeId);
            changeRecord.setUpdateTime(DateUtils.getCurrentTime());
            changeFlowService.updateRecord(changeRecord);
            return flowId;
        }
        return flowId;
    }

    private List<ChangeFile> buildChangeFileRecord(Long changeRecordId, List<ChangeFlowFile> files) {
        return files.stream().map(f -> {
            ChangeFile changeFile = new ChangeFile();
            changeFile.setChangeRecordId(changeRecordId);
            changeFile.setFileType(f.getFileType());
            changeFile.setFileName(f.getFileName());
            changeFile.setFileUrl(f.getFileUrl());
            changeFile.setCreateTime(DateUtils.getCurrentTime());
            changeFile.setUpdateTime(DateUtils.getCurrentTime());
            return changeFile;
        }).collect(Collectors.toList());
    }

    private List<ChangeExecRecord> buildChangeExecRecord(Long changeRecordId, List<ChangeExecConfigReq> changeExecProject) {
        return changeExecProject.stream().map(c -> {
            ChangeExecRecord changeExecRecord = new ChangeExecRecord();
            changeExecRecord.setChangeRecordId(changeRecordId);
            changeExecRecord.setChangeExecDepartment(c.getChangeExecDepartment());
            changeExecRecord.setChangeExecUserType(c.getChangeExecUserType());
            changeExecRecord.setChangeExecUser(c.getChangeExecUser());
            changeExecRecord.setChangeRiskDesc(c.getChangeRiskDesc());
            changeExecRecord.setChangeExecProject(c.getChangeExecProject());
            changeExecRecord.setChangeChecking(c.getChangeChecking());
            changeExecRecord.setChangeExecFinishTime(c.getChangeExecFinishTime());
            changeExecRecord.setChangeExecFinishDesc(c.getChangeExecFinishDesc());
            changeExecRecord.setCreateTime(DateUtils.getCurrentTime());
            changeExecRecord.setUpdateTime(DateUtils.getCurrentTime());
            return changeExecRecord;
        }).collect(Collectors.toList());
    }

    private List<ChangeExecRecord> buildChangeExecUpdateRecord(Long changeRecordId, List<ChangeExecProject> changeExecProject) {
        return changeExecProject.stream().map(c -> {
            ChangeExecRecord changeExecRecord = new ChangeExecRecord();
            changeExecRecord.setChangeRecordId(changeRecordId);
            changeExecRecord.setChangeExecDepartment(c.getChangeExecDepartment());
            changeExecRecord.setChangeExecUserType(c.getChangeExecUserType());
            changeExecRecord.setChangeExecUser(c.getChangeExecUser());
            changeExecRecord.setChangeRiskDesc(c.getChangeRiskDesc());
            changeExecRecord.setChangeExecProject(c.getChangeExecProject());
            changeExecRecord.setChangeChecking(c.getChangeChecking());
            changeExecRecord.setChangeExecFinishTime(c.getChangeExecFinishTime());
            changeExecRecord.setChangeExecFinishDesc(c.getChangeExecFinishDesc());
            changeExecRecord.setCreateTime(DateUtils.getCurrentTime());
            changeExecRecord.setUpdateTime(DateUtils.getCurrentTime());
            return changeExecRecord;
        }).collect(Collectors.toList());
    }

    private void checkSubject(ChangeFlowCreateReq changeFlowCreateReq) {
        ChangeSubjectEnum type = ChangeSubjectEnum.getByType(changeFlowCreateReq.getChangeSubject());
        Assert.notNull(type, "变更主体不存在");
        switch (type) {
            case PRODUCT:
                Assert.isTrue(StringUtils.isNotBlank(changeFlowCreateReq.getChangeItems()), " 变更商品不可为空");
                break;
            case SUPPLIER:
                Assert.isTrue(StringUtils.isNotBlank(changeFlowCreateReq.getChangeSupplier()), " 变更供应商不可为空");
                break;
            case OTHER:
                break;
            default:
                throw ExceptionFactory.createBiz(ResponseCode.CHANGE_SUBJECT_ERROR, "变更主体类型错误");
        }
    }

    private FlowDataDTO flowDetail(String flowId) {
        // 查询工单详情，根据详情提交
        FlowDataDTO flowDataDTO;
        AjaxResponse<FlowDataDTO> flowDetailResponse = flowRpcService.getDetail(CommonConstants.FLOWX_PRODUCT, flowId);
        if (ResponseCodeEnum.SUCCESS.getCode() == flowDetailResponse.getCode()) {
            flowDataDTO = flowDetailResponse.getData();
            log.info("[detailFlow] flowDataDTO={}", JSON.toJSONString(flowDataDTO));
        } else {
            log.error("[detailFlow] get flow detail failed, query={}, errMsg={}", JSON.toJSONString(flowId),
                JSON.toJSONString(flowDetailResponse));
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "查询工单详情错误");
        }
        return flowDataDTO;
    }

    private FlowCreateReqDTO buildFlowCreateReqDTO(String topoId, String uid, String content, String operateResult) {
        FlowCreateReqDTO flowCreateReqDTO = new FlowCreateReqDTO();
        flowCreateReqDTO.setTopoId(topoId);
        flowCreateReqDTO.setUid(uid);
        flowCreateReqDTO.setUserName(uid);
        flowCreateReqDTO.setOperateResult(operateResult);
        flowCreateReqDTO.setWorkOrderId(StringUtils.joinWith("-", topoId, UUID.randomUUID().toString()));
        flowCreateReqDTO.setContent(content);
        return flowCreateReqDTO;
    }

    private String submitFlow(String flowId, FlowDataDTO flowDataDTO, String uid, String topoId, String content) {
        NodeSubmitReqDTO nodeSubmitReqDTO = new NodeSubmitReqDTO();
        nodeSubmitReqDTO.setFlowId(flowId);
        nodeSubmitReqDTO.setNodeId(flowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId());
        nodeSubmitReqDTO.setUid(uid);
        nodeSubmitReqDTO.setUserName(uid);
        nodeSubmitReqDTO.setTopoId(topoId);
        nodeSubmitReqDTO.setContent(content);
        nodeSubmitReqDTO.setOperateResult(FlowxOperationEnum.SUBMIT.getName());
        nodeSubmitReqDTO.setCreateTime(System.currentTimeMillis());
        nodeSubmitReqDTO.setApproved(true);
        nodeSubmitReqDTO.setOccLock(flowDataDTO.getOccLock() + 1);
        log.info("[submitFlow] nodeSubmitReqDTO={}", JSON.toJSONString(nodeSubmitReqDTO));
        List<String> nextNodeIdList;
        // 提交工单
        AjaxResponse<List<String>> submitResponse = flowRpcService.submit(CommonConstants.FLOWX_PRODUCT,
            nodeSubmitReqDTO);
        if (ResponseCodeEnum.SUCCESS.getCode() == submitResponse.getCode()) {
            nextNodeIdList = submitResponse.getData();
            log.info("[submitFlow] nextNodeIdList={}", JSON.toJSONString(nextNodeIdList));
        } else {
            log.error("[submitFlow] submit flow failed, query={}, errMsg={}", JSON.toJSONString(nodeSubmitReqDTO),
                JSON.toJSONString(submitResponse));
            throw ExceptionFactory.createBiz(ResponseCode.SUBMIT_FLOW_ERROR, "提交工单错误");
        }
        if (CollectionUtils.isEmpty(nextNodeIdList)) {
            return ChangeFlowEnum.END.getNodeId();
        }
        return nextNodeIdList.get(0);
    }

    private String createFlow(FlowCreateReqDTO flowCreateReqDTO) {
        // 先创建工单，创建完成后直接提交
        log.info("[createFlow] flowCreateReqDTO={}", flowCreateReqDTO);
        AjaxResponse<String> flowCreateResponse = flowRpcService.create(CommonConstants.FLOWX_PRODUCT,
            flowCreateReqDTO);
        String flowId;
        if (ResponseCodeEnum.SUCCESS.getCode() == flowCreateResponse.getCode()) {
            flowId = flowCreateResponse.getData();
            log.info("[createFlow] flowId ={}", JSON.toJSONString(flowId));
        } else {
            log.error("[createFlow] create flow failed, query={}, errMsg={}", JSON.toJSONString(flowCreateReqDTO),
                JSON.toJSONString(flowCreateResponse));
            throw ExceptionFactory.createBiz(ResponseCode.CREATE_FLOW_ERROR, "创建工单错误");
        }
        return flowId;
    }

    private ChangeRecord buildChangeRecord(String flowId, String nodeId, ChangeFlowCreateReq changeFlowCreateReq,
                                           String changeCommander, String uid) {
        ChangeRecord changeRecord = new ChangeRecord();
        changeRecord.setFlowId(Long.parseLong(flowId));
        changeRecord.setFlowNode(nodeId);
        changeRecord.setChangeSubject(changeFlowCreateReq.getChangeSubject());
        changeRecord.setParentChangeClassId(changeFlowCreateReq.getParentChangeClassId());
        changeRecord.setSonChangeClassId(changeFlowCreateReq.getSonChangeClassId());
        changeRecord.setChangeLevel(changeFlowCreateReq.getChangeLevel());
        changeRecord.setChangeCommander(changeCommander);
        changeRecord.setChangeDepartment(changeFlowCreateReq.getChangeDepartment());
        List<ChangeExecConfigReq> changeExecProject = changeFlowCreateReq.getChangeExecProject();
        List<String> execDepartmentList = changeExecProject.stream().map(ChangeExecConfigReq::getChangeExecDepartment)
            .collect(Collectors.toList());
        changeRecord.setParticipateChangeExecDepartment(JSON.toJSONString(execDepartmentList));
        if (ChangeSubjectEnum.PRODUCT.getType().equals(changeFlowCreateReq.getChangeSubject())) {
            // 当变更类型是商品时有值
            changeRecord.setChangeItem(changeFlowCreateReq.getChangeItems());
        }
        if (ChangeSubjectEnum.SUPPLIER.getType().equals(changeFlowCreateReq.getChangeSubject())) {
            changeRecord.setChangeSupplier(changeFlowCreateReq.getChangeSupplier());
        }
        changeRecord.setChangeReason(changeFlowCreateReq.getChangeReason());
        changeRecord.setChangeContent(changeFlowCreateReq.getChangeContent());
        changeRecord.setChangeRiskDesc(changeFlowCreateReq.getChangeRiskDesc());
        changeRecord.setChangeProfit(changeFlowCreateReq.getChangeProfit());
        changeRecord.setChangeProfitDesc(changeFlowCreateReq.getChangeProfitDesc());
        changeRecord.setChangeConfirmResultTime(changeFlowCreateReq.getChangeConfirmResultTime());
        changeRecord.setState(ChangeStatusEnum.IN.getStatus());
        // 变更结论
        changeRecord.setCreateSource(changeFlowCreateReq.getCreateSource());
        changeRecord.setCreateSupplier(changeFlowCreateReq.getChangeSupplier());
        changeRecord.setCreateTime(DateUtils.getCurrentTime());
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
        changeRecord.setCreator(uid);
        return changeRecord;
    }

    public String submit(ChangeFlowSubmitReq changeFlowSubmitReq) {
        log.info("[submitFlow] changeFlowReq:{}", JSON.toJSONString(changeFlowSubmitReq));
        Long flowId = changeFlowSubmitReq.getFlowId();
        // 查询工单有效性
        ChangeRecord changeRecord = this.getFlowInfo(flowId);
        // 检查工单节点
        this.checkNode(changeRecord.getFlowNode(), changeFlowSubmitReq.getCurrentNodeId());
        // todo:检验操作权限
        String uid = RequestLocalBean.getUid();
        String changeCommander = changeRecord.getChangeCommander();
        /*if (!uid.equals(changeCommander)) {
            throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, ResponseCode.NO_AUTH.getMsg());
        }*/
        // 获取工单详情
        FlowDataDTO flowDataDTO = this.flowDetail(flowId.toString());
        if (flowDataDTO == null) {
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "工单查询错误，不存在");
        }

        return checkUpdateAndSubmit(flowId, flowDataDTO, uid, changeRecord, changeRecord.getFlowNode(), changeFlowSubmitReq);
    }

    private String checkUpdateAndSubmit(Long flowId, FlowDataDTO flowDataDTO, String uid, ChangeRecord changeRecord,
        String currentNode, ChangeFlowSubmitReq changeFlowSubmitReq) {
        ChangeFlowEnum node = ChangeFlowEnum.getByNodeId(currentNode);
        Assert.notNull(node, "节点配置不存在");
        log.debug("[checkUpdateAndSubmit] floeId:{}, nodeEnum:{}", flowId, node);
        // 工单流传
        Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.PASS.getValue());
        switch (node) {
            // 变更申请提交节点，可以修改数据，以新的数据为准
            case CHANGE_FLOW_SUBMIT:
                // 检验是否需要资料
                ChangeConfig changeConfig = changeConfigService.getSonChange(changeFlowSubmitReq.getSonChangeClassId());
                Assert.notNull(changeConfig, "二级变更类型不存在");
                Integer needFile = changeConfig.getNeedFile();
                if (NeedFileEnum.NEED.getStatus().equals(needFile)) {
                    Assert.notEmpty(changeFlowSubmitReq.getFiles(), "必须上传资料");
                }
                // 变更行动项不可为空，最多20项
                List<ChangeExecProject> changeExecProjectList = changeFlowSubmitReq.getChangeExecProjectList();
                Assert.isTrue(changeExecProjectList.size() <= appConfig.getChangeExecLimit(),
                        "变更行动方案配置数超限");
                changeRecord.setParentChangeClassId(changeFlowSubmitReq.getParentChangeClassId());
                changeRecord.setSonChangeClassId(changeFlowSubmitReq.getSonChangeClassId());
                List<String> execDepartmentList = changeExecProjectList.stream().map(ChangeExecProject::getChangeExecDepartment)
                        .collect(Collectors.toList());
                changeRecord.setParticipateChangeExecDepartment(JSON.toJSONString(execDepartmentList));
                changeRecord.setChangeItem(changeFlowSubmitReq.getChangeItems());
                changeRecord.setChangeSupplier(changeFlowSubmitReq.getChangeSupplier());
                changeRecord.setChangeReason(changeFlowSubmitReq.getChangeReason());
                changeRecord.setChangeContent(changeFlowSubmitReq.getChangeContent());
                changeRecord.setChangeRiskDesc(changeFlowSubmitReq.getChangeRiskDesc());
                changeRecord.setChangeChecking(changeFlowSubmitReq.getChangeChecking());
                changeRecord.setChangeProfit(changeFlowSubmitReq.getChangeProfit());
                changeRecord.setChangeProfitDesc(changeFlowSubmitReq.getChangeProfitDesc());
                changeRecord.setChangeConfirmResultTime(changeFlowSubmitReq.getChangeConfirmResultTime());
                changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                // 提交工单
                String submitNode = this.submitFlow(flowId.toString(), flowDataDTO, uid,
                        ChangeFlowEnum.CHANGE_FLOW_SUBMIT.getTopoId(), JSON.toJSONString(content));
                // 更新工单数据
                changeRecord.setFlowNode(submitNode);
                changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeFlowService.updateRecord(changeRecord);
                // 更新行动执行方案，覆盖操作，先删除，后插入
                Integer changeExecCount = changeFlowExecService.deleteByChangeRecordId(changeRecord.getId());
                log.info("[CHANGE_FLOW_SUBMIT] delete changeExecCount:{}", changeExecCount);
                // 保存变更行动方案记录
                List<ChangeExecRecord> changeExecRecords = buildChangeExecUpdateRecord(changeRecord.getId(), changeFlowSubmitReq.getChangeExecProjectList());
                changeExecRecords.forEach(exec->changeFlowExecService.saveRecord(exec));
                // 更新附件，覆盖操作，先删除，后插入
                List<ChangeFlowFile> files = changeFlowSubmitReq.getFiles();
                if (CollectionUtils.isNotEmpty(files)) {
                    changeFileService.deleteByChangeRecordId(changeRecord.getId());
                    List<ChangeFile> changeFiles = buildChangeFileRecord(changeRecord.getId(), files);
                    changeFiles.forEach(file -> changeFileService.saveRecord(file));
                }
                // todo: 发送邮件

                return submitNode;
            case CHANGE_FLOW_CONFIRM:
                Integer changeResult = changeFlowSubmitReq.getChangeResult();
                ChangeResultEnum changeResultStatus = ChangeResultEnum.getByStatus(changeResult);
                Assert.notNull(changeResultStatus, "变更结论类型错误");
                // 填写变更结论
                changeRecord.setChangeResult(changeResultStatus.getStatus());
                // 填写备注
                changeRecord.setRemark(changeFlowSubmitReq.getRemark());
                switch (changeResultStatus) {
                    case FINISH_ALL:
                    case FINISH_PART:
                        List<ChangeExecProject> execProjectList = changeFlowSubmitReq.getChangeExecProjectList();
                        List<ChangeExecRecord> changeExecRecordList = execProjectList.stream().map(exec -> {
                            Long changeExecFinishTime = exec.getChangeExecFinishTime();
                            Assert.notNull(changeExecFinishTime, "行动完成时间不可为空" + exec.getChangeTemplateId());
                            String changeExecFinishDesc = exec.getChangeExecFinishDesc();
                            Assert.isTrue(StringUtils.isNotBlank(changeExecFinishDesc),
                                    "行动完成时间不可为空" + exec.getChangeTemplateId());
                            ChangeExecRecord changeExecRecord = new ChangeExecRecord();
                            changeExecRecord.setId(exec.getChangeTemplateId());
                            changeExecRecord.setChangeExecFinishTime(exec.getChangeExecFinishTime());
                            changeExecRecord.setChangeExecFinishDesc(exec.getChangeExecFinishDesc());
                            return changeExecRecord;
                        }).collect(Collectors.toList());
                        // 完成直接完结
                        String confirmNode = this.submitFlow(flowId.toString(), flowDataDTO, uid,
                                ChangeFlowEnum.CHANGE_FLOW_CONFIRM.getTopoId(), JSON.toJSONString(content));
                        changeRecord.setFlowNode(confirmNode);
                        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                        changeFlowService.updateRecord(changeRecord);
                        changeExecRecordList.forEach(exec->changeFlowExecService.update(exec));
                        //todo: 发送邮件
                        return confirmNode;
                    case CANCEL:
                        // 取消填写取消原因
                        String cancelReason = changeFlowSubmitReq.getCancelReason();
                        Assert.isTrue(StringUtils.isNotBlank(cancelReason), "变更取消原因不可为空");
                        changeRecord.setCancelReason(cancelReason);
                        String cancelNode = this.submitFlow(flowId.toString(), flowDataDTO, uid,
                                ChangeFlowEnum.CHANGE_FLOW_CONFIRM.getTopoId(), JSON.toJSONString(content));
                        changeRecord.setFlowNode(cancelNode);
                        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                        changeFlowService.updateRecord(changeRecord);
                        //todo: 发送邮件
                        return cancelNode;
                    case DELAY:
                        // 延期不流转工单
                        long changeConfirmResultTime = changeFlowSubmitReq.getChangeConfirmResultTime();
                        Long tomorrowSpecificTime = DateUtils.getTomorrowSpecificTime("09:00:00");
                        Assert.isTrue(changeConfirmResultTime >= tomorrowSpecificTime, "时间不可晚于下次执行时间");
                        List<ChangeExecProject> execProjectList1 = changeFlowSubmitReq.getChangeExecProjectList();
                        List<ChangeExecRecord> changeExecRecords1 = execProjectList1.stream().map(exec -> {
                            Long changeExecFinishTime = exec.getChangeExecFinishTime();
                            Assert.notNull(changeExecFinishTime, "行动完成时间不可为空" + exec.getChangeTemplateId());
                            String changeExecFinishDesc = exec.getChangeExecFinishDesc();
                            Assert.isTrue(StringUtils.isNotBlank(changeExecFinishDesc),
                                "行动完成时间不可为空" + exec.getChangeTemplateId());
                            ChangeExecRecord changeExecRecord = new ChangeExecRecord();
                            changeExecRecord.setId(exec.getChangeTemplateId());
                            changeExecRecord.setChangeExecFinishTime(exec.getChangeExecFinishTime());
                            changeExecRecord.setChangeExecFinishDesc(exec.getChangeExecFinishDesc());
                            return changeExecRecord;
                        }).collect(Collectors.toList());
                        // 更改变更结果确认时间
                        changeRecord.setChangeConfirmResultTime(changeConfirmResultTime);
                        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                        changeFlowService.updateRecord(changeRecord);
                        changeExecRecords1.forEach(exec->changeFlowExecService.update(exec));
                        // 变更行动项
                        //todo: 发送邮件
                        return null;
                    default:
                        return null;
                }
                // 其他几点，不可提交
            case CHANGE_FLOW_EXE:
            case CHANGE_FLOW_START:
            case END:
            default:
                throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, "不可提交节点" + currentNode);
        }
    }

    private ChangeRecord getFlowInfo(Long flowId) {
        ChangeRecord changeRecord = changeFlowService.getByFlowId(flowId);
        if (changeRecord == null) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "工单id不存在");
        }
        return changeRecord;
    }

    private void checkNode(String recordNode, String currentNodeId){
        if (!recordNode.equals(currentNodeId)) {
            throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, "工单已流转至其他节点");
        }
    };

    public Boolean cancel(ChangeFlowCancelReq req) {
        log.info("[cancel] req:{}", JSON.toJSONString(req));
        Long flowId = req.getFlowId();
        // 查询工单有效性
        ChangeRecord changeRecord = getFlowInfo(flowId);
        // 检查工单节点
        this.checkNode(changeRecord.getFlowNode(), ChangeFlowEnum.CHANGE_FLOW_EXE.getNodeId());
        // todo:检查审核人

        // 获取工单详情
        FlowDataDTO flowDataDTO = this.flowDetail(flowId.toString());
        if (flowDataDTO == null) {
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "工单查询错误，不存在");
        }
        // 工单流转
        Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.REFUSE.getValue());
        // String uid = RequestLocalBean.getUid();
        String uid = "1@.com";
        String nextNodeId = this.submitFlow(String.valueOf(flowId), flowDataDTO, uid, ChangeFlowEnum.CHANGE_FLOW_EXE.getTopoId(), JSON.toJSONString(content));
        log.info("[cancel] nextNodeId:{}", nextNodeId);
        // 填充更新数据
        changeRecord.setFlowNode(nextNodeId);
        changeRecord.setState(ChangeStatusEnum.CANCEL.getStatus());
        changeRecord.setCancelReason(req.getCancelReason());
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
        return changeFlowService.updateRecord(changeRecord);
    }

    public BasicChangeFlowVO quote(Long flowId) {
        // 获取工单详情
        ChangeRecord changeRecord = this.getFlowInfo(flowId);
        // 获取附件
        List<ChangeFlowFile> changeFileList = changeFileService.getChangeFileList(changeRecord.getId());
        return BasicChangeFlowVO.builder().parentChangeClassId(changeRecord.getParentChangeClassId())
            .sonChangeClassId(changeRecord.getSonChangeClassId()).changeSubject(changeRecord.getChangeSubject())
            .changeItem(changeRecord.getChangeItem()).changeSupplier(changeRecord.getChangeSupplier())
            .changeReason(changeRecord.getChangeReason()).changeContent(changeRecord.getChangeContent())
            .changeRiskDesc(changeRecord.getChangeRiskDesc()).changeProfit(changeRecord.getChangeProfit())
            .changeProfitDesc(changeRecord.getChangeProfitDesc()).files(changeFileList).build();
    }

    public ChangeFlowVO detail(Long id) {
        ChangeRecord changeRecord = changeFlowService.getById(id);
        if (changeRecord == null) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "工单id不存在");
        }
        ChangeFlowVO changeFlowVO = new ChangeFlowVO();
        changeFlowVO.setId(changeRecord.getId());
        changeFlowVO.setFlowId(changeRecord.getFlowId());
        changeFlowVO.setChangeSubject(changeRecord.getChangeSubject());
        ChangeType parentChangeType = changeTypeService.getChangeTypeById(changeRecord.getParentChangeClassId());
        if (parentChangeType == null) {
            throw ExceptionFactory.createBiz(ResponseCode.CHANGE_TYPE_NOT_EXIST, "变更类型不存在");
        }
        ChangeType sonChangeType = changeTypeService.getChangeTypeById(changeRecord.getSonChangeClassId());
        if (sonChangeType == null) {
            throw ExceptionFactory.createBiz(ResponseCode.CHANGE_TYPE_NOT_EXIST, "变更类型配置不存在");
        }
        changeFlowVO.setChangeType(parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
        changeFlowVO.setChangeLevel(changeRecord.getChangeLevel());
        changeFlowVO.setChangeDepartment(changeRecord.getChangeDepartment());
        changeFlowVO.setChangeCommander(changeRecord.getChangeCommander());
        changeFlowVO.setSupplier(changeRecord.getChangeSupplier());
        changeFlowVO.setChangeReason(changeRecord.getChangeReason());
        changeFlowVO.setChangeContent(changeRecord.getChangeContent());
        changeFlowVO.setChangeRiskDesc(changeRecord.getChangeRiskDesc());
        // 获取附件
        List<ChangeFlowFile> changeFileList = changeFileService.getChangeFileList(changeRecord.getId());
        changeFlowVO.setFiles(changeFileList);
        changeFlowVO.setChangeProfit(changeRecord.getChangeProfit());
        changeFlowVO.setChangeProfitDesc(changeRecord.getChangeProfitDesc());
        String itemJsonStr = changeRecord.getChangeItem();
        if (StringUtils.isNotBlank(itemJsonStr)) {
            changeFlowVO.setChangeItems(JSON.parseArray(itemJsonStr, ItemVO.class));
        }
        changeFlowVO.setChangeState(changeRecord.getState());
        changeFlowVO.setChangeCreator(changeRecord.getCreator());
        List<ChangeFlowExecVO> changeFlowExecRecord = changeFlowExecService.getChangeFlowExecRecord(changeRecord.getId());
        changeFlowVO.setChangeExecDepartment(changeFlowExecRecord);
        changeFlowVO.setCreateTime(changeRecord.getCreateTime());
        changeFlowVO.setChangeConfirmResultTime(changeRecord.getChangeConfirmResultTime());
        changeFlowVO.setChangeResult(changeRecord.getChangeResult());
        return changeFlowVO;
    }

    public PageInfo<ChangeRecord> query(Integer page, Integer pageSize, ChangeFlowListQueryReq changeFlowListQueryReq) {
        log.info("[query] page:{}, pageSize:{}, changeFlowListQueryReq:{}", page, pageSize, JSON.toJSONString(changeFlowListQueryReq));
        //进行分页
        PageHelper.startPage(page,pageSize);
        return new PageInfo<ChangeRecord>(changeRecordMapper.selectByCondition(changeFlowListQueryReq));
        // List<ChangeFlowVO> list = new ArrayList<>();
        // 处理数据
        /*if (CollectionUtils.isNotEmpty(changeRecords)) {
            list = changeRecords.stream().map(c -> {
                ChangeFlowVO changeFlowVO = new ChangeFlowVO();
                changeFlowVO.setId(c.getId());
                changeFlowVO.setFlowId(c.getFlowId());
                changeFlowVO.setChangeSubject(c.getChangeSubject());
                ChangeType parentChangeType = changeTypeService.getChangeTypeById(c.getParentChangeClassId());
                if (parentChangeType == null) {
                    throw ExceptionFactory.createBiz(ResponseCode.CHANGE_TYPE_NOT_EXIST, "变更类型不存在");
                }
                ChangeType sonChangeType = changeTypeService.getChangeTypeById(c.getSonChangeClassId());
                if (sonChangeType == null) {
                    throw ExceptionFactory.createBiz(ResponseCode.CHANGE_TYPE_NOT_EXIST, "变更类型配置不存在");
                }
                changeFlowVO.setChangeType(parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
                changeFlowVO.setChangeDepartment(c.getChangeDepartment());
                changeFlowVO.setChangeContent(c.getChangeContent());
                changeFlowVO.setChangeCommander(c.getChangeCommander());
                String itemJsonStr = c.getChangeItem();
                if (StringUtils.isNotBlank(itemJsonStr)) {
                    changeFlowVO.setChangeItems(JSON.parseArray(itemJsonStr, ItemVO.class));
                }
                changeFlowVO.setSupplier(c.getChangeSupplier());
                changeFlowVO.setChangeState(c.getState());
                changeFlowVO.setChangeCreator(c.getCreator());
                String participateChangeExecDepartment = c.getParticipateChangeExecDepartment();
                if (StringUtils.isNotBlank(participateChangeExecDepartment)) {
                    changeFlowVO.setExecDepartmentStrList(JSON.parseArray(participateChangeExecDepartment, String.class));
                }
                changeFlowVO.setCreateTime(c.getCreateTime());
                changeFlowVO.setChangeConfirmResultTime(c.getChangeConfirmResultTime());
                changeFlowVO.setCancelReason(c.getCancelReason());
                changeFlowVO.setRemark(c.getRemark());
                return changeFlowVO;
            }).collect(Collectors.toList());
        }*/
    }

    public UserBaseContainerDTO getOperator(InterfaceInputDTO interfaceInput) {
        log.info("[getOperator] interfaceInput:{}", JSON.toJSONString(interfaceInput));
        // 根据工单flowId获取审批人，创建工单时设置，唯一
        String flowId = interfaceInput.getPublicFieldDTO().getFlowMeta().getFlowId();
        String nodeId = interfaceInput.getPublicFieldDTO().getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
        ChangeRecord changeRecord = changeFlowService.getByFlowId(Long.valueOf(flowId));
        String changeCommander = changeRecord.getChangeCommander();
        if (null == changeCommander) {
            throw ExceptionFactory.createBiz(ResponseCode.OPERATOR_NOT_EXIST, ResponseCode.OPERATOR_NOT_EXIST.getMsg());
        }
        UserReachDTO userReachDTO = new UserReachDTO();
        userReachDTO.setUserName(changeCommander);
        userReachDTO.setUid(changeCommander);
        UserReachDTO userReachDTO2 = new UserReachDTO();
        userReachDTO2.setUserName("test2");
        userReachDTO2.setUid("test22");
        List<UserReachDTO> totalUserList = Arrays.asList(userReachDTO);
        UserBaseContainerDTO userBaseContainer = new UserBaseContainerDTO();
        userBaseContainer.setUserList(totalUserList);
        return userBaseContainer;
    }
}
