/**
 * 同舟端状态枚举
 * 用于同舟端传入的状态编码，映射到内部详细状态
 * 
 * @Author AI Assistant
 * @Date 2025/12/16
 */
package com.netease.mail.yanxuan.change.common.enums;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 同舟端状态枚举
 * 可选值：待发起、待行动方案确认、待变更审批、待变更执行、待确认执行结果、已完结、已取消
 * 
 * 映射关系：
 * - 待发起 -> 6 (WAIT_SUBMIT_CHANGE_APPLY)
 * - 待行动方案确认 -> 7 (WAIT_CONFIRM_CHANGE_PLAN)
 * - 待变更审批 -> 8,9,10 (WAIT_DEPT_LEADER_APPROVE, WAIT_CHANGE_ADMIN_APPROVE, WAIT_QUALITY_LEADER_APPROVE)
 * - 待变更执行 -> 11 (WAIT_EXEC_CHANGE_PLAN)
 * - 待确认执行结果 -> 12 (WAIT_CONFIRM_EXEC_RESULT)
 * - 已完结 -> 13 (FINISHED)
 * - 已取消 -> 14 (CANCELLED)
 * 
 * 同舟端枚举编码（向后扩展）：
 * - 待发起 -> 100
 * - 待行动方案确认 -> 101
 * - 待变更审批 -> 102
 * - 待变更执行 -> 103
 * - 待确认执行结果 -> 104
 * - 已完结 -> 105
 * - 已取消 -> 106
 */
public enum TongZhouStatusEnum {
    
    /**
     * 待发起
     */
    WAIT_START("待发起", 100, Arrays.asList(ChangeStatusEnum.WAIT_SUBMIT_CHANGE_APPLY.getStatus())),
    
    /**
     * 待行动方案确认
     */
    WAIT_CONFIRM_ACTION_PLAN("待行动方案确认", 101, Arrays.asList(ChangeStatusEnum.WAIT_CONFIRM_CHANGE_PLAN.getStatus())),
    
    /**
     * 待变更审批（包含：待部门负责人审批、待变更管理员审批、待质量部负责人审批）
     */
    WAIT_CHANGE_APPROVE("待变更审批", 102, Arrays.asList(
            ChangeStatusEnum.WAIT_DEPT_LEADER_APPROVE.getStatus(),
            ChangeStatusEnum.WAIT_CHANGE_ADMIN_APPROVE.getStatus(),
            ChangeStatusEnum.WAIT_QUALITY_LEADER_APPROVE.getStatus()
    )),
    
    /**
     * 待变更执行
     */
    WAIT_CHANGE_EXEC("待变更执行", 103, Arrays.asList(ChangeStatusEnum.WAIT_EXEC_CHANGE_PLAN.getStatus())),
    
    /**
     * 待确认执行结果
     */
    WAIT_CONFIRM_RESULT("待确认执行结果", 104, Arrays.asList(ChangeStatusEnum.WAIT_CONFIRM_EXEC_RESULT.getStatus())),
    
    /**
     * 已完结
     */
    FINISHED("已完结", 105, Arrays.asList(ChangeStatusEnum.FINISHED.getStatus())),
    
    /**
     * 已取消
     */
    CANCELLED("已取消", 106, Arrays.asList(ChangeStatusEnum.CANCELLED.getStatus()));
    
    private final Integer code; // 同舟端枚举编码（用于向后扩展）
    private final List<Integer> statusList;
    
    TongZhouStatusEnum(String desc, Integer code, List<Integer> statusList) {
        this.code = code;
        this.statusList = statusList;
    }
    
    /**
     * 根据同舟端枚举编码转换为内部状态列表
     * @param code 同舟端枚举编码（100-106）
     * @return 内部状态列表，如果不存在返回null
     */
    public static List<Integer> convertToStatusListByCode(Integer code) {
        if (code == null) {
            return null;
        }
        for (TongZhouStatusEnum statusEnum : values()) {
            if (statusEnum.code.equals(code)) {
                return new ArrayList<>(statusEnum.statusList);
            }
        }
        return null;
    }
}

