/**
 * @(#)ChangeFlowController.java, 2022/11/14.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.mail.yanxuan.change.web.controller;

import javax.validation.Valid;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.logging.log4j.util.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.netease.mail.yanxuan.change.biz.biz.ChangeFlowBiz;
import com.netease.mail.yanxuan.change.biz.config.TitleConfig;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowService;
import com.netease.mail.yanxuan.change.biz.service.DESExportService;
import com.netease.mail.yanxuan.change.biz.service.InteriorChangeConfigService;
import com.netease.mail.yanxuan.change.common.bean.AjaxResult;
import com.netease.mail.yanxuan.change.common.bean.RequestLocalBean;
import com.netease.mail.yanxuan.change.common.enums.CreateSourceEnum;
import com.netease.mail.yanxuan.change.dal.meta.model.po.QueryCommanderEmailReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCancelReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCreateReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowDeliverReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowRequest;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowSubmitReq;
import com.netease.mail.yanxuan.change.dal.meta.model.rpc.CommanderResponse;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.BasicChangeFlowVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeFlowListVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeFlowVO;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.FlowDataDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.InterfaceInputDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.UserBaseContainerDTO;

import lombok.extern.slf4j.Slf4j;

import java.util.List;

/**
 * @Author zcwang
 * @Date 2022/11/14
 */
@Slf4j
@RestController
@RequestMapping("/change/flow")
public class ChangeFlowController {

    @Autowired
    private ChangeFlowService changeFlowService;

    @Autowired
    private ChangeFlowBiz changeFlowBiz;

    @Autowired
    private InteriorChangeConfigService interiorChangeConfigService;

    @Autowired
    private DESExportService desExportService;

    /**
     * 变更工单详情查询（改）
     * 
     * @return
     */
    @GetMapping("/detail")
    public AjaxResult<ChangeFlowVO> detail(@RequestParam Long flowId) {
        return AjaxResult.success(changeFlowBiz.detail(flowId));
    }

    /**
     * 变更工单列表查询（改）
     * 
     * @return
     */
    @GetMapping("/query")
    public AjaxResult<ChangeFlowListVO> query(@RequestParam(name = "page", defaultValue = "1") Integer page,
                                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                              @RequestParam(name = "createSource") Integer createSource,
                                              @RequestParam(name = "itemId", required = false) String itemId,
                                              @RequestParam(name = "skuId", required = false) String skuId,
                                              @RequestParam(name = "supplier", required = false) String supplier,
                                              @RequestParam(name = "parentChangeClassId", required = false) Long parentChangeClassId,
                                              @RequestParam(name = "sonChangeClassId", required = false) Long sonChangeClassId,
                                              @RequestParam(name = "changeState", required = false) Integer changeState,
                                              @RequestParam(name = "changeCommander", required = false) String changeCommander,
                                              @RequestParam(name = "changeDepartment", required = false) String changeDepartment,
                                              @RequestParam(name = "changeExecDepartment", required = false) String changeExecDepartment,
                                              @RequestParam(name = "changeExecUser", required = false) String changeExecUser,
                                              @RequestParam(name = "startTime", required = false) Long startTime,
                                              @RequestParam(name = "endTime", required = false) Long endTime,
                                              @RequestParam(name = "flowId", required = false) Long flowId,
                                              @RequestParam(name = "onlyMyFollowed", defaultValue = "true") Boolean onlyMyFollowed) {
        ChangeFlowListQueryReq changeFlowListQueryReq = new ChangeFlowListQueryReq();
        changeFlowListQueryReq.setChangeExecUser(changeExecUser);
        changeFlowListQueryReq.setItemId(itemId);
        changeFlowListQueryReq.setSkuId(skuId);
        changeFlowListQueryReq.setSupplier(supplier);
        changeFlowListQueryReq.setParentChangeClassId(parentChangeClassId);
        changeFlowListQueryReq.setSonChangeClassId(sonChangeClassId);
        changeFlowListQueryReq.setChangeState(changeState);
        changeFlowListQueryReq.setChangeCommander(changeCommander);
        changeFlowListQueryReq.setChangeDepartment(changeDepartment);
        changeFlowListQueryReq.setChangeExecDepartment(changeExecDepartment);
        changeFlowListQueryReq.setStartTime(startTime);
        changeFlowListQueryReq.setEndTime(endTime);
        changeFlowListQueryReq.setFlowId(flowId);
        changeFlowListQueryReq.setOnlyMyFollowed(onlyMyFollowed);
        // QC端/协同中心可以查看所有，不加入过滤条件
        if (CreateSourceEnum.QC.getType().equals(createSource)
                || CreateSourceEnum.COLLABORATION_CENTER.getType().equals(createSource)) {
            // 不设置 createSource，可以查看所有
        } else if (CreateSourceEnum.TONG_ZHOU.getType().equals(createSource)) {
            Assert.isTrue(Strings.isNotBlank(supplier), "同舟端查询供应商id不可为空");
            changeFlowListQueryReq.setCreateSource(createSource);
            changeFlowListQueryReq.setSupplier(null);
            changeFlowListQueryReq.setCreateSupplier(supplier);
        } else {
            changeFlowListQueryReq.setCreateSource(createSource);
        }
        return AjaxResult.success(changeFlowBiz.query(page, pageSize, changeFlowListQueryReq));
    }

    /**
     * 返回提示文案
     * 
     * @return
     */
    @GetMapping("/title")
    public AjaxResult<TitleConfig> title() {
        return AjaxResult.success(changeFlowService.getTitleConfig());
    }

    /**
     * 发起变更，如果发起人=审核人，自动跳转到第一个节点（改）
     * 
     * @return
     */
    @PostMapping("/create")
    public AjaxResult<String> createAndSubmit(@RequestBody @Valid ChangeFlowCreateReq changeFlowCreateReq) {
        return AjaxResult.success(changeFlowBiz.createAndSubmit(changeFlowCreateReq));
    }

    /**
     * 变更工单提交（改）
     * 
     * @return
     */
    @PostMapping("/submit")
    public AjaxResult<String> submit(@RequestBody @Valid ChangeFlowSubmitReq changeFlowSubmitReq) {
        return AjaxResult.success(changeFlowBiz.submit(changeFlowSubmitReq));
    }

    /**
     * 工单转交
     * 
     * @return
     */
    @PostMapping("/deliver")
    public AjaxResult<Void> deliver(@RequestBody @Valid ChangeFlowDeliverReq req) {
        changeFlowBiz.deliver(req);
        return AjaxResult.success();
    }

    /**
     * 工单取消
     * 
     * @return
     */
    @PostMapping("/cancel")
    public AjaxResult<Boolean> cancel(@RequestBody @Valid ChangeFlowCancelReq changeFlowCancelReq) {
        return AjaxResult.success(changeFlowBiz.cancel(changeFlowCancelReq));
    }

    /**
     * 引用变更工单
     * 
     * @return
     */
    @GetMapping("/quote")
    public AjaxResult<BasicChangeFlowVO> quote(@RequestParam Long flowId,
        @RequestParam(name = "supplier", required = false) String supplier,
        @RequestParam(name = "createSource", required = false, defaultValue = "1") Integer createSource) {
        return AjaxResult.success(changeFlowBiz.quote(flowId, supplier, createSource));
    }

    /**
     * 导出变更（改）
     * 
     * @return
     */
    @GetMapping("/export")
    public AjaxResult<String> export(@RequestParam(name = "itemId", required = false) String itemId,
                                    @RequestParam(name = "skuId", required = false) String skuId,
                                    @RequestParam(name = "supplier", required = false) String supplier,
                                    @RequestParam(name = "parentChangeClassId", required = false) Long parentChangeClassId,
                                    @RequestParam(name = "sonChangeClassId", required = false) Long sonChangeClassId,
                                    @RequestParam(name = "changeState", required = false) Integer changeState,
                                    @RequestParam(name = "changeCommander", required = false) String changeCommander,
                                    @RequestParam(name = "changeDepartment", required = false) String changeDepartment,
                                    @RequestParam(name = "changeExecDepartment", required = false) String changeExecDepartment,
                                    @RequestParam(name = "changeExecUser", required = false) String changeExecUser,
                                    @RequestParam(name = "startTime", required = false) Long startTime,
                                    @RequestParam(name = "endTime", required = false) Long endTime,
                                    @RequestParam(name = "createSource", required = false, defaultValue = "1") Integer createSource,
                                    @RequestParam(name = "serviceCode") String serviceCode,
                                    @RequestParam(name = "flowId", required = false) Long flowId,
                                    @RequestParam(name = "onlyMyFollowed", defaultValue = "true") Boolean onlyMyFollowed) {
        ChangeFlowRequest changeFlowRequest = new ChangeFlowRequest();
        changeFlowRequest.setItemId(itemId);
        changeFlowRequest.setSkuId(skuId);
        changeFlowRequest.setSupplier(supplier);
        changeFlowRequest.setParentChangeClassId(parentChangeClassId);
        changeFlowRequest.setSonChangeClassId(sonChangeClassId);
        changeFlowRequest.setChangeState(changeState);
        changeFlowRequest.setChangeCommander(changeCommander);
        changeFlowRequest.setChangeDepartment(changeDepartment);
        changeFlowRequest.setChangeExecDepartment(changeExecDepartment);
        changeFlowRequest.setChangeExecUser(changeExecUser);
        changeFlowRequest.setStartTime(startTime);
        changeFlowRequest.setEndTime(endTime);
        changeFlowRequest.setServiceCode(serviceCode);
        changeFlowRequest.setFlowId(flowId);
        changeFlowRequest.setOnlyMyFollowed(onlyMyFollowed);
        // QC端/协同中心可以查看所有，不加入过滤条件
        if (CreateSourceEnum.QC.getType().equals(createSource)
                || CreateSourceEnum.COLLABORATION_CENTER.getType().equals(createSource)) {
            // 不设置 createSource，可以查看所有
        } else if (CreateSourceEnum.TONG_ZHOU.getType().equals(createSource)) {
            Assert.isTrue(Strings.isNotBlank(supplier), "同舟端查询供应商id不可为空");
            changeFlowRequest.setCreateSource(createSource);
            changeFlowRequest.setSupplier(null);
            changeFlowRequest.setCreateSupplier(supplier);
        } else {
            changeFlowRequest.setCreateSource(createSource);
        }
        String s = this.exportAddressBook(changeFlowRequest);
        return AjaxResult.success(s);
    }

    public String exportAddressBook(ChangeFlowRequest req) {
        req.setUid(RequestLocalBean.getUid());
        // 数据可见范围：仅查看我跟进的工单（默认true）
        if (req.getOnlyMyFollowed() != null && req.getOnlyMyFollowed()) {
            String currentUser = RequestLocalBean.getUid();
            if (Strings.isNotBlank(currentUser)) {
                req.setCurrentUser(currentUser);
                log.info("[export] 数据可见范围：仅查看我跟进的工单, currentUser:{}", currentUser);
            }
        }
        return desExportService.changeFlowExport(req);
    }

    /**
     * 获取审批人
     * 
     * @param interfaceInput
     * @return
     */
    @PostMapping("/getOperator")
    public AjaxResult<UserBaseContainerDTO> getOperator(@RequestBody InterfaceInputDTO interfaceInput) {
        return AjaxResult.success(changeFlowBiz.getOperator(interfaceInput));
    }

    /**
     * 查询负责人信息
     * @param req
     * @return
     */
    @PostMapping("/queryCommanderInfo")
    public AjaxResult<CommanderResponse> queryCommanderInfo(@RequestBody QueryCommanderEmailReq req) {
        List<CommanderResponse> commanderResponseList = interiorChangeConfigService.queryCommanderEmail(req);
        if (CollectionUtils.isEmpty(commanderResponseList)) {
            return AjaxResult.success();
        }
        return AjaxResult.success(commanderResponseList.get(0));
    }

    /**
     * 查询负责人信息
     * @param req
     * @return
     */
    @PostMapping("/queryCommanderInfoV2")
    public AjaxResult<List<CommanderResponse>> queryCommanderInfoV2(@RequestBody QueryCommanderEmailReq req) {
        return AjaxResult.success(interiorChangeConfigService.queryCommanderEmail(req));
    }

}
