/**
 * @(#)ChangeFlowBiz.java, 2022/11/15.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.mail.yanxuan.change.biz.biz;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.*;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import com.netease.mail.yanxuan.change.common.enums.*;
import com.netease.mail.yanxuan.change.common.constants.FlowTransitionType;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import com.alibaba.fastjson.JSON;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.netease.mail.yanxuan.change.biz.config.AppConfig;
import com.netease.mail.yanxuan.change.biz.meta.exception.ExceptionFactory;
import com.netease.mail.yanxuan.change.biz.service.BuildAndSendEmail;
import com.netease.mail.yanxuan.change.biz.service.ChangeFileService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowExecService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowService;
import com.netease.mail.yanxuan.change.biz.service.ChangeSubFlowRecordService;
import com.netease.mail.yanxuan.change.biz.service.InteriorChangeConfigService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeConfigService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeTypeService;
import com.netease.mail.yanxuan.change.biz.service.rpc.FlowService;
import com.netease.mail.yanxuan.change.biz.service.rpc.ItemService;
import com.netease.mail.yanxuan.change.biz.service.rpc.IusService;
import com.netease.mail.yanxuan.change.biz.service.rpc.QCService;
import com.netease.mail.yanxuan.change.biz.service.rpc.SupplierSendService;
import com.netease.mail.yanxuan.change.biz.service.rpc.SupplierService;
import com.netease.mail.yanxuan.change.biz.util.PageUtils;
import com.netease.mail.yanxuan.change.common.bean.CommonConstants;
import com.netease.mail.yanxuan.change.common.bean.RequestLocalBean;
import com.netease.mail.yanxuan.change.common.bean.ResponseCode;
import com.netease.mail.yanxuan.change.common.util.DateUtils;
import com.netease.mail.yanxuan.change.dal.entity.ChangeConfig;
import com.netease.mail.yanxuan.change.dal.entity.ChangeExecRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeFile;
import com.netease.mail.yanxuan.change.dal.entity.ChangeRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeSubFlowRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeType;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeExecRecordMapper;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeRecordMapper;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeCommanderPO;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeConfigPo;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeGoodsPrincipalPO;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeExecConfigReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCancelReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCreateReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowDeliverReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowFile;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowSubmitReq;
import com.netease.mail.yanxuan.change.dal.meta.model.rpc.GoodsResponseRpc;
import com.netease.mail.yanxuan.change.integration.email.enums.EmailTemplateEnum;
import com.netease.mail.yanxuan.change.integration.email.service.IEmailService;
import com.netease.mail.yanxuan.change.integration.excel.ChangeFlowExcelDTO;
import com.netease.mail.yanxuan.change.integration.flow.UserQueryDTO;
import com.netease.mail.yanxuan.change.integration.flow.ius.IusRpcService;
import com.netease.mail.yanxuan.change.integration.flow.ius.req.IusDepartmentReq;
import com.netease.mail.yanxuan.change.integration.flow.ius.rsp.IusUserInfoRsp;
import com.netease.mail.yanxuan.change.integration.flow.ius.rsp.SecondaryDepartments;
import com.netease.mail.yanxuan.change.integration.flow.supplier.rsp.SupplierSimpleRsp;
import com.netease.mail.yanxuan.change.integration.item.SimplePhyCateGoryResultCo;
import com.netease.mail.yanxuan.change.integration.item.meta.SpuTO;
import com.netease.mail.yanxuan.change.integration.item.param.BatchQuerySpuInfoParam;
import com.netease.mail.yanxuan.change.integration.item.param.CommonIdsParamQuery;
import com.netease.mail.yanxuan.change.integration.qc.meta.QcCategoryVO;
import com.netease.yanxuan.flowx.sdk.meta.controller.communal.AjaxResponse;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.FlowDataDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.UserBaseDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.UserReachDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.InterfaceInputDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.UserBaseContainerDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.flow.FlowCreateReqDTO;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import lombok.extern.slf4j.Slf4j;

/**
 * @Author zcwang
 * @Date 2022/11/15
 */
@Component
@Slf4j
public class ChangeFlowBiz {

    @Autowired
    private ChangeConfigService changeConfigService;

    @Autowired
    private ChangeTypeService changeTypeService;

    @Autowired
    private ChangeFlowService changeFlowService;

    @Autowired
    private AppConfig appConfig;

    @Autowired
    private FlowService flowService;

    @Autowired
    private ChangeFileService changeFileService;

    @Autowired
    private ChangeFlowExecService changeFlowExecService;

    @Autowired
    private ChangeSubFlowRecordService changeSubFlowRecordService;

    @Autowired
    private ChangeRecordMapper changeRecordMapper;

    @Autowired
    private ChangeExecRecordMapper changeExecRecordMapper;

    @Autowired
    private ItemService itemService;

    @Autowired
    private InteriorChangeConfigService interiorChangeConfigService;

    @Autowired
    private IEmailService iEmailService;

    @Autowired
    private IusService iusService;

    @Autowired
    private IusRpcService iusRpcService;

    @Autowired
    private DepartmentLeaderBiz departmentLeaderBiz;

    @Autowired
    private SupplierSendService sendSupplierEmail;

    @Autowired
    private SupplierService supplierService;

    @Autowired
    private QCService qcService;

    @Autowired
    private BuildAndSendEmail buildAndSendEmail;

    @Autowired
    private ChangeExecRecordBiz changeExecRecordBiz;

    public String createAndSubmit(ChangeFlowCreateReq changeFlowCreateReq) {
        String uid = RequestLocalBean.getUid();
        String name = RequestLocalBean.getName();
        log.info("[create] createReq={}, uid:{}, name:{}", JSON.toJSONString(changeFlowCreateReq), uid, name);
        
        // 校验变更主体类型
        ChangeSubjectEnum subjectEnum = ChangeSubjectEnum.getByType(changeFlowCreateReq.getChangeSubject());
        Assert.notNull(subjectEnum, "变更主体类型不能为空或不存在");
        
        // 根据变更类型分别处理
        switch (subjectEnum) {
            case PRODUCT:
                return createProductChangeFlow(changeFlowCreateReq, uid, name);
            case SUPPLIER:
                return createSupplierChangeFlow(changeFlowCreateReq, uid, name);
            case OTHER:
                return createOtherChangeFlow(changeFlowCreateReq, uid, name);
            default:
                throw ExceptionFactory.createBiz(ResponseCode.CHANGE_SUBJECT_ERROR, "不支持的变更主体类型");
        }
    }

    /**
     * 创建商品变更工单
     */
    private String createProductChangeFlow(ChangeFlowCreateReq changeFlowCreateReq, String uid, String name) {
        // 商品变更：校验商品不能为空
        Assert.isTrue(StringUtils.isNotBlank(changeFlowCreateReq.getChangeItems()), "变更商品不可为空");
        // 商品变更需要关联供应商
        if (CreateSourceEnum.TONG_ZHOU.getType().equals(changeFlowCreateReq.getCreateSource())) {
            // 供应商发起：取当前人为供应商
            changeFlowCreateReq.setChangeSupplier(uid);
        } else {
            Assert.isTrue(StringUtils.isNotBlank(changeFlowCreateReq.getChangeSupplier()), "变更供应商不可为空");
        }
        // 解析商品ID列表
        List<Long> itemIds = parseItemIds(changeFlowCreateReq.getChangeItems());
        // 执行公共创建流程（内部已处理邮件逻辑）
        return executeCommonCreateFlow(changeFlowCreateReq, uid, name, itemIds);
    }

    /**
     * 创建供应商变更工单
     */
    private String createSupplierChangeFlow(ChangeFlowCreateReq changeFlowCreateReq, String uid, String name) {
        Assert.isTrue(StringUtils.isNotBlank(changeFlowCreateReq.getChangeSupplier()), "变更供应商不可为空");
        // 执行公共创建流程（内部已处理邮件逻辑）
        return executeCommonCreateFlow(changeFlowCreateReq, uid, name, null);
    }

    /**
     * 创建其他变更工单
     */
    private String createOtherChangeFlow(ChangeFlowCreateReq changeFlowCreateReq, String uid, String name) {
        // 执行公共创建流程（内部已处理邮件逻辑）
        return executeCommonCreateFlow(changeFlowCreateReq, uid, name, null);
    }

    /**
     * 解析商品ID列表
     */
    private List<Long> parseItemIds(String changeItems) {
        List<ItemVO> itemVOS = JSON.parseArray(changeItems, ItemVO.class);
        return itemVOS.stream().map(ItemVO::getItemId).collect(Collectors.toList());
    }

    /**
     * 执行公共的创建流程逻辑
     */
    private String executeCommonCreateFlow(ChangeFlowCreateReq changeFlowCreateReq, String uid, String name,
                                          List<Long> itemIds) {
        Long parentChangeClassId = changeFlowCreateReq.getParentChangeClassId();
        Long sonChangeClassId = changeFlowCreateReq.getSonChangeClassId();
        // 获取工单负责人
        String changeCommander = getChangeCommander(changeFlowCreateReq, uid, parentChangeClassId, sonChangeClassId, itemIds);
        Assert.notNull(changeCommander, "未查询到负责人信息");
        // 检验是否需要资料
        ChangeConfig changeConfig = changeConfigService.getSonChange(sonChangeClassId);
        Assert.notNull(changeConfig, "二级变更类型不存在");
        Integer needFile = changeConfig.getNeedFile();
        if (NeedFileEnum.NEED.getStatus().equals(needFile)) {
            Assert.notEmpty(changeFlowCreateReq.getUploadFiles(), "必须上传资料");
        }
        changeFlowCreateReq.setChangeDepartment(changeConfig.getChangeDepartment());
        // 变更行动项不可为空，最多20项
        List<ChangeExecConfigReq> changeExecProject = changeFlowCreateReq.getChangeExecProject();
        if (CollectionUtils.isEmpty(changeExecProject)) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "行动项列表不能为空");
        }
        Assert.isTrue(changeExecProject.size() <= appConfig.getChangeExecLimit(),
                "变更行动方案配置数超限");
        // 校验每个行动项的必填字段
        changeExecRecordBiz.validateChangeExecProjectRequiredFields(changeExecProject);
        // 校验变更行动方案中是否有重复的变更行动人
        validateDuplicateChangeExecUser(changeExecProject);
        // 结束时间不可晚于第二天定时任务执行时间
        Long tomorrowSpecificTime = DateUtils.getTomorrowSpecificTime("00:00:00");
        Assert.isTrue(changeFlowCreateReq.getChangeConfirmResultTime() >= tomorrowSpecificTime, "时间不可晚于下次执行时间");
        // 构建工单内容
        Map<String, Object> content = buildFlowContent(uid);
        // 构建变更类型
        StringBuilder changeType = buildChangeType(changeFlowCreateReq);
        // 变更工单命名：变更主体+变更类型+变更商品（SPU+商品名称）/变更供应商（供应商ID+供应商名称）
        String subject = ChangeSubjectEnum.getDescByType(changeFlowCreateReq.getChangeSubject());
        String flowName = subject + changeType;
        flowName += buildFlowNameBySubject(changeFlowCreateReq);

        // 组装topo工单创建数据
        FlowCreateReqDTO flowCreateReqDTO = buildFlowCreateReqDTO(ChangeFlowEnum.NEW_CHANGE_FLOW_START.getTopoId(), uid,
                JSON.toJSONString(content), FlowxOperationEnum.CREATE.getName(), name, flowName);
        // 创建工单
        String flowId = flowService.createFlow(flowCreateReqDTO);
        // 查询工单详情
        FlowDataDTO flowDataDTO = flowService.flowDetail(flowId);
        String nodeId = flowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
        // 保存工单数据
        ChangeRecord changeRecord = buildChangeRecord(flowId, nodeId, changeFlowCreateReq, changeCommander, uid);
        changeFlowService.saveRecord(changeRecord);
        // 保存变更行动方案记录（不绑定子流程，待触发时再创建）
        List<ChangeExecRecord> changeExecRecords = changeExecRecordBiz.buildChangeExecRecord(changeRecord.getId(), changeExecProject, null, null);
        changeExecRecords.forEach(exec -> changeFlowExecService.saveRecord(exec));
        
        // 保存附件
        saveChangeFiles(changeRecord.getId(), changeFlowCreateReq);
        
        // 如果发起人=变更负责人，直接提交到确认变更方案节点，创建子流程并发送邮件
        if (changeCommander.equals(uid)) {
            submitToExecutionNode(flowId, flowDataDTO, uid, content, changeRecord);
            // 创建子流程并绑定
            createAndBindSubFlows(changeRecord, flowName);
            buildAndSendEmail.buildAndSendEmailCreate(changeRecord, changeExecRecords, changeType, changeCommander, changeFlowCreateReq);
        } else {
            // 如果发起人≠变更负责人，只发送邮件（不提交节点，不创建子流程）
            sendCreateFlowEmail(changeRecord, changeCommander, changeType, changeFlowCreateReq);
        }
        return flowId;
    }

    /**
     * 保存变更文件
     */
    private void saveChangeFiles(Long changeRecordId, ChangeFlowCreateReq changeFlowCreateReq) {
        List<ChangeFile> allFiles = new ArrayList<>();
        // 变更前后图片/视频，非必填
        List<ChangeFlowFile> changeFiles = changeFlowCreateReq.getChangeFiles();
        if (CollectionUtils.isNotEmpty(changeFiles)) {
            allFiles.addAll(buildChangeFileRecord(changeRecordId, changeFiles, FileTypeEnum.CHANGE.getType()));
        }
        // 保存附件，根据配置看是否必传
        List<ChangeFlowFile> uploadFiles = changeFlowCreateReq.getUploadFiles();
        if (CollectionUtils.isNotEmpty(uploadFiles)) {
            allFiles.addAll(buildChangeFileRecord(changeRecordId, uploadFiles, FileTypeEnum.UPLOAD.getType()));
        }
        if (CollectionUtils.isNotEmpty(allFiles)) {
            allFiles.forEach(file -> changeFileService.saveRecord(file));
        }
    }

    /**
     * 创建变更行动工单并绑定行动项
     * 说明：按照 changeExecDepartment + changeExecUserEmail 维度分组，每个分组创建一个变更行动工单
     * 前端传值是拆分的（每个行动项一条记录），但需要汇总到同一个变更行动工单
     *
     * @param changeRecord 主流程记录
     * @param flowName 主流程名称
     */
    private void createAndBindSubFlows(ChangeRecord changeRecord, String flowName) {
        // 从数据库查询执行记录列表
        List<ChangeExecRecord> execRecords = changeFlowExecService.getChangeExecRecordList(changeRecord.getId());
        if (CollectionUtils.isEmpty(execRecords)) {
            log.warn("[createAndBindSubFlows] 未查询到执行记录, changeRecordId:{}", changeRecord.getId());
            return;
        }
        
        // 按照 changeExecDepartment + changeExecUserEmail 维度分组
        Map<String, List<ChangeExecRecord>> groupedRecords = execRecords.stream()
                .filter(record -> StringUtils.isNotBlank(record.getChangeExecUserEmail()))
                .collect(Collectors.groupingBy(record -> 
                    (StringUtils.isNotBlank(record.getChangeExecDepartment()) ? record.getChangeExecDepartment() : "") 
                    + "_" + record.getChangeExecUserEmail()));
        
        if (groupedRecords.isEmpty()) {
            log.warn("[createAndBindSubFlows] 未找到有效的执行记录（行动人邮箱为空）, changeRecordId:{}", changeRecord.getId());
            return;
        }
        
        // 为每个分组创建一个变更行动工单
        for (Map.Entry<String, List<ChangeExecRecord>> entry : groupedRecords.entrySet()) {
            List<ChangeExecRecord> groupRecords = entry.getValue();
            
            if (CollectionUtils.isEmpty(groupRecords)) {
                continue;
            }
            
            // 取第一个记录作为代表（同一分组下的记录，changeExecDepartment 和 changeExecUserEmail 相同）
            ChangeExecRecord firstRecord = groupRecords.get(0);
            String execUserEmail = firstRecord.getChangeExecUserEmail();
            String execDepartment = firstRecord.getChangeExecDepartment();
            
            // 查询行动人名称
            IusUserInfoRsp execUser = iusService.queryUserInfo(execUserEmail);
            String execUserName = execUser == null || StringUtils.isBlank(execUser.getName()) 
                    ? execUserEmail : execUser.getName();
            
            // 构建变更行动工单内容
            Map<String, Object> subFlowContent = buildFlowContent(execUserEmail);
            
            // 构建变更行动工单名称：主流程名称 + 变更行动部门 + 行动人（截取前100字符）
            String subFlowName = flowName + "-" + execDepartment + "-" + execUserName;
            if (subFlowName.length() > 100) {
                subFlowName = subFlowName.substring(0, 100) + "...";
            }
            
            // 创建变更行动工单
            FlowCreateReqDTO subFlowCreateReqDTO = buildFlowCreateReqDTO(ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), 
                    execUserEmail, JSON.toJSONString(subFlowContent), FlowxOperationEnum.CREATE.getName(), 
                    execUserName, subFlowName);
            String subFlowId = flowService.createFlow(subFlowCreateReqDTO);
            // 查询工单详情
            FlowDataDTO subFlowDataDTO = flowService.flowDetail(subFlowId);
            // 获取流程节点ID
            String subNodeId = subFlowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
            
            // 创建变更行动工单记录
            // 变更行动工单审批人：第一次新建时就是行动人
            String approverJson = JSON.toJSONString(Collections.singletonList(execUserEmail));
            // 初始状态：待确认行动方案
            ChangeSubFlowRecord subFlowRecord = ChangeSubFlowRecord.builder()
                    .changeRecordId(changeRecord.getId())
                    .subFlowId(subFlowId)
                    .subFlowNode(subNodeId)
                    .status(ChangeSubFlowStatusEnum.WAIT_CONFIRM_ACTION_PLAN.getStatus())
                    .approver(approverJson)
                    .changeExecUserEmail(execUserEmail)
                    .changeExecDepartment(execDepartment)
                    .changeCommander(changeRecord.getChangeCommander())
                    .changeConfirmResultTime(changeRecord.getChangeConfirmResultTime())
                    .createTime(DateUtils.getCurrentTime())
                    .updateTime(DateUtils.getCurrentTime())
                    .build();
            changeSubFlowRecordService.saveRecord(subFlowRecord);
            
            log.info("[createAndBindSubFlows] changeRecordId:{}, subFlowRecordId:{}, subFlowId:{}, nodeId:{}, execDepartment:{}, execUserEmail:{}, execCount:{}",
                    changeRecord.getId(), subFlowRecord.getId(), subFlowId, subNodeId, execDepartment, execUserEmail, groupRecords.size());
            
            // 将该分组下的所有行动项关联到变更行动工单
            for (ChangeExecRecord execRecord : groupRecords) {
                execRecord.setSubFlowRecordId(subFlowRecord.getId());
                execRecord.setSubFlowId(subFlowId);
                execRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeFlowExecService.update(execRecord);
            }
        }
    }

    /**
     * 提交到执行节点
     */
    private void submitToExecutionNode(String flowId, FlowDataDTO flowDataDTO, String uid, Map<String, Object> content, ChangeRecord changeRecord) {
        log.debug("[submitToExecutionNode] flowId:{}, uid:{}", flowId, uid);
        // 使用 paramMap 方式提交，type=1 表示审批通过/前进到下一节点
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("type", FlowTransitionType.TYPE_APPROVED);
        String nextNodeId = flowService.submitFlowWithParamMap(flowId, flowDataDTO, uid,
                ChangeFlowEnum.NEW_CHANGE_FLOW_START.getTopoId(), JSON.toJSONString(content), paramMap,
                FlowxOperationEnum.SUBMIT.getName(), "提交工单", changeRecord.getCreateTime());
        // 更新节点id，使用返回的节点ID
        changeRecord.setFlowNode(nextNodeId);
        changeRecord.setState(ChangeStatusEnum.WAIT_CONFIRM_CHANGE_PLAN.getStatus());
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
        changeFlowService.updateRecord(changeRecord);
    }

    /**
     * 发送创建流程邮件
     *
     * @param changeRecord 变更记录
     * @param changeCommander 变更负责人
     * @param changeType 变更类型
     * @param changeFlowCreateReq 创建请求
     */
    private void sendCreateFlowEmail(ChangeRecord changeRecord, String changeCommander, StringBuilder changeType, ChangeFlowCreateReq changeFlowCreateReq) {
        String uid = RequestLocalBean.getUid();
        
        // 如果发起人=变更负责人，已经在 executeCommonCreateFlow 中发送了邮件
        if (changeCommander.equals(uid)) {
            return;
        }
        
        // 如果发起人≠变更负责人，停留在变更申请提交节点，发送邮件
        HashMap<String, Object> param = new HashMap<>();
        param.put("changeId", changeRecord.getFlowId());
        param.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
        param.put("changeType", changeType.toString());
        param.put("flowUrl", changeRecord.getFlowId());
        String subjectParam = changeRecord.getFlowId().toString();
        
        // 发起变更,收件人：变更负责人
        List<String> receiver = Collections.singletonList(changeCommander);
        List<String> ccList = departmentLeaderBiz.getDepartmentLeaders(receiver);
        // 发起变更，抄送：变更负责人上一级主管、变更管理QM
        ccList.add(appConfig.getChangeManageQM());
        qcSendEmail(receiver, ccList, subjectParam, EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, param);
        
        // 如果是同舟端，发送供应商邮件
        if (changeFlowCreateReq.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())) {
            sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), subjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, param);
        } else if (ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
            // 如果是其他端发起但是是供应商变更，发送供应商邮件
            sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), subjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, param);
        }
    }


    /**
     * 严选QC端发送邮件
     *
     * @param subjectParam      主体参数
     * @param receiver          收件人
     * @param emailTemplateEnum 邮件模板
     * @param param             正文参数
     */
    public void qcSendEmail(List<String> receiver, List<String> ccList, String subjectParam, EmailTemplateEnum emailTemplateEnum,
                            Map<String, Object> param) {

        try {
            iEmailService.sendEmail(receiver, ccList, param, emailTemplateEnum,
                    subjectParam);
        } catch (Exception e) {
            log.error("[op:qcSendEmail] error,receiver:{}, ccList:{}, subjectParam:{}, e:{}", receiver, ccList, subjectParam,
                    e);
        }
    }

    private List<ChangeFile> buildChangeFileRecord(Long changeRecordId, List<ChangeFlowFile> files, Integer type) {
        return files.stream().map(f -> {
            ChangeFile changeFile = new ChangeFile();
            changeFile.setChangeRecordId(changeRecordId);
            changeFile.setFileType(type);
            changeFile.setFileName(f.getFileName());
            changeFile.setFileUrl(f.getFileUrl());
            changeFile.setCreateTime(DateUtils.getCurrentTime());
            changeFile.setUpdateTime(DateUtils.getCurrentTime());
            return changeFile;
        }).collect(Collectors.toList());
    }

    /**
     * 构建工单内容
     *
     * @param uid 创建人
     * @return 工单内容
     */
    private Map<String, Object> buildFlowContent(String uid) {
        Map<String, Object> content = new HashMap<>(10);
        content.put("createUserName", uid);
        content.put("createUser", uid);
        content.put("createTime", System.currentTimeMillis());
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.PASS.getValue());
        return content;
    }

    private FlowCreateReqDTO buildFlowCreateReqDTO(String topoId, String uid, String content, String operateResult,
        String name, String flowName) {
        FlowCreateReqDTO flowCreateReqDTO = new FlowCreateReqDTO();
        flowCreateReqDTO.setTopoId(topoId);
        flowCreateReqDTO.setUid(uid);
        flowCreateReqDTO.setUserName(name);
        flowCreateReqDTO.setOperateResult(operateResult);
        flowCreateReqDTO.setWorkOrderId(StringUtils.joinWith("-", topoId, UUID.randomUUID().toString()));
        flowCreateReqDTO.setContent(content);
        flowCreateReqDTO.setFlowName(flowName);
        return flowCreateReqDTO;
    }


    private ChangeRecord buildChangeRecord(String flowId, String nodeId, ChangeFlowCreateReq changeFlowCreateReq,
                                           String changeCommander, String uid) {
        ChangeRecord changeRecord = new ChangeRecord();
        changeRecord.setFlowId(Long.parseLong(flowId));
        changeRecord.setFlowNode(nodeId);
        changeRecord.setChangeSubject(changeFlowCreateReq.getChangeSubject());
        changeRecord.setParentChangeClassId(changeFlowCreateReq.getParentChangeClassId());
        changeRecord.setSonChangeClassId(changeFlowCreateReq.getSonChangeClassId());
        changeRecord.setChangeLevel(changeFlowCreateReq.getChangeLevel());
        changeRecord.setChangeCommander(changeCommander);
        changeRecord.setApprover(JSON.toJSONString(Collections.singletonList(changeCommander)));
        changeRecord.setChangeDepartment(changeFlowCreateReq.getChangeDepartment());
        List<ChangeExecConfigReq> changeExecProject = changeFlowCreateReq.getChangeExecProject();
        List<String> execDepartmentList = changeExecProject.stream().map(ChangeExecConfigReq::getChangeExecDepartment)
                .collect(Collectors.toList());
        changeRecord.setParticipateChangeExecDepartment(JSON.toJSONString(execDepartmentList));
        if (ChangeSubjectEnum.PRODUCT.getType().equals(changeFlowCreateReq.getChangeSubject())) {
            // 当变更类型是商品时有值
            changeRecord.setChangeItem(changeFlowCreateReq.getChangeItems());
            changeRecord.setChangeSku(changeFlowCreateReq.getChangeSkus());
            // 商品变更需要关联供应商
            changeRecord.setChangeSupplier(changeFlowCreateReq.getChangeSupplier());
        }
        if (ChangeSubjectEnum.SUPPLIER.getType().equals(changeFlowCreateReq.getChangeSubject())) {
            changeRecord.setChangeSupplier(changeFlowCreateReq.getChangeSupplier());
        }
        changeRecord.setChangeReason(changeFlowCreateReq.getChangeReason());
        changeRecord.setChangeContent(changeFlowCreateReq.getChangeContent());
        changeRecord.setChangeRiskDesc(changeFlowCreateReq.getChangeRiskDesc());
        changeRecord.setChangeProfit(changeFlowCreateReq.getChangeProfit());
        changeRecord.setChangeProfitDesc(changeFlowCreateReq.getChangeProfitDesc());
        changeRecord.setChangeConfirmResultTime(changeFlowCreateReq.getChangeConfirmResultTime());
        changeRecord.setState(ChangeStatusEnum.WAIT_SUBMIT_CHANGE_APPLY.getStatus());
        // 变更结论
        changeRecord.setCreateSource(changeFlowCreateReq.getCreateSource());
        changeRecord.setCreateSupplier(changeFlowCreateReq.getSupplier());
        changeRecord.setCreateTime(DateUtils.getCurrentTime());
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
        changeRecord.setCreator(uid);
        changeRecord.setChangeProfitAmount(changeFlowCreateReq.getChangeProfitAmount());
        // 批次说明，非必传，有的话需要填充
        if (StringUtils.isNotBlank(changeFlowCreateReq.getBatchDescription())) {
            changeRecord.setBatchDescription(changeFlowCreateReq.getBatchDescription());
        }
        return changeRecord;
    }

    public String submit(ChangeFlowSubmitReq changeFlowSubmitReq) {
        log.info("[submitFlow] changeFlowReq:{}", JSON.toJSONString(changeFlowSubmitReq));
        Long flowId = changeFlowSubmitReq.getFlowId();
        // 查询工单有效性
        ChangeRecord changeRecord = this.getFlowInfo(flowId);
        String uid = RequestLocalBean.getUid();
        // 历史工单负责人建单时固定，新流程有不同的节点处理人
        //String changeCommander = changeRecord.getChangeCommander();
        //if (!uid.equals(changeCommander)) {
        //    throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, ResponseCode.NO_AUTH.getMsg());
        //}
        // 获取工单详情（topo实际节点信息）
        FlowDataDTO flowDataDTO = flowService.flowDetail(flowId.toString());
        if (flowDataDTO == null) {
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "工单查询错误，不存在");
        }
        // 使用topo实际节点信息进行节点检查
        String actualNodeId = flowDataDTO.getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
        if (!actualNodeId.equals(changeFlowSubmitReq.getCurrentNodeId())) {
            throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, "工单已流转至其他节点");
        }

        return checkUpdateAndSubmit(flowId, flowDataDTO, uid, changeRecord, actualNodeId, changeFlowSubmitReq);
    }

    private String checkUpdateAndSubmit(Long flowId, FlowDataDTO flowDataDTO, String uid, ChangeRecord changeRecord,
                                        String currentNode, ChangeFlowSubmitReq changeFlowSubmitReq) {
        ChangeFlowEnum node = ChangeFlowEnum.getByNodeId(currentNode);
        Assert.notNull(node, "节点配置不存在");
        log.debug("[checkUpdateAndSubmit] flowId:{}, nodeEnum:{}, changeFlowSubmitReq:{}", flowId, node,
                JSON.toJSONString(changeFlowSubmitReq));
        // 工单流传
        Map<String, Object> content = buildFlowContent(FlowOperationTypeEnum.PASS);
        switch (node) {
            // 提交变更申请节点，可以修改数据，以新的数据为准
            case NEW_CHANGE_FLOW_START:
                // 检验是否需要资料
                ChangeConfig changeConfig = changeConfigService.getSonChange(changeFlowSubmitReq.getSonChangeClassId());
                Assert.notNull(changeConfig, "二级变更类型不存在");
                Integer needFile = changeConfig.getNeedFile();
                if (NeedFileEnum.NEED.getStatus().equals(needFile)) {
                    List<ChangeFlowFile> uploadFiles = changeFlowSubmitReq.getUploadFiles();
                    if (CollectionUtils.isEmpty(uploadFiles)) {
                        throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "必须上传资料");
                    }
                }
                // 变更行动项不可为空，最多20项
                List<ChangeExecConfigReq> changeExecProjectList = changeFlowSubmitReq.getChangeExecProjectList();
                if (CollectionUtils.isEmpty(changeExecProjectList)) {
                    throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "变更行动方案不能为空");
                }
                if (changeExecProjectList.size() > appConfig.getChangeExecLimit()) {
                    throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "变更行动方案配置数超限");
                }
                // 校验每个行动项的必填字段
                changeExecRecordBiz.validateChangeExecProjectRequiredFields(changeExecProjectList);
                // 校验变更行动方案中是否有重复的变更行动人
                validateDuplicateChangeExecUser(changeExecProjectList);
                changeRecord.setParentChangeClassId(changeFlowSubmitReq.getParentChangeClassId());
                changeRecord.setSonChangeClassId(changeFlowSubmitReq.getSonChangeClassId());
                List<String> execDepartmentList = changeExecProjectList.stream().map(ChangeExecConfigReq::getChangeExecDepartment)
                        .collect(Collectors.toList());
                changeRecord.setParticipateChangeExecDepartment(JSON.toJSONString(execDepartmentList));
                changeRecord.setChangeItem(changeFlowSubmitReq.getChangeItems());
                changeRecord.setChangeSku(changeFlowSubmitReq.getChangeSkus());
                changeRecord.setChangeSupplier(changeFlowSubmitReq.getChangeSupplier());
                changeRecord.setChangeReason(changeFlowSubmitReq.getChangeReason());
                changeRecord.setChangeContent(changeFlowSubmitReq.getChangeContent());
                changeRecord.setChangeRiskDesc(changeFlowSubmitReq.getChangeRiskDesc());
                changeRecord.setChangeChecking(changeFlowSubmitReq.getChangeChecking());
                changeRecord.setChangeProfit(changeFlowSubmitReq.getChangeProfit());
                changeRecord.setChangeProfitDesc(changeFlowSubmitReq.getChangeProfitDesc());
                changeRecord.setChangeConfirmResultTime(changeFlowSubmitReq.getChangeConfirmResultTime());
                changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeRecord.setChangeProfitAmount(changeFlowSubmitReq.getChangeProfitAmount());
                // 提交工单并更新节点
                submitToExecutionNode(flowId.toString(), flowDataDTO, uid, content, changeRecord);
                String submitNode = changeRecord.getFlowNode();
                // 更新行动执行方案，覆盖操作，先删除，后插入
                Integer changeExecCount = changeFlowExecService.deleteByChangeRecordId(changeRecord.getId());
                log.debug("[NEW_CHANGE_FLOW_START] delete id:{},  changeExecCount:{}", changeRecord.getId(), changeExecCount);
                // 保存变更行动方案记录
                List<ChangeExecRecord> changeExecRecords = changeExecRecordBiz.buildChangeExecRecord(changeRecord.getId(),
                        changeExecProjectList, null, null);
                changeExecRecords.forEach(exec -> changeFlowExecService.saveRecord(exec));
                // 更新附件，覆盖操作，先删除，后插入
                Integer fileCount = changeFileService.deleteByChangeRecordId(changeRecord.getId());
                log.debug("[NEW_CHANGE_FLOW_START] delete fileCount:{}", fileCount);
                List<ChangeFile> allFiles = new ArrayList<>();
                List<ChangeFlowFile> changeFiles = changeFlowSubmitReq.getChangeFiles();
                if (CollectionUtils.isNotEmpty(changeFiles)) {
                    allFiles.addAll(buildChangeFileRecord(changeRecord.getId(), changeFiles, FileTypeEnum.CHANGE.getType()));
                }
                // 保存附件，根据配置看是否必传
                List<ChangeFlowFile> uploadFiles = changeFlowSubmitReq.getUploadFiles();
                if (CollectionUtils.isNotEmpty(uploadFiles)) {
                    allFiles.addAll(buildChangeFileRecord(changeRecord.getId(), uploadFiles, FileTypeEnum.UPLOAD.getType()));
                }
                if (CollectionUtils.isNotEmpty(allFiles)) {
                    allFiles.forEach(file -> changeFileService.saveRecord(file));
                }
                log.debug("[NEW_CHANGE_FLOW_START] changeFiles:{}, uploadFiles:{}, allFiles:{}", JSON.toJSONString(changeFiles),
                    JSON.toJSONString(uploadFiles), JSON.toJSONString(allFiles));
                // 创建子流程并绑定（使用行动人作为创建人和审批人）
                String flowName = flowDataDTO.getFlowMeta() != null && StringUtils.isNotBlank(flowDataDTO.getFlowMeta().getFlowName()) 
                        ? flowDataDTO.getFlowMeta().getFlowName() : "";
                createAndBindSubFlows(changeRecord, flowName);
                buildAndSendEmail.buildAndSendEmailSubmit(changeRecord, changeExecRecords);
                return submitNode;
            case NEW_CHANGE_FLOW_CONFIRM_EXEC_PLAN:
                // 确认变更方案节点：由子单审批通过后自动触发，不应该由前端主动调用
                throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, 
                        "该节点不允许主动提交，需等待所有子单审批通过后自动流转");
            case NEW_CHANGE_FLOW_OWNER_APPROVE:
                // 部门负责人审批节点：设置下一个审批人为变更管理员
                String changeAdmin = appConfig.getChangeAdmin();
                if (StringUtils.isBlank(changeAdmin)) {
                    log.error("[NEW_CHANGE_FLOW_OWNER_APPROVE] 变更管理员未配置，flowId:{}", flowId);
                }
                Map<String, Object> ownerApproveParamMap = new HashMap<>();
                ownerApproveParamMap.put("type", FlowTransitionType.TYPE_APPROVED);
                return handleMainFlowApprovalSubmit(changeFlowSubmitReq, flowId, flowDataDTO, uid, changeRecord, node, content,
                        ownerApproveParamMap, changeAdmin, ChangeStatusEnum.WAIT_CHANGE_ADMIN_APPROVE);
                
            case NEW_CHANGE_FLOW_ADMIN_APPROVE:
                // 变更管理员审批节点：根据变更等级判断下一步
                // 重要变更（changeLevel=1）：需要质量部负责人审批，type=1
                // 一般变更（changeLevel=2）：跳过质量部审批，直接到执行变更方案，type=4
                Integer changeLevel = changeRecord.getChangeLevel();
                Map<String, Object> adminApproveParamMap = new HashMap<>();
                String nextApproverForAdmin;
                ChangeStatusEnum nextStatusForAdmin;
                
                if (ChangeLevelEnum.IMPORTANT.getType().equals(changeLevel)) {
                    // 重要变更：走质量部负责人审批
                    adminApproveParamMap.put("type", FlowTransitionType.TYPE_APPROVED);
                    nextApproverForAdmin = appConfig.getChangeQualityLeader();
                    if (StringUtils.isBlank(nextApproverForAdmin)) {
                        log.error("[NEW_CHANGE_FLOW_ADMIN_APPROVE] 质量部负责人未配置，flowId:{}", flowId);
                    }
                    nextStatusForAdmin = ChangeStatusEnum.WAIT_QUALITY_LEADER_APPROVE;
                    log.info("[NEW_CHANGE_FLOW_ADMIN_APPROVE] 重要变更，需要质量部负责人审批");
                } else {
                    // 一般变更：跳过质量部负责人审批，下一个审批人为变更负责人
                    adminApproveParamMap.put("type", FlowTransitionType.TYPE_SKIP_QUALITY_APPROVE);
                    nextApproverForAdmin = changeRecord.getChangeCommander();
                    nextStatusForAdmin = ChangeStatusEnum.WAIT_EXEC_CHANGE_PLAN;
                    log.info("[NEW_CHANGE_FLOW_ADMIN_APPROVE] 一般变更，跳过质量部负责人审批，下一个审批人为变更负责人：{}", nextApproverForAdmin);
                }
                return handleMainFlowApprovalSubmit(changeFlowSubmitReq, flowId, flowDataDTO, uid, changeRecord, node, content,
                        adminApproveParamMap, nextApproverForAdmin, nextStatusForAdmin);
                
            case NEW_CHANGE_FLOW_QUALITY_APPROVE:
                // 质量部负责人审批节点：审批通过后流转到执行变更方案，下一个审批人为变更负责人
                Map<String, Object> qualityApproveParamMap = new HashMap<>();
                qualityApproveParamMap.put("type", FlowTransitionType.TYPE_APPROVED);
                String nextApproverForQuality = changeRecord.getChangeCommander();
                log.info("[NEW_CHANGE_FLOW_QUALITY_APPROVE] 质量部负责人审批通过，下一个审批人为变更负责人：{}", nextApproverForQuality);
                return handleMainFlowApprovalSubmit(changeFlowSubmitReq, flowId, flowDataDTO, uid, changeRecord, node, content,
                        qualityApproveParamMap, nextApproverForQuality, ChangeStatusEnum.WAIT_EXEC_CHANGE_PLAN);
            case NEW_CHANGE_FLOW_EXE:
                // 执行变更方案节点：等待所有子单完成后自动流转，不允许主动提交
                throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST,
                        "该节点不允许主动提交，需等待所有行动工单执行变更方案后自动流转");
            case NEW_CHANGE_FLOW_CONFIRM:
                // 确认变更结果节点：由变更负责人进行确认，流转到结束节点
                // 校验操作人必须是变更负责人
                String changeCommander = changeRecord.getChangeCommander();
                if (!uid.equals(changeCommander)) {
                    throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, "只有变更负责人可以确认变更结果");
                }
                
                // 校验审批结果必须为确认（true）
                Boolean approved = changeFlowSubmitReq.getApproved();
                if (approved == null || !approved) {
                    throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "确认变更结果节点只支持确认操作，取消请使用取消接口");
                }
                
                // 构建提交内容
                Map<String, Object> confirmContent = buildFlowContent(FlowOperationTypeEnum.PASS);
                
                // 流转到结束节点
                String endNodeId = flowService.submitFlow(flowId.toString(), flowDataDTO, uid,
                        ChangeFlowEnum.NEW_CHANGE_FLOW.getTopoId(), JSON.toJSONString(confirmContent), true,
                        FlowxOperationEnum.SUBMIT.getName(), "确认变更结果", changeRecord.getCreateTime());
                
                // 更新主工单节点和状态为"已完结"
                changeRecord.setFlowNode(endNodeId);
                changeRecord.setState(ChangeStatusEnum.FINISHED.getStatus());
                changeRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeFlowService.updateRecord(changeRecord);
                
                log.info("[NEW_CHANGE_FLOW_CONFIRM] 主工单确认变更结果完成，flowId:{}, endNodeId:{}, state:{}",
                        flowId, endNodeId, ChangeStatusEnum.FINISHED.getStatus());
                
                // todo: 发送完结邮件通知（待业务确认邮件内容）
                // buildAndSendEmail.sendConfirmResultFinishEmail(changeRecord);
                
                return endNodeId;
                
            default:
                throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, "不可提交节点：" + currentNode);
        }
    }

    /**
     * 构建工单提交内容（包含 updateTime 和 FLOW_OPERATION_KEY）
     * 
     * @param operationType 操作类型（PASS 或 REFUSE），如果为 null 则不设置 FLOW_OPERATION_KEY
     * @return 工单提交内容 Map
     */
    private Map<String, Object> buildFlowContent(FlowOperationTypeEnum operationType) {
        Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
        content.put("updateTime", System.currentTimeMillis());
        if (operationType != null) {
            content.put(CommonConstants.FLOW_OPERATION_KEY, operationType.getValue());
        }
        return content;
    }

    private ChangeRecord getFlowInfo(Long flowId) {
        ChangeRecord changeRecord = changeFlowService.getByFlowId(flowId);
        if (changeRecord == null) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "工单id不存在");
        }
        return changeRecord;
    }

    private void checkNode(String recordNode, List<String> checkNode) {
        Optional<String> nodeOptional = checkNode.stream().filter(check -> check.equals(recordNode)).findAny();
        if (!nodeOptional.isPresent()) {
            throw ExceptionFactory.createBiz(ResponseCode.NODE_ERROR, "工单已流转至其他节点");
        }
    }

    public Boolean cancel(ChangeFlowCancelReq req) {
        log.info("[cancel] req:{}", JSON.toJSONString(req));
        Long flowId = req.getFlowId();
        // 查询工单有效性
        ChangeRecord changeRecord = getFlowInfo(flowId);
        String currentNodeId = changeRecord.getFlowNode();
        log.info("[cancel] nodeId:{}", currentNodeId);
        
        List<String> allowedCancelNodes = Arrays.asList(
                ChangeFlowEnum.NEW_CHANGE_FLOW_START.getNodeId(),
                ChangeFlowEnum.NEW_CHANGE_FLOW_OWNER_APPROVE.getNodeId(),
                ChangeFlowEnum.NEW_CHANGE_FLOW_ADMIN_APPROVE.getNodeId(),
                ChangeFlowEnum.NEW_CHANGE_FLOW_QUALITY_APPROVE.getNodeId(),
                ChangeFlowEnum.NEW_CHANGE_FLOW_CONFIRM.getNodeId()
        );
        this.checkNode(currentNodeId, allowedCancelNodes);
        
        String uid = RequestLocalBean.getUid();
        String changeCommander = changeRecord.getChangeCommander();
        if (!uid.equals(changeCommander)) {
            throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, ResponseCode.NO_AUTH.getMsg());
        }
        // 获取工单详情
        FlowDataDTO flowDataDTO = flowService.flowDetail(flowId.toString());
        if (flowDataDTO == null) {
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "工单查询错误，不存在");
        }
        // 工单流转：使用 paramMap 方式，type=2 表示回退
        Map<String, Object> content = buildFlowContent(FlowOperationTypeEnum.REFUSE);
        Map<String, Object> cancelParamMap = new HashMap<>();
        cancelParamMap.put("type", FlowTransitionType.TYPE_REJECTED);
        String nextNodeId = flowService.submitFlowWithParamMap(String.valueOf(flowId), flowDataDTO, uid,
                ChangeFlowEnum.NEW_CHANGE_FLOW.getTopoId(), JSON.toJSONString(content), cancelParamMap,
                FlowxOperationEnum.CANCEL.getName(), "取消变更", changeRecord.getCreateTime());
        log.info("[cancel] flowId:{}, nextNodeId:{}", flowId, nextNodeId);
        // 填充更新数据
        changeRecord.setFlowNode(nextNodeId);
        changeRecord.setState(ChangeStatusEnum.CANCEL.getStatus());
        changeRecord.setCancelReason(req.getCancelReason());
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());
        
        // 取消所有行动工单，并获取所有子工单列表（用于后续获取行动人邮箱）
        List<ChangeSubFlowRecord> allSubFlows = cancelAllSubFlows(changeRecord.getId(), req.getCancelReason(), uid);
        
        // 立即更新主工单状态，确保状态正确，不被后续邮件服务阻塞
        Boolean updateResult = changeFlowService.updateRecord(changeRecord);
        log.info("[cancel] 主工单状态已更新，flowId:{}, state:{}, updateResult:{}", flowId, 
                ChangeStatusEnum.CANCEL.getStatus(), updateResult);
        
        try {
            Map<String, Object> cancelMap = new HashMap<>();
            cancelMap.put("changeId", changeRecord.getFlowId());
            cancelMap.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
            cancelMap.put("changeContent", changeRecord.getChangeContent());
            IusUserInfoRsp user = iusService.queryUserInfo(changeRecord.getChangeCommander());
            cancelMap.put("changeCommander", user == null ? changeRecord.getChangeCommander() : user.getName());
            cancelMap.put("cancelReason", changeRecord.getCancelReason());
            
            // 从行动工单获取行动人邮箱（去重）
            List<String> userEmailList = extractExecUserEmailsFromSubFlows(allSubFlows);
            
            String cancelSubjectParam = changeRecord.getFlowId().toString();
            List<String> receiver = new ArrayList<>(Collections.singletonList(changeCommander));
            receiver.addAll(userEmailList);
            // 取消变更，抄送人：变更负责人+变更行动人上一级主管、变更管理QM（cuiyixian@corp.netease.com）
            userEmailList.addAll(receiver);
            List<String> ccList = departmentLeaderBiz.getDepartmentLeaders(userEmailList);
            ccList.add(appConfig.getChangeManageQM());
            // 取消变更，收件人：变更发起人（供应商邮箱号或严选发起人）、变更负责人、变更行动人
            String creator = changeRecord.getCreator();
            if (!creator.equals(changeCommander)) {
                receiver.add(creator);
            }
            qcSendEmail(receiver, ccList, cancelSubjectParam, EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_CANCEL, cancelMap);
            // 如果是供应商，再次发送供应商邮件
            if (changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())) {
                sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), cancelSubjectParam,
                        EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_CANCEL, cancelMap);
            }
            // 如果是其他端发起但是是供应商变更，再次发送供应商邮件
            if (!changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())
                    && ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
                sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), cancelSubjectParam,
                        EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_CANCEL, cancelMap);
            }
        } catch (Exception e) {
            // 邮件发送失败不影响主流程，只记录日志
            log.error("[cancel] 发送取消邮件失败，flowId:{}, error:{}", flowId, e.getMessage(), e);
        }
        
        return updateResult;
    }

    /**
     * 取消所有行动工单
     * 
     * @param changeRecordId 主工单ID
     * @param cancelReason 取消原因
     * @param uid 操作人
     * @return 所有子工单列表（包括已取消和已完成的）
     */
    private List<ChangeSubFlowRecord> cancelAllSubFlows(Long changeRecordId, String cancelReason, String uid) {
        List<ChangeSubFlowRecord> allSubFlows = changeSubFlowRecordService.getByChangeRecordId(changeRecordId);
        if (CollectionUtils.isEmpty(allSubFlows)) {
            log.info("[cancelAllSubFlows] 主工单下没有行动工单，changeRecordId:{}", changeRecordId);
            return allSubFlows;
        }
        
        for (ChangeSubFlowRecord subFlowRecord : allSubFlows) {
            // 跳过已经取消的子工单（已完成的子工单也需要取消）
            Integer subFlowStatus = subFlowRecord.getStatus();
            if (ChangeSubFlowStatusEnum.CANCELLED.getStatus().equals(subFlowStatus)) {
                log.info("[cancelAllSubFlows] 子工单已取消，跳过, subFlowId:{}, status:{}", 
                        subFlowRecord.getSubFlowId(), subFlowStatus);
                continue;
            }
            
            try {
                // 查询子工单流程详情
                FlowDataDTO subFlowDataDTO = flowService.flowDetail(subFlowRecord.getSubFlowId());
                if (subFlowDataDTO == null) {
                    log.warn("[cancelAllSubFlows] 子工单流程查询失败，跳过, subFlowId:{}", subFlowRecord.getSubFlowId());
                    continue;
                }
                
                // 构建提交内容
                Map<String, Object> subFlowContent = buildFlowContent(FlowOperationTypeEnum.REFUSE);
                // 流转子工单到结束节点
                String subFlowCancelNodeId = flowService.submitFlow(subFlowRecord.getSubFlowId(), subFlowDataDTO, uid,
                    ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), JSON.toJSONString(subFlowContent), false,
                    FlowxOperationEnum.CANCEL.getName(), "主工单取消，取消行动工单", subFlowRecord.getCreateTime());
                
                // 更新子工单状态为已取消
                subFlowRecord.setSubFlowNode(subFlowCancelNodeId);
                subFlowRecord.setStatus(ChangeSubFlowStatusEnum.CANCELLED.getStatus());
                subFlowRecord.setCancelReason("主工单取消：" + cancelReason);
                subFlowRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeSubFlowRecordService.update(subFlowRecord);
                
                log.info("[cancelAllSubFlows] 子工单已取消，subFlowId:{}, cancelNodeId:{}", 
                        subFlowRecord.getSubFlowId(), subFlowCancelNodeId);
            } catch (Exception e) {
                log.error("[cancelAllSubFlows] 取消子工单失败，subFlowId:{}, error:{}", 
                        subFlowRecord.getSubFlowId(), e.getMessage(), e);
                // 继续取消下一个子工单，不中断整个流程
            }
        }
        
        return allSubFlows;
    }

    /**
     * 从行动工单列表中提取行动人邮箱（去重）
     * 
     * @param allSubFlows 所有行动工单列表
     * @return 行动人邮箱列表（已去重）
     */
    private List<String> extractExecUserEmailsFromSubFlows(List<ChangeSubFlowRecord> allSubFlows) {
        Set<String> execUserEmailSet = new HashSet<>();
        if (CollectionUtils.isNotEmpty(allSubFlows)) {
            for (ChangeSubFlowRecord subFlow : allSubFlows) {
                String execUserEmail = subFlow.getChangeExecUserEmail();
                if (StringUtils.isNotBlank(execUserEmail)) {
                    execUserEmailSet.add(execUserEmail);
                }
            }
        }
        return new ArrayList<>(execUserEmailSet);
    }

    public BasicChangeFlowVO quote(Long flowId, String supplier, Integer createSource) {
        // 获取工单详情
        ChangeRecord changeRecord = this.getFlowInfo(flowId);
        // 同舟端需校验只能引用同一供应商下的工单
        if (CreateSourceEnum.TONG_ZHOU.getType().equals(createSource)) {
            Assert.isTrue(changeRecord.getCreateSupplier().equals(supplier), "不能引用非本供应商工单");
        }
        String changeSupplier = changeRecord.getChangeSupplier();
        String changeSupplierName = "";
        if (StringUtils.isNotBlank(changeSupplier)) {
            List<SupplierSimpleRsp> supplierSimple = supplierService.getSupplierName(changeSupplier);
            if (CollectionUtils.isNotEmpty(supplierSimple)) {
                SupplierSimpleRsp supplierInfo = supplierSimple.get(0);
                changeSupplierName = supplierInfo.getSupplierName();
            }
        }
        // 获取附件
        List<ChangeFlowFile> changeFileList = changeFileService.getChangeFileList(changeRecord.getId());
        return BasicChangeFlowVO.builder().parentChangeClassId(changeRecord.getParentChangeClassId())
            .sonChangeClassId(changeRecord.getSonChangeClassId()).changeSubject(changeRecord.getChangeSubject())
            .changeItem(changeRecord.getChangeItem()).changeSupplier(changeSupplier)
            .changeSupplierName(changeSupplierName).changeReason(changeRecord.getChangeReason())
            .changeContent(changeRecord.getChangeContent()).changeRiskDesc(changeRecord.getChangeRiskDesc())
            .changeProfit(changeRecord.getChangeProfit()).changeProfitDesc(changeRecord.getChangeProfitDesc())
            .files(changeFileList).build();
    }

    public ChangeFlowVO detail(Long flowId) {
        ChangeRecord changeRecord = changeFlowService.getByFlowId(flowId);
        if (changeRecord == null) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "变更工单id不存在");
        }
        log.debug("[detail] detail:{}", JSON.toJSONString(changeRecord));
        ChangeFlowVO changeFlowVO = new ChangeFlowVO();
        BeanUtils.copyProperties(changeRecord, changeFlowVO);
        // 设置变更类型，一级变更类型>二级变更类型
        String changeType = this.buildChangeType(changeRecord.getParentChangeClassId(), changeRecord.getSonChangeClassId());
        changeFlowVO.setChangeType(changeType);
        // 设置供应商名称
        String changeSupplierName = this.buildChangeSupplierName(changeRecord.getChangeSupplier());
        changeFlowVO.setChangeSupplierName(changeSupplierName);
        // 获取附件
        List<ChangeFlowFile> changeFileList = changeFileService.getChangeFileList(changeRecord.getId());
        changeFlowVO.setFiles(changeFileList);
        // 组装商品信息
        this.buildItemBasicInfoList(changeRecord, changeFlowVO);
        changeFlowVO.setChangeState(changeRecord.getState());
        changeFlowVO.setChangeStateDesc(ChangeStatusEnum.getDescByStatus(changeRecord.getState()));
        changeFlowVO.setChangeCreator(changeRecord.getCreator());
        List<ChangeFlowExecVO> changeFlowExecRecord = changeFlowExecService.getChangeFlowExecRecord(changeRecord.getId());
        // 填充行动人姓名
        this.fillChangeExecUserName(changeFlowExecRecord);
        changeFlowVO.setChangeExecProjectList(changeFlowExecRecord);
        changeFlowVO.setTopoId(ChangeFlowEnum.NEW_CHANGE_FLOW.getTopoId());
        // 设置负责人和创建人视图
        this.buildCommanderAndCreatorView(changeRecord, changeFlowVO);
        return changeFlowVO;
    }

    private void buildItemBasicInfoList(ChangeRecord changeRecord, ChangeFlowVO changeFlowVO) {
        String itemJsonStr = changeRecord.getChangeItem();
        if (StringUtils.isBlank(itemJsonStr)) {
            changeFlowVO.setChangeItems(null);
            changeFlowVO.setItemBasicInfoList(null);
            return;
        }
        
        String changeSkuJson = changeRecord.getChangeSku();
        List<ItemVO> itemList = JSON.parseArray(itemJsonStr, ItemVO.class);
        List<ItemSkuVO> itemSkuVOS = new ArrayList<>();
        if (StringUtils.isNotBlank(changeSkuJson)) {
            itemSkuVOS.addAll(JSON.parseArray(changeSkuJson, ItemSkuVO.class));
        }
        itemList.forEach(i -> {
            Optional<ItemSkuVO> anySkuInfo = itemSkuVOS.stream().filter(s -> s.getItemId().equals(i.getItemId()))
                .findAny();
            if (anySkuInfo.isPresent()) {
                i.setSkuIds(anySkuInfo.get().getSkuId());
            }
        });
        changeFlowVO.setChangeItems(itemList);
        List<Long> itemIds = itemList.stream().map(ItemVO::getItemId).collect(Collectors.toList());
        // 批量查询spu信息
        List<SpuTO> spuTOS = itemService.batchQuerySpuInfo(BatchQuerySpuInfoParam.builder().ids(itemIds)
            .commonProps(new ArrayList<>()).spuProps(Arrays.asList("itemSetupType", "businessForm")).build());
        log.debug("[detail] spuTOS:{}", JSON.toJSONString(spuTOS));
        // 批量查询物理类目
        Map<Long, List<SimplePhyCateGoryResultCo>> categoryChain = itemService
            .queryBySpuIds(CommonIdsParamQuery.builder().ids(itemIds).build());
        // 查询商品对应负责人信息
        Map<Long, ChangeGoodsPrincipalPO> longChangeGoodsPrincipalPOMap = interiorChangeConfigService
            .queryGoodsPrincipalInfo(itemIds);
        List<ItemBasicInfoVO> itemBasicInfoVOS = itemList.stream().map(itemVO -> {
            ItemBasicInfoVO itemBasicInfoVO = new ItemBasicInfoVO();
            Long itemId = itemVO.getItemId();
            itemBasicInfoVO.setItemId(itemId);
            itemBasicInfoVO.setItemName(itemVO.getItemName());
            List<QcCategoryVO> qcCategoryList = qcService.getQcCategoryListByItemId(itemId);
            if (CollectionUtils.isNotEmpty(qcCategoryList)) {
                String collect = qcCategoryList.stream().map(QcCategoryVO::getCategoryName)
                    .collect(Collectors.joining("/"));
                itemBasicInfoVO.setQcCategory(collect);
            } else {
                itemBasicInfoVO.setQcCategory("/");
            }
            List<SimplePhyCateGoryResultCo> simplePhyCateGoryResultCos = categoryChain.get(itemId);
            log.debug("[detail] itemId:{}, simplePhyCateGoryResultCos:{}", itemId,
                JSON.toJSONString(simplePhyCateGoryResultCos));
            if (CollectionUtils.isEmpty(simplePhyCateGoryResultCos)) {
                itemBasicInfoVO.setPhyCategory(null);
            } else {
                List<CategoryInfoVO> categoryInfoVOS = simplePhyCateGoryResultCos.stream().map(phy -> {
                    CategoryInfoVO categoryInfoVO = new CategoryInfoVO();
                    categoryInfoVO.setId(phy.getId());
                    categoryInfoVO.setName(phy.getName());
                    return categoryInfoVO;
                }).collect(Collectors.toList());
                itemBasicInfoVO.setPhyCategory(categoryInfoVOS);
            }
            Optional<SpuTO> optionalSpuTO = spuTOS.stream().filter(spu -> spu.getId() == itemId).findAny();
            log.debug("[detail] itemId:{}, optionalSpuTO:{}", itemId, JSON.toJSONString(optionalSpuTO));
            if (!optionalSpuTO.isPresent()) {
                itemBasicInfoVO.setItemSetupType(null);
                itemBasicInfoVO.setBusinessForm(null);
                itemBasicInfoVO.setStatus(null);
            } else {
                SpuTO spuTO = optionalSpuTO.get();
                Map<String, String> propertyMap = spuTO.getPropertyMap();
                itemBasicInfoVO.setItemSetupType(Integer.valueOf(propertyMap.get("itemSetupType")));
                String businessForm = propertyMap.get("businessForm");
                int business = StringUtils.isBlank(businessForm) ? 0 : Integer.parseInt(businessForm);
                itemBasicInfoVO.setBusinessForm(business);
                itemBasicInfoVO.setStatus(spuTO.getStatus());
            }
            ChangeGoodsPrincipalPO changeGoodsPrincipalPO = longChangeGoodsPrincipalPOMap.get(itemId);
            log.debug("[detail] itemId:{}, changeGoodsPrincipalPO:{}", itemId,
                JSON.toJSONString(changeGoodsPrincipalPO));
            itemBasicInfoVO.setChangeGoodsPrincipal(changeGoodsPrincipalPO);
            Optional<ItemVO> anyHasSku = itemList.stream().filter(i -> i.getItemId().equals(itemId)).findAny();
            if (anyHasSku.isPresent()) {
                itemBasicInfoVO.setSkuIds(anyHasSku.get().getSkuIds());
            }
            return itemBasicInfoVO;
        }).collect(Collectors.toList());
        changeFlowVO.setItemBasicInfoList(itemBasicInfoVOS);
    }

    /**
     * 构建变更类型字符串
     * 
     * @param parentChangeClassId 一级变更类型ID
     * @param sonChangeClassId 二级变更类型ID
     * @return 变更类型字符串，格式：一级类型>二级类型
     */
    private String buildChangeType(Long parentChangeClassId, Long sonChangeClassId) {
        ChangeType parentChangeType = changeTypeService.getChangeTypeById(parentChangeClassId);
        if (parentChangeType == null) {
            throw ExceptionFactory.createBiz(ResponseCode.CHANGE_TYPE_NOT_EXIST, "变更类型不存在");
        }
        ChangeType sonChangeType = changeTypeService.getChangeTypeById(sonChangeClassId);
        if (sonChangeType == null) {
            throw ExceptionFactory.createBiz(ResponseCode.CHANGE_TYPE_NOT_EXIST, "变更类型配置不存在");
        }
        return parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName();
    }

    /**
     * 构建供应商名称
     * 
     * @param changeSupplier 供应商ID
     * @return 供应商名称，如果查询不到则返回空字符串
     */
    private String buildChangeSupplierName(String changeSupplier) {
        if (StringUtils.isBlank(changeSupplier)) {
            return "";
        }
        List<SupplierSimpleRsp> supplierSimple = supplierService.getSupplierName(changeSupplier);
        if (CollectionUtils.isEmpty(supplierSimple)) {
            return "";
        }
        return supplierSimple.get(0).getSupplierName();
    }

    /**
     * 构建负责人和创建人视图
     * 
     * @param changeRecord 变更记录
     * @param changeFlowVO 变更工单视图对象
     */
    private void buildCommanderAndCreatorView(ChangeRecord changeRecord, ChangeFlowVO changeFlowVO) {
        try {
            Set<String> commanderList = new HashSet<>();
            commanderList.add(changeRecord.getChangeCommander());
            commanderList.add(changeRecord.getCreator());
            
            IusDepartmentReq iusDepartmentReq = new IusDepartmentReq();
            iusDepartmentReq.setUids(new ArrayList<>(commanderList));
            iusDepartmentReq.setIcac(true);
            HashMap<String, List<SecondaryDepartments>> orgMap = iusService.queryDepartment(iusDepartmentReq);
            
            List<SecondaryDepartments> commander = orgMap.get(changeRecord.getChangeCommander());
            List<SecondaryDepartments> creator = orgMap.get(changeRecord.getCreator());
            
            // 设置负责人视图：姓名(三级部门名称)
            if (!CollectionUtils.isEmpty(commander)) {
                Optional<SecondaryDepartments> anyOrg = commander.stream()
                    .filter(o -> o.getOrgPosLevel() == 97)
                    .findAny();
                String orgName = anyOrg.isPresent() ? anyOrg.get().getOrgPosName() : "无三级部门";
                changeFlowVO.setChangeCommanderView(commander.get(0).getUserName() + "(" + orgName + ")");
            }
            
            // 设置创建人视图
            if (CreateSourceEnum.TONG_ZHOU.getType().equals(changeRecord.getCreateSource())) {
                // 同舟端发起变更，发起人展示供应商id
                changeFlowVO.setChangeCreatorView(changeRecord.getCreateSupplier());
            } else if (!CollectionUtils.isEmpty(creator)) {
                // 非同舟端展示人名及三级部门名称
                Optional<SecondaryDepartments> anyOrg = creator.stream()
                    .filter(o -> o.getOrgPosLevel() == 97)
                    .findAny();
                String orgName = anyOrg.isPresent() ? anyOrg.get().getOrgPosName() : "无三级部门";
                changeFlowVO.setChangeCreatorView(creator.get(0).getUserName() + "(" + orgName + ")");
            }
        } catch (Exception ex) {
            log.error("add org has ex", ex);
        }
    }

    /**
     * 填充变更行动人姓名
     * 
     * @param changeFlowExecRecord 变更行动方案列表
     */
    private void fillChangeExecUserName(List<ChangeFlowExecVO> changeFlowExecRecord) {
        if (CollectionUtils.isEmpty(changeFlowExecRecord)) {
            return;
        }
        
        try {
            Set<String> userEmail = changeFlowExecRecord.stream()
                .map(ChangeFlowExecVO::getChangeExecUserEmail)
                .collect(Collectors.toSet());
            
            // 批量查询用户信息
            AjaxResponse<List<IusUserInfoRsp>> userListInfo = iusRpcService.queryUserListInfo(
                UserQueryDTO.builder().uids(new ArrayList<>(userEmail)).build());
            List<IusUserInfoRsp> data = userListInfo.getData();
            
            // 填充用户姓名
            changeFlowExecRecord.forEach(i -> {
                Optional<IusUserInfoRsp> anyExeUser = data.stream()
                    .filter(u -> u.getUid().equals(i.getChangeExecUserEmail()))
                    .findAny();
                
                if (anyExeUser.isPresent()) {
                    i.setChangeExecUserName(anyExeUser.get().getName());
                } else {
                    // 批量查询没查到的人名，通过全量查询单独再查一次
                    IusUserInfoRsp user = iusService.queryUserInfo(i.getChangeExecUserEmail());
                    if (user != null && StringUtils.isNotBlank(user.getName())) {
                        i.setChangeExecUserName(user.getName());
                    }
                }
            });
        } catch (Exception ex) {
            log.error("query user info has ex", ex);
        }
    }

    public ChangeFlowListVO query(Integer page, Integer pageSize, ChangeFlowListQueryReq changeFlowListQueryReq) {
        log.info("[query] page:{}, pageSize:{}, changeFlowListQueryReq:{}", page, pageSize,
                JSON.toJSONString(changeFlowListQueryReq));
        
        if (StringUtils.isNotBlank(changeFlowListQueryReq.getChangeExecUser())) {
            List<Long> recordIds = changeFlowExecService.queryByExecUser(changeFlowListQueryReq.getChangeExecUser());
            changeFlowListQueryReq.setChangeRecordIds(recordIds);
            log.info("recordIds: {}", recordIds);
            if (CollectionUtils.isEmpty(recordIds)) {
                ChangeFlowListVO changeFlowListVO = new ChangeFlowListVO();
                PageVO pageVO = PageUtils.buildPageVo(0L, pageSize, page);
                changeFlowListVO.setPageVo(pageVO);
                changeFlowListVO.setChangeFlowList(new ArrayList<>());
                return changeFlowListVO;
            }
        }

        // 数据可见范围：仅查看我跟进的工单（默认true）
        if (changeFlowListQueryReq.getOnlyMyFollowed()) {
            String currentUser = RequestLocalBean.getUid();
            if (StringUtils.isNotBlank(currentUser)) {
                changeFlowListQueryReq.setCurrentUser(currentUser);
                log.info("[query] 数据可见范围：仅查看我跟进的工单, currentUser:{}", currentUser);
            }
        }

        //进行分页
        PageHelper.startPage(page, pageSize);
        PageInfo<ChangeRecord> changeRecordPageInfo = new PageInfo<>(
                changeRecordMapper.selectByCondition(changeFlowListQueryReq));
        List<ChangeRecord> changeRecords = changeRecordPageInfo.getList();
        List<ChangeFlowVO> list = new ArrayList<>();
        
        // 获取当前用户（用于设置 userRole）
        String currentUser = RequestLocalBean.getUid();
        
        // 处理数据
        if (CollectionUtils.isNotEmpty(changeRecords)) {
            // 批量查询所有变更行动工单，构建 Map<changeRecordId, List<subFlowId>>，提升性能
            final Map<Long, List<String>> subFlowIdsMap;
            // 批量查询所有行动工单，构建 Map<changeRecordId, List<ChangeSubFlowRecord>>，用于后续过滤和超期判断
            final Map<Long, List<ChangeSubFlowRecord>> subFlowRecordsMap;
            List<Long> changeRecordIds = changeRecords.stream().map(ChangeRecord::getId).collect(Collectors.toList());
            List<ChangeSubFlowRecord> subFlowRecords = changeSubFlowRecordService.getByChangeRecordIds(changeRecordIds);
            if (CollectionUtils.isNotEmpty(subFlowRecords)) {
                subFlowIdsMap = subFlowRecords.stream().filter(record -> StringUtils.isNotBlank(record.getSubFlowId()))
                    .collect(Collectors.groupingBy(ChangeSubFlowRecord::getChangeRecordId,
                        Collectors.mapping(ChangeSubFlowRecord::getSubFlowId, Collectors.toList())));
                // 按 changeRecordId 分组，不过滤状态，后续在内部方法中根据业务需求过滤
                subFlowRecordsMap = subFlowRecords.stream()
                    .collect(Collectors.groupingBy(ChangeSubFlowRecord::getChangeRecordId));
            } else {
                subFlowIdsMap = new HashMap<>();
                subFlowRecordsMap = new HashMap<>();
            }
            list = changeRecords.stream().map(c -> {
                ChangeFlowVO changeFlowVO = new ChangeFlowVO();
                changeFlowVO.setId(c.getId());
                changeFlowVO.setFlowId(c.getFlowId());
                changeFlowVO.setChangeSubject(c.getChangeSubject());
                ChangeType parentChangeType = changeTypeService.getChangeTypeById(c.getParentChangeClassId());
                ChangeType sonChangeType = changeTypeService.getChangeTypeById(c.getSonChangeClassId());
                changeFlowVO.setChangeType(parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
                changeFlowVO.setChangeDepartment(c.getChangeDepartment());
                changeFlowVO.setChangeContent(c.getChangeContent());
                changeFlowVO.setChangeCommander(c.getChangeCommander());
                String itemJsonStr = c.getChangeItem();
                if (StringUtils.isNotBlank(itemJsonStr)) {
                    List<ItemVO> itemVOS = JSON.parseArray(itemJsonStr, ItemVO.class);
                    String changeSkuStr = c.getChangeSku();
                    if (StringUtils.isNotBlank(changeSkuStr)) {
                        List<ItemSkuVO> itemSkuVOS = JSON.parseArray(changeSkuStr, ItemSkuVO.class);
                        itemVOS.forEach(i -> {
                            Optional<ItemSkuVO> anySkuInfo = itemSkuVOS.stream()
                                    .filter(s -> s.getItemId().equals(i.getItemId())).findAny();
                            if (anySkuInfo.isPresent()) {
                                i.setSkuIds(anySkuInfo.get().getSkuId());
                            }
                        });
                    }
                    changeFlowVO.setChangeItems(itemVOS);
                }
                changeFlowVO.setChangeSupplier(c.getChangeSupplier());
                Integer changeState = c.getState() == 3 ? ChangeStatusEnum.IN.getStatus() : c.getState();
                changeFlowVO.setChangeState(changeState);
                changeFlowVO.setChangeStateDesc(ChangeStatusEnum.getDescByStatus(changeState));
                // 同舟端展示供应商id
                changeFlowVO.setChangeCreator(
                    CreateSourceEnum.TONG_ZHOU.getType().equals(c.getCreateSource()) ? c.getCreateSupplier()
                        : c.getCreator());
                String participateChangeExecDepartment = c.getParticipateChangeExecDepartment();
                if (StringUtils.isNotBlank(participateChangeExecDepartment)) {
                    changeFlowVO.setExecDepartmentStrList(JSON.parseArray(participateChangeExecDepartment, String.class));
                    HashSet<String> validDep = new HashSet<>(changeFlowVO.getExecDepartmentStrList());
                    changeFlowVO.setExecDepartmentStrList(new ArrayList<>(validDep));
                }
                changeFlowVO.setCreateTime(c.getCreateTime());
                changeFlowVO.setChangeConfirmResultTime(c.getChangeConfirmResultTime());
                changeFlowVO.setCancelReason(c.getCancelReason());
                changeFlowVO.setRemark(c.getRemark());
                
                // 从 Map 中获取变更行动工单 flowId 列表（已批量查询，提升性能）
                List<String> subFlowIds = subFlowIdsMap.getOrDefault(c.getId(), new ArrayList<>());
                changeFlowVO.setSubFlowIds(subFlowIds);
                
                // 设置主单归属关系
                if (StringUtils.isNotBlank(currentUser)) {
                    boolean isCreator = currentUser.equals(c.getCreator());
                    boolean isChangeCommander = currentUser.equals(c.getChangeCommander());
                    boolean isApprover = false;
                    String approverJson = c.getApprover();
                    if (StringUtils.isNotBlank(approverJson)) {
                        isApprover = approverJson.contains(currentUser);
                    }
                    ChangeFlowOwnershipVO flowOwnership = ChangeFlowOwnershipVO.builder()
                            .isCreator(isCreator)
                            .isChangeCommander(isChangeCommander)
                            .isApprover(isApprover)
                            .build();
                    changeFlowVO.setFlowOwnership(flowOwnership);
                }
                
                // 设置变更完结时间和是否超期标识
                buildChangeEndTime(c, changeFlowVO, subFlowRecordsMap);
                
                return changeFlowVO;
            }).collect(Collectors.toList());
            
            try {
                Set<String> commanderList = list.stream().map(ChangeFlowVO::getChangeCommander).collect(Collectors.toSet());
                Set<String> creatorList = list.stream().map(ChangeFlowVO::getChangeCreator).collect(Collectors.toSet());
                commanderList.addAll(creatorList);
                IusDepartmentReq iusDepartmentReq = new IusDepartmentReq();
                iusDepartmentReq.setUids(new ArrayList<>(commanderList));
                iusDepartmentReq.setIcac(true);
                HashMap<String, List<SecondaryDepartments>> orgMap = iusService.queryDepartment(iusDepartmentReq);
                //AjaxResponse<List<IusUserInfoRsp>> userListInfo = iusRpcService.queryUserListInfo(UserQueryDTO.builder().uids(new ArrayList<>(commanderList)).build());
                log.debug("query user: {} info: {}", commanderList, JSON.toJSONString(orgMap));
                //List<IusUserInfoRsp> data = userListInfo.getData();
                list.forEach(i -> {
                    List<SecondaryDepartments> commander = orgMap.get(i.getChangeCommander());
                    List<SecondaryDepartments> creator = orgMap.get(i.getChangeCreator());
                    //Optional<IusUserInfoRsp> commander = data.stream().filter(u -> u.getUid().equals(i.getChangeCommander())).findAny();
                    //Optional<IusUserInfoRsp> creator = data.stream().filter(u -> u.getUid().equals(i.getChangeCreator())).findAny();
                    if (!CollectionUtils.isEmpty(commander)) {
                        Optional<SecondaryDepartments> anyOrg = commander.stream().filter(o -> o.getOrgPosLevel() == 97).findAny();
                        String orgName = "无三级部门";
                        if (anyOrg.isPresent()) {
                            orgName = anyOrg.get().getOrgPosName();
                        }
                        i.setChangeCommander(commander.get(0).getUserName() + "(" + orgName + ")");
                    }
                    // 供应商查询不到数据，不会覆盖，展示供应商id
                    if (!CollectionUtils.isEmpty(creator)) {
                        Optional<SecondaryDepartments> anyOrg = creator.stream().filter(o -> o.getOrgPosLevel() == 97).findAny();
                        String orgName = "无三级部门";
                        if (anyOrg.isPresent()) {
                            orgName = anyOrg.get().getOrgPosName();
                        }
                        i.setChangeCreator(creator.get(0).getUserName() + "(" + orgName + ")");
                    }
                });
            } catch (Exception ex) {
                log.error("add org has ex", ex);
            }
        }
        
        PageVO pageVO = PageUtils.buildPageVo(changeRecordPageInfo.getTotal(), pageSize, page);
        ChangeFlowListVO changeFlowListVO = new ChangeFlowListVO();
        changeFlowListVO.setPageVo(pageVO);
        changeFlowListVO.setChangeFlowList(list);
        return changeFlowListVO;
    }


    public UserBaseContainerDTO getOperator(InterfaceInputDTO interfaceInput) {
        log.info("[getOperator] interfaceInput:{}", JSON.toJSONString(interfaceInput));
        // 根据工单flowId获取审批人，创建工单时设置，唯一
        String flowId = interfaceInput.getPublicFieldDTO().getFlowMeta().getFlowId();
        String nodeId = interfaceInput.getPublicFieldDTO().getFlowMeta().getCurrNodeDataList().get(0).getNodeId();
        log.info("[getOperator] flowId:{}, nodeId:{}", flowId, nodeId);
        ChangeRecord changeRecord;
        changeRecord = changeFlowService.getByFlowId(Long.valueOf(flowId));
        log.info("[getOperator] first time get changeCommander:{}", changeRecord);
        // 创建工单后落库，此时审批人还未落库，工单平台无法查询到审批人，方法休眠500ms
        if (changeRecord == null) {
            try {
                Thread.sleep(500);
                changeRecord = changeFlowService.getByFlowId(Long.valueOf(flowId));
                log.info("[getOperator] second time get changeCommander:{}", changeRecord);
            } catch (Exception e) {
                log.info("[getOperator] flowId:{}, nodeId:{}, e={}", flowId, nodeId, e);
            }
        }
        String changeCommander = changeRecord.getChangeCommander();
        UserReachDTO userReachDTO = new UserReachDTO();
        // 查询邮箱对应的名字
        IusUserInfoRsp user = iusService.queryUserInfo(changeRecord.getChangeCommander());
        // 流程拓扑图，如果没查询到名字，以邮箱兜底
        userReachDTO.setUserName(
            user == null || StringUtils.isBlank(user.getName()) ? changeRecord.getChangeCommander() : user.getName());
        userReachDTO.setUid(changeCommander);
        List<UserReachDTO> totalUserList = Collections.singletonList(userReachDTO);
        UserBaseContainerDTO userBaseContainer = new UserBaseContainerDTO();
        userBaseContainer.setUserList(totalUserList);
        return userBaseContainer;
    }

    @Deprecated
    public void export(ChangeFlowListQueryReq changeFlowListQueryReq, HttpServletResponse response) {
        List<ChangeRecord> changeRecords = changeRecordMapper.selectByCondition(changeFlowListQueryReq);
        if (CollectionUtils.isEmpty(changeRecords)) {
            throw ExceptionFactory.createBiz(ResponseCode.EMPTY, "无导出数据");
        }
        List<ChangeFlowExcelDTO> changeFlowExcelDTOList = changeRecords.stream().map(record -> {
            ChangeFlowExcelDTO changeFlowExcelDTO = new ChangeFlowExcelDTO();
            /*changeFlowExcelDTO.setId(record.getId());
            changeFlowExcelDTO.setChangeSubject(ChangeSubjectEnum.getChangeSubjectEnum(record.getChangeSubject()).getDesc());
            ChangeType parentChangeType = changeTypeService.getChangeTypeById(record.getParentChangeClassId());
            ChangeType sonChangeType = changeTypeService.getChangeTypeById(record.getSonChangeClassId());
            changeFlowExcelDTO.setChangeType(parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
            changeFlowExcelDTO.setChangeDepartment(record.getChangeDepartment());
            changeFlowExcelDTO.setChangeContent(record.getChangeContent());
            changeFlowExcelDTO.setChangeCommander(record.getChangeCommander());
            String changeItemJsonStr = record.getChangeItem();
            if (StringUtils.isNotBlank(changeItemJsonStr)) {
                List<ItemVO> itemVOS = JSON.parseArray(changeItemJsonStr, ItemVO.class);
                changeFlowExcelDTO.setChangeItems(itemVOS.stream().map(item -> item.getItemId() + item.getItemName())
                        .collect(Collectors.joining(",")));
            }
            if (StringUtils.isNotBlank(record.getChangeSupplier())) {
                changeFlowExcelDTO.setSupplier(record.getChangeSupplier());
            }
            changeFlowExcelDTO.setChangeState(record.getState());
            // todo:变更发起人组织查询：姓名（组织架构三级部门）/供应商ID/system+发起时间
            changeFlowExcelDTO.setChangeCreator(record.getCreator());
            changeFlowExcelDTO.setExecDepartmentStrList(record.getParticipateChangeExecDepartment());
            changeFlowExcelDTO.setCreateTime(record.getCreateTime());*/
            return changeFlowExcelDTO;
        }).collect(Collectors.toList());
        Workbook workbook = ExcelExportUtil.exportExcel(new ExportParams(), ChangeFlowExcelDTO.class, changeFlowExcelDTOList);
        downLoadExcel("变更管理列表", response, workbook);
    }

    private void downLoadExcel(String fileName, HttpServletResponse response, Workbook workbook) {
        try {
            response.setCharacterEncoding("UTF-8");
            response.setHeader("content-Type", "application/vnd.ms-excel");
            response.setHeader("Content-Disposition",
                    "attachment;filename=" + URLEncoder.encode(fileName, "UTF-8") + ".xls");
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            workbook.write(baos);
            response.setHeader("Content-Length", String.valueOf(baos.size()));
            OutputStream out = response.getOutputStream();
            out.write(baos.toByteArray());

        } catch (IOException ex) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST);
        }
    }

    public void deliver(@Valid ChangeFlowDeliverReq req) {
        log.info("[deliver] req:{}", JSON.toJSONString(req));
        Long flowId = req.getFlowId();
        // 查询工单有效性
        ChangeRecord changeRecord = getFlowInfo(flowId);
        log.info("[deliver] nodeId:{}", changeRecord.getFlowNode());
        // 检查工单节点
        List<String> nodeList = Arrays.asList(ChangeFlowEnum.CHANGE_FLOW_SUBMIT.getNodeId(),
                ChangeFlowEnum.CHANGE_FLOW_CONFIRM.getNodeId());
        this.checkNode(changeRecord.getFlowNode(), nodeList);
        String uid = RequestLocalBean.getUid();
        String name = RequestLocalBean.getName();
        String changeCommander = changeRecord.getChangeCommander();
        if (!uid.equals(changeCommander)) {
            throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, ResponseCode.NO_AUTH.getMsg());
        }
        if (changeCommander.equals(req.getDeliverUser())) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "不能转交给自己！");
        }
        // 工单审核人转交
        String deliverUser = req.getDeliverUser();
        UserBaseDTO userBaseDTO = new UserBaseDTO();
        userBaseDTO.setUserName(deliverUser);
        userBaseDTO.setUid(deliverUser);
        String remark = req.getRemark();
        Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
        content.put("updateTime", System.currentTimeMillis());
        flowService.updateApprovers(ChangeFlowEnum.CHANGE_FLOW_EXE.getTopoId(), flowId.toString(),
                changeRecord.getFlowNode(), Collections.singletonList(userBaseDTO), uid,
                name, remark, JSON.toJSONString(content));
        // 更新工单负责人
        changeRecord.setChangeCommander(deliverUser);
        changeRecord.setRemark(remark);
        changeRecord.setUpdateTime(DateUtils.getCurrentTime());

        Map<String, Object> deliverMap = new HashMap<>();
        deliverMap.put("changeId", changeRecord.getFlowId());
        deliverMap.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
        deliverMap.put("changeContent", changeRecord.getChangeContent());
        // 转交人名称
        IusUserInfoRsp oldUser = iusService.queryUserInfo(changeCommander);
        deliverMap.put("changeCommander", oldUser == null ? changeCommander : oldUser.getName());
        deliverMap.put("changeCommanderEmail", changeCommander);
        // 被转交人名称
        IusUserInfoRsp newUser = iusService.queryUserInfo(deliverUser);
        deliverMap.put("restsChangeCommander", newUser == null ? deliverUser : newUser.getName());
        deliverMap.put("restsChangeCommanderEmail", deliverUser);
        String deliverSubjectParam = changeRecord.getFlowId().toString();
        // 转交，收件人：变更转交人（工单接收人）、变更负责人（工单转交人）
        List<String> receiver = Arrays.asList(changeCommander, deliverUser);
        // 转交，抄送人：变更发起人+原变更负责人上一级主管、变更转交人（工单接收人）上一级主管+变更行动项负责人
        List<String> ccList = departmentLeaderBiz.getDepartmentLeaders(receiver);
        ccList.add(changeRecord.getCreator());
        List<ChangeFlowExecVO> execRecord = changeFlowExecService
                .getChangeFlowExecRecord(changeRecord.getId());
        List<String> userEmailList = execRecord.stream().map(ChangeFlowExecVO::getChangeExecUserEmail)
                .collect(Collectors.toList());
        ccList.addAll(userEmailList);
        qcSendEmail(receiver, ccList, deliverSubjectParam, EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_FORWARD, deliverMap);
        // 如果是供应商，再次发送供应商邮件
        if (changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())) {
            sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), deliverSubjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_FORWARD, deliverMap);
        }
        // 如果是其他端发起但是是供应商变更，再次发送供应商邮件
        if (!changeRecord.getCreateSource().equals(CreateSourceEnum.TONG_ZHOU.getType())
                && ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
            sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), deliverSubjectParam,
                    EmailTemplateEnum.YX_QC_CHANGE_SUBMIT_FORWARD, deliverMap);
        }
        changeFlowService.updateRecord(changeRecord);
    }

    /**
     * 获取工单负责人
     *
     * @param changeFlowCreateReq 创建请求
     * @param uid 当前用户ID
     * @param parentChangeClassId 一级变更类型ID
     * @param sonChangeClassId 二级变更类型ID
     * @param itemIds 商品ID列表
     * @return 工单负责人邮箱
     */
    private String getChangeCommander(ChangeFlowCreateReq changeFlowCreateReq, String uid,
                                     Long parentChangeClassId, Long sonChangeClassId, List<Long> itemIds) {
        // 非同舟端：负责人=发起人
        if (!CreateSourceEnum.TONG_ZHOU.getType().equals(changeFlowCreateReq.getCreateSource())) {
            return uid;
        }
        
        // 同舟端：通过RPC查询负责人信息
        GoodsResponseRpc goodsResponseRpc = null;
        try {
            goodsResponseRpc = interiorChangeConfigService
                    .queryCommanderInfo(ChangeCommanderPO.builder()
                            .parentChangeClassId(parentChangeClassId)
                            .sonChangeClassId(sonChangeClassId)
                            .changeSupplierId(changeFlowCreateReq.getChangeSupplier())
                            .goodsInfos(itemIds)
                            .flowCreator(uid)
                            .createSource(changeFlowCreateReq.getCreateSource())
                            .supplier(changeFlowCreateReq.getSupplier())
                            .build(),
                            changeFlowCreateReq.getChangeCommander());
            return goodsResponseRpc != null ? goodsResponseRpc.getEmail() : null;
        } catch (Exception e) {
            log.error("[getChangeCommander] queryCommanderInfo error, goodsResponseRpc:{}, e:{}",
                    JSON.toJSONString(goodsResponseRpc), e);
            return null;
        }
    }

    /**
     * 构建变更类型
     *
     * @param changeFlowCreateReq 创建请求
     * @return 变更类型
     */
    private StringBuilder buildChangeType(ChangeFlowCreateReq changeFlowCreateReq) {
        Long parentChangeClassId = changeFlowCreateReq.getParentChangeClassId();
        Long sonChangeClassId = changeFlowCreateReq.getSonChangeClassId();
        
        // 变更类型， 一级变更类型>二级变更类型
        ChangeConfigPo changeConfigPo = changeConfigService.queryInfoPo(parentChangeClassId, sonChangeClassId);
        StringBuilder changeType = new StringBuilder();
        if (changeConfigPo != null && changeConfigPo.getChangeTypes() != null && changeConfigPo.getChangeTypes().size() >= 2) {
            try {
                changeType.append(changeConfigPo.getChangeTypes().get(0).getTypeName());
                changeType.append(changeConfigPo.getChangeTypes().get(1).getTypeName());
            } catch (Exception e) {
                log.error("queryInfoPo error:{}", JSON.toJSONString(changeConfigPo), e);
            }
        }
        return changeType;
    }

    /**
     * 根据变更主体类型构建工单名称后缀（辅助方法，由 buildFlowName 调用）
     *
     * @param changeFlowCreateReq 创建请求
     * @return 工单名称后缀
     */
    private String buildFlowNameBySubject(ChangeFlowCreateReq changeFlowCreateReq) {
        ChangeSubjectEnum subjectEnum = ChangeSubjectEnum.getByType(changeFlowCreateReq.getChangeSubject());
        switch (Objects.requireNonNull(subjectEnum)) {
            case PRODUCT:
                // 商品变更：解析商品ID列表，查询SPU信息，拼接商品ID和名称
                List<Long> itemIds = parseItemIds(changeFlowCreateReq.getChangeItems());
                List<SpuTO> spuTOS = itemService.batchQuerySpuInfo(BatchQuerySpuInfoParam.builder().ids(itemIds)
                    .commonProps(new ArrayList<>()).spuProps(new ArrayList<>()).build());
                log.debug("[buildFlowNameBySubject] spuTOS:{}", JSON.toJSONString(spuTOS));
                return spuTOS.stream().map(spu -> "(" + spu.getId() + spu.getName() + ")")
                    .collect(Collectors.joining(","));

            case SUPPLIER:
                // 供应商变更：查询供应商信息，拼接供应商ID和名称
                List<SupplierSimpleRsp> supplierSimple = supplierService.getSupplierName(changeFlowCreateReq.getChangeSupplier());
                if (CollectionUtils.isNotEmpty(supplierSimple)) {
                    SupplierSimpleRsp supplier = supplierSimple.get(0);
                    return supplier.getSupplierId() + supplier.getSupplierName();
                }
                return "";

            case OTHER:
            default:
                return "";
        }
    }

    /**
     * 查询topo实际节点信息
     * @param flowId 工单ID
     * @return topo实际节点信息
     */
    public FlowDataDTO getFlowNodeInfo(Long flowId) {
        return flowService.flowDetail(flowId.toString());
    }

    /**
     * 校验变更行动方案中是否有重复的变更行动人
     * 
     * 规则：
     * 1. 供应商关联角色/商品关联角色：如果 changeExecUserType 和 changeExecUser 都相同，则重复
     * 2. 固定人：如果 changeExecUserType 相同且 changeExecUser（邮箱）相同，则重复
     * 3. 发起人：如果 changeExecUserType 相同，则重复
     * 
     * @param changeExecProjectList 变更行动方案列表
     */
    private void validateDuplicateChangeExecUser(List<ChangeExecConfigReq> changeExecProjectList) {
        if (CollectionUtils.isEmpty(changeExecProjectList)) {
            return;
        }
        
        // 用于记录邮箱对应的部门：邮箱 -> 部门
        Map<String, String> userEmailToDepartment = new HashMap<>();
        
        for (ChangeExecConfigReq execConfig : changeExecProjectList) {
            // 校验部门+人员，需要邮箱和部门都不为空
            if (StringUtils.isBlank(execConfig.getChangeExecUserEmail()) 
                    || StringUtils.isBlank(execConfig.getChangeExecDepartment())) {
                continue;
            }
            
            String changeExecUserEmail = execConfig.getChangeExecUserEmail();
            String changeExecDepartment = execConfig.getChangeExecDepartment();
            
            // 检查该邮箱是否已经在其他部门出现过
            String existingDepartment = userEmailToDepartment.get(changeExecUserEmail);
            if (existingDepartment != null && !existingDepartment.equals(changeExecDepartment)) {
                String errorMsg = String.format("变更行动方案中，同一人员不能出现在多个部门：%s 已在部门 %s 中，不能同时出现在部门 %s", 
                    changeExecUserEmail, existingDepartment, changeExecDepartment);
                log.warn("[validateDuplicateChangeExecUser] 检测到同一人员出现在多个部门: email={}, existingDept={}, newDept={}", 
                    changeExecUserEmail, existingDepartment, changeExecDepartment);
                throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, errorMsg);
            }
            
            // 记录邮箱和部门的对应关系（如果已存在相同部门，覆盖不影响）
            userEmailToDepartment.put(changeExecUserEmail, changeExecDepartment);
            log.debug("[validateDuplicateChangeExecUser] 添加变更行动人: email={}, department={}", 
                changeExecUserEmail, changeExecDepartment);
        }
    }

    /**
     * 所有子单审批通过后，流转主工单到部门负责人审批节点
     * 从 NEW_CHANGE_FLOW_CONFIRM_EXEC_PLAN 流转到 NEW_CHANGE_FLOW_OWNER_APPROVE
     * 
     * @param changeRecordId 主工单ID
     */
    /**
     * 统一处理主单审批节点的流转逻辑（包含审批通过和审批不通过）
     * 
     * @param changeFlowSubmitReq 提交请求
     * @param flowId 工单ID
     * @param flowDataDTO 工单详情
     * @param uid 当前用户
     * @param changeRecord 主工单记录
     * @param node 当前节点
     * @param content 提交内容
     * @param approvedParamMap 审批通过时的 paramMap
     * @param nextApprover 下一个审批人邮箱（可选，为空则不设置）
     * @param nextStatus 下一个状态（可选，为空则不设置）
     * @return 下一节点ID
     */
    private String handleMainFlowApprovalSubmit(ChangeFlowSubmitReq changeFlowSubmitReq, Long flowId, FlowDataDTO flowDataDTO, 
                                         String uid, ChangeRecord changeRecord, ChangeFlowEnum node, 
                                         Map<String, Object> content, Map<String, Object> approvedParamMap,
                                         String nextApprover, ChangeStatusEnum nextStatus) {
        Boolean approved = changeFlowSubmitReq.getApproved();
        if (approved == null) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "审批结果不能为空");
        }
        
        String nextNodeId;
        
        if (approved) {
            // 审批通过：流转到下一节点
            log.info("[{}] 审批通过，flowId:{}", node.getName(), flowId);
            
            nextNodeId = flowService.submitFlowWithParamMap(flowId.toString(), flowDataDTO, uid,
                    ChangeFlowEnum.NEW_CHANGE_FLOW.getTopoId(), JSON.toJSONString(content), approvedParamMap,
                    FlowxOperationEnum.SUBMIT.getName(), "审批通过", changeRecord.getCreateTime());
            
            // 更新主工单记录
            changeRecord.setFlowNode(nextNodeId);
            changeRecord.setUpdateTime(DateUtils.getCurrentTime());
            
            // 设置下一个审批人（如果提供）
            if (StringUtils.isNotBlank(nextApprover)) {
                changeRecord.setApprover(JSON.toJSONString(Collections.singletonList(nextApprover)));
                log.info("[{}] 下一个审批人：{}", node.getName(), nextApprover);
            }
            
            // 设置下一个状态（如果提供）
            if (nextStatus != null) {
                changeRecord.setState(nextStatus.getStatus());
            }
            
            changeFlowService.updateRecord(changeRecord);
            
            log.info("[checkUpdateAndSubmit] 主工单流转完成，flowId:{}, 审批节点:{}, 新节点:{}, 审批结果:通过", 
                    flowId, node.getName(), nextNodeId);
        } else {
            // 审批不通过：回退到确认变更方案节点，并重置指定的子单
            log.warn("[{}] 审批不通过，flowId:{}", node.getName(), flowId);
            
            // 验证必须选择要驳回的子单
            List<String> rejectSubFlowIds = changeFlowSubmitReq.getRejectSubFlowIds();
            if (CollectionUtils.isEmpty(rejectSubFlowIds)) {
                throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "审批不通过时必须选择要驳回的行动工单");
            }
            
            // 验证驳回的子单归属于当前主工单，并返回要驳回的子单列表
            List<ChangeSubFlowRecord> subFlowsToReject = validateAndFilterSubFlowsToReject(changeRecord.getId(), rejectSubFlowIds);
            
            String rejectReason = changeFlowSubmitReq.getRejectReason();
            if (StringUtils.isBlank(rejectReason)) {
                rejectReason = "主单审批不通过";
            }
            
            Map<String, Object> rejectParamMap = new HashMap<>();
            rejectParamMap.put("type", FlowTransitionType.TYPE_BACK_TO_CONFIRM_PLAN);
            
            nextNodeId = flowService.submitFlowWithParamMap(flowId.toString(), flowDataDTO, uid,
                    ChangeFlowEnum.NEW_CHANGE_FLOW.getTopoId(), JSON.toJSONString(content), rejectParamMap,
                    FlowxOperationEnum.SUBMIT.getName(), "审批不通过", changeRecord.getCreateTime());
            
            // 更新主工单记录（回到确认变更方案节点）
            changeRecord.setFlowNode(nextNodeId);
            changeRecord.setState(ChangeStatusEnum.WAIT_CONFIRM_CHANGE_PLAN.getStatus());
            changeRecord.setUpdateTime(DateUtils.getCurrentTime());
            changeFlowService.updateRecord(changeRecord);
            
            // 重置被驳回的子单（已过滤，直接重置并流转工单）
            int resetCount = resetRejectedSubFlows(subFlowsToReject, node.getName(), rejectReason);
            
            // 发送审批不通过邮件通知
            // sendMainFlowRejectEmail(changeRecord, node.getName(), rejectReason, resetCount);
            
            log.info("[checkUpdateAndSubmit] 主工单流转完成，flowId:{}, 审批节点:{}, 新节点:确认变更方案, 审批结果:不通过，已驳回子单数:{}", 
                    flowId, node.getName(), resetCount);
        }
        
        return nextNodeId;
    }

    /**
     * 验证子单归属于主工单（查询一次，返回所有子单供后续复用）
     * 
     * @param changeRecordId 主工单ID
     * @param subFlowIds 子单ID列表
     * @return 主工单下所有子单列表
     */
    /**
     * 验证并过滤出要驳回的子单
     * 
     * @param changeRecordId 主工单ID
     * @param rejectSubFlowIds 要驳回的子单ID列表
     * @return 过滤后的要驳回的子单列表
     */
    private List<ChangeSubFlowRecord> validateAndFilterSubFlowsToReject(Long changeRecordId, List<String> rejectSubFlowIds) {
        // 查询该主工单下所有子单（只查询一次）
        List<ChangeSubFlowRecord> allSubFlows = changeSubFlowRecordService.getByChangeRecordId(changeRecordId);
        if (CollectionUtils.isEmpty(allSubFlows)) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, 
                    "主工单下没有子单，无法驳回");
        }
        
        // 构建子单ID到记录的映射
        Map<String, ChangeSubFlowRecord> subFlowMap = allSubFlows.stream()
                .collect(Collectors.toMap(ChangeSubFlowRecord::getSubFlowId, subFlow -> subFlow));
        
        // 检查要驳回的子单是否都存在
        List<String> notFoundSubFlowIds = rejectSubFlowIds.stream()
                .filter(subFlowId -> !subFlowMap.containsKey(subFlowId))
                .collect(Collectors.toList());
        
        if (CollectionUtils.isNotEmpty(notFoundSubFlowIds)) {
            String notFoundIdsStr = String.join(",", notFoundSubFlowIds);
            log.warn("[validateAndFilterSubFlowsToReject] 部分行动工单不存在或不归属于当前主工单, changeRecordId:{}, notFoundSubFlowIds:{}", 
                    changeRecordId, notFoundIdsStr);
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, 
                    "部分行动工单不存在或不归属于当前主工单：" + notFoundIdsStr);
        }
        
        // 过滤出要驳回的子单
        List<ChangeSubFlowRecord> subFlowsToReject = rejectSubFlowIds.stream()
                .map(subFlowMap::get)
                .collect(Collectors.toList());
        
        log.info("[validateAndFilterSubFlowsToReject] 子单归属校验通过, changeRecordId:{}, 总子单数:{}, 待驳回数:{}", 
                changeRecordId, allSubFlows.size(), subFlowsToReject.size());
        
        return subFlowsToReject;
    }
    
    /**
     * 发送主单审批不通过邮件通知
     * 
     * @param changeRecord 主工单记录
     * @param rejectNodeName 审批不通过的节点名称
     * @param rejectReason 拒绝原因
     * @param resetCount 重置的子单数量
     */
    private void sendMainFlowRejectEmail(ChangeRecord changeRecord, String rejectNodeName, String rejectReason, int resetCount) {
        try {
            // 构建邮件参数
            Map<String, Object> emailParam = new HashMap<>();
            emailParam.put("changeId", changeRecord.getFlowId());
            emailParam.put("changeSubject", ChangeSubjectEnum.getChangeSubjectEnum(changeRecord.getChangeSubject()).getDesc());
            
            // 构建变更类型
            ChangeType parentChangeType = changeTypeService.getChangeTypeById(changeRecord.getParentChangeClassId());
            ChangeType sonChangeType = changeTypeService.getChangeTypeById(changeRecord.getSonChangeClassId());
            if (parentChangeType != null && sonChangeType != null) {
                emailParam.put("changeType", parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
            }
            
            emailParam.put("changeContent", changeRecord.getChangeContent());
            emailParam.put("rejectNodeName", rejectNodeName);
            emailParam.put("rejectReason", rejectReason);
            emailParam.put("resetSubFlowCount", resetCount);
            emailParam.put("flowUrl", changeRecord.getFlowId());
            
            String subjectParam = changeRecord.getFlowId().toString();
            
            // 收件人：变更负责人、变更发起人
            List<String> receiver = new ArrayList<>();
            String changeCommander = changeRecord.getChangeCommander();
            receiver.add(changeCommander);
            
            String creator = changeRecord.getCreator();
            if (!creator.equals(changeCommander)) {
                receiver.add(creator);
            }
            
            // 抄送人：变更负责人上一级主管、变更管理QM、所有行动人
            List<String> ccList = new ArrayList<>();
            ccList.addAll(departmentLeaderBiz.getDepartmentLeaders(Collections.singletonList(changeCommander)));
            ccList.add(appConfig.getChangeManageQM());
            
            // 添加所有行动人到抄送列表
            List<ChangeFlowExecVO> execRecord = changeFlowExecService.getChangeFlowExecRecord(changeRecord.getId());
            if (CollectionUtils.isNotEmpty(execRecord)) {
                List<String> execUserEmails = execRecord.stream()
                        .map(ChangeFlowExecVO::getChangeExecUserEmail)
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.toList());
                ccList.addAll(execUserEmails);
            }
            
            // 发送邮件（使用"发起变更"模板，因为需要重新提交）
            qcSendEmail(receiver, ccList, subjectParam, EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, emailParam);
            
            // 如果是供应商相关，发送供应商邮件
            if (CreateSourceEnum.TONG_ZHOU.getType().equals(changeRecord.getCreateSource())) {
                sendSupplierEmail.sendSupplierEmail(changeRecord.getCreateSupplier(), subjectParam,
                        EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, emailParam);
            } else if (ChangeSubjectEnum.SUPPLIER.getType().equals(changeRecord.getChangeSubject())) {
                sendSupplierEmail.sendSupplierEmail(changeRecord.getChangeSupplier(), subjectParam,
                        EmailTemplateEnum.YX_QC_CHANGE_RELEASE_FLOW, emailParam);
            }
            
            log.info("[sendMainFlowRejectEmail] 审批不通过邮件已发送，flowId:{}, rejectNode:{}", 
                    changeRecord.getFlowId(), rejectNodeName);
        } catch (Exception e) {
            log.error("[sendMainFlowRejectEmail] 发送审批不通过邮件失败，flowId:{}, error:{}", 
                    changeRecord.getFlowId(), e.getMessage(), e);
        }
    }
    
    /**
     * 重置被驳回的子单（主单审批不通过时调用）
     * 将子单状态和节点重置到初始状态，保留流程记录
     * 
     * @param subFlowsToReject 要驳回的子单列表（已过滤，直接处理）
     * @param rejectNodeName 驳回节点名称（用于记录是哪个审批节点驳回的）
     * @param rejectReason 驳回原因
     * @return 实际重置的子单数量
     */
    private int resetRejectedSubFlows(List<ChangeSubFlowRecord> subFlowsToReject, String rejectNodeName, String rejectReason) {
        if (CollectionUtils.isEmpty(subFlowsToReject)) {
            log.info("[resetRejectedSubFlows] 未指定要驳回的子单，驳回节点:{}", rejectNodeName);
            return 0;
        }
        
        log.info("[resetRejectedSubFlows] 开始重置被驳回的子单，驳回节点:{}, 待重置数量:{}, 驳回原因:{}", 
                rejectNodeName, subFlowsToReject.size(), rejectReason);
        
        int resetCount = 0;
        int skippedCount = 0;
        
        // 直接遍历要驳回的子单并重置（需要流转工单流程）
        for (ChangeSubFlowRecord subFlowRecord : subFlowsToReject) {
            String subFlowId = subFlowRecord.getSubFlowId();
            String oldNodeId = subFlowRecord.getSubFlowNode();
            
            try {
                Integer currentStatus = subFlowRecord.getStatus();
                
                // 跳过已取消的子单
                if (ChangeSubFlowStatusEnum.CANCELLED.getStatus().equals(currentStatus)) {
                    log.info("[resetRejectedSubFlows] 子单已取消，跳过, subFlowId:{}, status:{}", subFlowId, currentStatus);
                    skippedCount++;
                    continue;
                }
                
                // 查询子单流程详情
                FlowDataDTO subFlowDataDTO = flowService.flowDetail(subFlowId);
                if (subFlowDataDTO == null) {
                    log.warn("[resetRejectedSubFlows] 子单流程查询失败，跳过, subFlowId:{}", subFlowId);
                    skippedCount++;
                    continue;
                }
                
                // 构建提交内容
                String currentUid = RequestLocalBean.getUid();
                Map<String, Object> subFlowContent = buildFlowContent(currentUid);
                
                // 使用 paramMap 流转子单工单（type=2 回到初始节点）
                Map<String, Object> paramMap = new HashMap<>();
                paramMap.put("type", FlowTransitionType.TYPE_REJECTED);
                
                String operateDesc = String.format("变更工单，%s审批不通过，驳回行动工单", rejectNodeName);
                String startNodeId = flowService.submitFlowWithParamMap(subFlowId, subFlowDataDTO, 
                        currentUid, ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), 
                        JSON.toJSONString(subFlowContent), paramMap,
                        FlowxOperationEnum.SUBMIT.getName(), operateDesc, subFlowRecord.getCreateTime());
                
                // 重置子单记录到初始状态
                subFlowRecord.setSubFlowNode(startNodeId);
                subFlowRecord.setStatus(ChangeSubFlowStatusEnum.WAIT_CONFIRM_ACTION_PLAN.getStatus());
                // 重置审批人为行动人自己
                subFlowRecord.setApprover(JSON.toJSONString(Collections.singletonList(subFlowRecord.getChangeExecUserEmail())));
                // 清空审批相关字段，保存驳回原因
                subFlowRecord.setRejectReason(rejectReason);
                subFlowRecord.setChangeResult(null);
                subFlowRecord.setRemark(null);
                subFlowRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeSubFlowRecordService.update(subFlowRecord);
                
                resetCount++;
                log.info("[resetRejectedSubFlows] 子单已驳回并重置到初始状态, subFlowId:{}, 驳回节点:{}, 原节点:{}, 新节点:{}, status:{}", 
                        subFlowId, rejectNodeName, oldNodeId, startNodeId, ChangeSubFlowStatusEnum.WAIT_CONFIRM_ACTION_PLAN.getStatus());
                
            } catch (Exception e) {
                log.error("[resetRejectedSubFlows] 重置子单失败, subFlowId:{}, error:{}", subFlowId, e.getMessage(), e);
                // 继续重置下一个子单，不中断整个流程
            }
        }
        
        log.info("[resetRejectedSubFlows] 子单重置完成，驳回节点:{}, 待重置数:{}, 已重置:{}, 跳过:{}", 
                rejectNodeName, subFlowsToReject.size(), resetCount, skippedCount);
        
        return resetCount;
    }

    /**
     * 构建变更完结时间和是否超期标识
     * 
     * 时间展示逻辑：
     * - 未完成的工单：展示预期完成时间（changeConfirmResultTime）
     * - 已完成工单（END）：展示实际完成时间（updateTime）
     * - 取消的工单（CANCEL）：展示取消时间（updateTime）
     * 
     * 超期判断逻辑（仅针对非完结/取消的变更单）：
     * - 变更单的 changeConfirmResultTime 超过两个工作日未处理
     * - 或者工单下属的执行项里的 changeExecFinishTime 超过两个工作日未处理（仅判断未完结或未取消的行动工单对应的行动项）
     * 
     * @param changeRecord 变更记录
     * @param changeFlowVO 变更工单视图对象
     * @param subFlowRecordsMap 所有行动工单Map，key为changeRecordId，value为该主单下所有行动工单列表
     */
    private void buildChangeEndTime(ChangeRecord changeRecord, ChangeFlowVO changeFlowVO, Map<Long, List<ChangeSubFlowRecord>> subFlowRecordsMap) {
        Integer state = changeRecord.getState();
        String flowNode = changeRecord.getFlowNode();
        Long changeConfirmResultTime = changeRecord.getChangeConfirmResultTime();
        Long updateTime = changeRecord.getUpdateTime();
        
        // 先根据节点判断：新老topo的结束节点都是9999
        if (ChangeFlowEnum.END.getNodeId().equals(flowNode) || ChangeFlowEnum.NEW_END.getNodeId().equals(flowNode)) {
            // 已结束的工单：展示实际完成时间（updateTime）
            changeFlowVO.setChangeEndTime(updateTime);
            changeFlowVO.setIsOverdue(false);
            return;
        }
        
        // 如果节点不是结束节点，再根据状态判断（兼容新老状态：完结和取消合并为一个集合）
        Set<Integer> finishedOrCancelledStatusSet = new HashSet<>(Arrays.asList(
                ChangeStatusEnum.FINISHED.getStatus(),
                ChangeStatusEnum.END.getStatus(),
                ChangeStatusEnum.CANCELLED.getStatus(),
                ChangeStatusEnum.CANCEL.getStatus()
        ));
        
        if (finishedOrCancelledStatusSet.contains(state)) {
            // 已完成或取消的工单：展示实际完成/取消时间（updateTime）
            changeFlowVO.setChangeEndTime(updateTime);
            changeFlowVO.setIsOverdue(false);
        } else {
            // 未完成的工单：展示预期完成时间（changeConfirmResultTime）
            changeFlowVO.setChangeEndTime(changeConfirmResultTime);
            
            // 超期判断：非完结/取消的变更单，判断是否超过两个工作日未处理
            boolean isOverdue = false;
            Long currentTime = DateUtils.getCurrentTime();
            
            // 判断变更单的 changeConfirmResultTime 是否超过两个工作日
            if (isOverdueByWorkdays(changeConfirmResultTime, currentTime, 2)) {
                isOverdue = true;
            }
            
            // 判断执行项的 changeExecFinishTime 是否超过两个工作日
            // 注意：只判断未完结或未取消的行动工单对应的行动项
            if (!isOverdue) {
                // 从Map中获取该主单下所有行动工单
                List<ChangeSubFlowRecord> allSubFlows = subFlowRecordsMap.getOrDefault(changeRecord.getId(), new ArrayList<>());
                // 内部过滤：只保留未完结或未取消的行动工单
                List<ChangeSubFlowRecord> activeSubFlows = allSubFlows.stream()
                        .filter(subFlow -> {
                            Integer status = subFlow.getStatus();
                            return status != null 
                                && !ChangeSubFlowStatusEnum.FINISHED.getStatus().equals(status)
                                && !ChangeSubFlowStatusEnum.CANCELLED.getStatus().equals(status);
                        })
                        .collect(Collectors.toList());
                
                if (CollectionUtils.isNotEmpty(activeSubFlows)) {
                    // 提取未完结或未取消的行动工单的ID
                    Set<Long> activeSubFlowRecordIds = activeSubFlows.stream()
                            .map(ChangeSubFlowRecord::getId)
                            .collect(Collectors.toSet());
                    
                    // 批量查询这些行动工单对应的行动项
                    List<ChangeExecRecord> execRecords = changeExecRecordMapper.selectBySubFlowRecordIds(new ArrayList<>(activeSubFlowRecordIds));
                    if (CollectionUtils.isNotEmpty(execRecords)) {
                        for (ChangeExecRecord execRecord : execRecords) {
                            Long changeExecFinishTime = execRecord.getChangeExecFinishTime();
                            if (changeExecFinishTime != null && isOverdueByWorkdays(changeExecFinishTime, currentTime, 2)) {
                                isOverdue = true;
                                break;
                            }
                        }
                    }
                }
            }
            
            changeFlowVO.setIsOverdue(isOverdue);
        }
    }

    /**
     * 判断指定时间是否超过指定工作日数未处理
     * 
     * @param targetTime 目标时间（需要判断的时间）
     * @param currentTime 当前时间
     * @param workdays 工作日数
     * @return true: 超过指定工作日数未处理；false: 未超过
     */
    private boolean isOverdueByWorkdays(Long targetTime, Long currentTime, int workdays) {
        if (targetTime == null || currentTime == null) {
            return false;
        }
        
        // 计算目标时间加上指定工作日数后的时间
        Long deadlineTime = addWorkdays(targetTime, workdays);
        
        // 如果当前时间超过截止时间，则超期
        return currentTime > deadlineTime;
    }

    /**
     * 在指定时间基础上增加指定工作日数（排除周末）
     * 
     * @param time 起始时间
     * @param workdays 工作日数
     * @return 增加工作日数后的时间
     */
    private Long addWorkdays(Long time, int workdays) {
        if (time == null || workdays <= 0) {
            return time;
        }
        
        java.util.Calendar calendar = java.util.Calendar.getInstance();
        calendar.setTimeInMillis(time);
        
        int addedDays = 0;
        while (addedDays < workdays) {
            calendar.add(java.util.Calendar.DAY_OF_MONTH, 1);
            int dayOfWeek = calendar.get(java.util.Calendar.DAY_OF_WEEK);
            // 排除周六（7）和周日（1）
            if (dayOfWeek != java.util.Calendar.SATURDAY && dayOfWeek != java.util.Calendar.SUNDAY) {
                addedDays++;
            }
        }
        
        return calendar.getTimeInMillis();
    }

}
