package com.netease.mail.yanxuan.change.integration.email.service.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.netease.mail.yanxuan.change.integration.email.conig.RpcTemplate;
import com.netease.mail.yanxuan.change.integration.email.dto.*;
import com.netease.mail.yanxuan.change.integration.email.email.*;
import com.netease.mail.yanxuan.change.integration.email.service.IIusService;
import com.netease.mail.yanxuan.change.common.bean.AjaxResult;
import com.netease.mail.yanxuan.change.integration.email.enums.OrgUserTypeEnum;
import com.netease.mail.yanxuan.change.integration.email.enums.StaffStatusEnum;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.UserBaseDTO;
import org.assertj.core.util.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import com.alibaba.fastjson.JSON;
import com.netease.mailsaas.core.utils.Assert;
import com.netease.yanxuan.qc.parent.utils.LoginUserUtils;

@Service
public class IusServiceImpl implements IIusService {

    @Value("${rpc.url.ius.basic}")
    private String iusUrl;

    private Logger logger = LoggerFactory.getLogger(IusServiceImpl.class);

    @Autowired
    private RpcTemplate rpcTemplate;

    /**
     * 接口地址
     */
    private static final String USER_INFO_SUFFIX = "/icac/user/getUserInfo.json";

    private static final String QUERY_USER_INFO_SUFFIX = "/v1/%s/user/query";

    private static final String DEPT_LEVEL_LEADER = "/icac/user/uniteOrg/getPartLeaders.json";

    private static final String FUZZY_USERINFO = "/icac/user/uniteOrg/fuzzyPartSubUserByOrgPosId.json";

    private static final String ALL_USER_ROLE_INFOS = "/carrier/user/business/getAllUserProductRoleInfos.json";

    private static final String LIST_USER_BY_ROLE_ID = "/%s/user/listUserByRoleId.json";

    private static final String LIST_PRODUCT_HAS_UID = "/qc/product/listProductHasUid.json";

    private static final String THIRD_ORG_POS_URL = "/v1/ehc/allTeam/pos/query";

    private static final Integer THIRD_ORG_POS_LEVEL = 97;

    @Override
    @Cacheable(value = Constants.EHC_CACHE_STRING, key = "'getUserInfo'+#uid")
    public UserInfoDTO getUserInfo(String uid) {
        if (!StringUtils.hasText(uid)) {
            return null;
        }
        Map<String, Object> params = new HashMap<>();
        params.put("uid", uid);
        ResponseResult result = null;
        try {
            result = rpcTemplate.post(iusUrl + USER_INFO_SUFFIX, params, 5000,
                new IusRpcResult<>(ResponseResult.class));
        } catch (Exception e) {
            logger.error("[getUserInfo] uid={}, e={}", uid, e);
        }
        if (result == null) {
            return null;
        }
        return (UserInfoDTO) result.getData();
    }

    @Override
    public List<UserInfoDTO> queryUserInfo(String productCode, String uid) {
        if (!StringUtils.hasText(uid)) {
            return null;
        }
        QueryUserInfoVO req = new QueryUserInfoVO();
        req.setUids(Lists.newArrayList(uid));
        req.setLocked(false);
        QueryUserInfoResponseResult result = null;
        String url = iusUrl + String.format(QUERY_USER_INFO_SUFFIX, productCode);
        try {
            result = rpcTemplate.postJson(url, JSON.toJSONString(req), 5000,
                new RpcObjectHandler<>(QueryUserInfoResponseResult.class));
        } catch (Exception e) {
            logger.error("[queryUserInfo] uid={}, e={}", uid, e);
        }
        if (result == null) {
            return null;
        }
        return result.getData();
    }

    @Override
    @Cacheable(value = Constants.EHC_CACHE_STRING, key = "'getOrgByUid'+#uid")
    public String getOrgByUid(String uid) {
        UserInfoDTO userInfoDTO = getUserInfo(uid);
        if (userInfoDTO == null) {
            return null;
        }
        String orgName = userInfoDTO.getDeptLevel3();
        if (!StringUtils.hasText(orgName)) {
            orgName = "无三级部门";
        }
        //        if (!StringUtils.hasText(orgName)) {
        //            orgName = userInfoDTO.getDeptLevel2();
        //        }
        //        if (!StringUtils.hasText(orgName)) {
        //            orgName = userInfoDTO.getDeptLevel1();
        //        }
        logger.info("[getOrgByUid] orgName={}", orgName);
        return orgName;
    }

    @Override
    public boolean checkUserIsLeave(String uid) {
        UserInfoDTO userInfoDTO = getUserInfo(uid);
        if (userInfoDTO != null && !StaffStatusEnum.DIMISSION.getType().equals(userInfoDTO.getStaffStatus())) {
            // 离职
            return true;
        }
        return false;
    }

    @Override
    public List<DeptIdLevelLeaderDTO> getDeptIdLevelLeader(String uid, Integer orgPosId) {
        Map<String, Object> params = new HashMap<>();
        params.put("uid", uid);
        params.put("orgPosId", orgPosId);
        DeptIdLevelResponseResult levelResponseResult = null;
        try {
            levelResponseResult = rpcTemplate.post(iusUrl + DEPT_LEVEL_LEADER, params, 500,
                new RpcObjectHandler<>(DeptIdLevelResponseResult.class));
        } catch (Exception ex) {
            logger.error("[getDeptIdLevelLeader] ex={}", ex);
        }
        if (levelResponseResult == null) {
            return null;
        }
        return levelResponseResult.getData();
    }

    @Override
    public List<DeptIdLevelLeaderDTO> getThreeLevelDeptLeader(String uid) {
        UserInfoDTO userInfoDTO = getUserInfo(uid);
        logger.info("[getThreeLevelDeptLeader] userInfoDTO={}", JSON.toJSONString(userInfoDTO));
        if (userInfoDTO == null) {
            return null;
        }
        List<DeptIdLevelLeaderDTO> userList = getDeptIdLevelLeader(userInfoDTO.getUid(), userInfoDTO.getDeptIdLevel3());
        if (!CollectionUtils.isEmpty(userList)) {
            List<DeptIdLevelLeaderDTO> filterList = userList.stream()
                .filter(user -> OrgUserTypeEnum.PRINCIPAL.getValue() == user.getUserType()
                    && userInfoDTO.getDeptIdLevel3().equals(user.getOrgPosId()))
                .collect(Collectors.toList());
            return filterList;
        }
        return null;
    }

    @Override
    public List<DeptIdLevelLeaderDTO> getForthLevelDeptLeader(String uid) {
        UserInfoDTO userInfoDTO = getUserInfo(uid);
        logger.info("[getForthLevelDeptLeader] userInfoDTO={}", JSON.toJSONString(userInfoDTO));
        if (userInfoDTO == null) {
            return null;
        }
        List<DeptIdLevelLeaderDTO> userList = getDeptIdLevelLeader(userInfoDTO.getUid(), userInfoDTO.getDeptIdLevel4());
        logger.debug("[getForthLevelDeptLeader] userList={}", JSON.toJSONString(userList));
        if (!CollectionUtils.isEmpty(userList)) {
            List<DeptIdLevelLeaderDTO> filterList = userList.stream()
                .filter(user -> OrgUserTypeEnum.PRINCIPAL.getValue() == user.getUserType()
                    && userInfoDTO.getDeptIdLevel4().equals(user.getOrgPosId()))
                .collect(Collectors.toList());
            logger.debug("[getForthLevelDeptLeader] filterList={}", JSON.toJSONString(filterList));
            return filterList;
        }
        return Lists.newArrayList();
    }

    @Override
    public List<UserUniteOrgMetaDTO> fuzzyQueryUserInfo(Long orgPosId, int level, int type, int queryType,
                                                        String keyword) {
        Map<String, Object> params = new HashMap<>();
        params.put("orgPosId", orgPosId);
        params.put("level", level);
        params.put("type", type);
        params.put("queryType", queryType);
        params.put("keyword", keyword);
        UserUniteOrgMetaResponseResult result = null;
        try {
            result = rpcTemplate.post(iusUrl + FUZZY_USERINFO, params, 500,
                new RpcObjectHandler<>(UserUniteOrgMetaResponseResult.class));
        } catch (Exception ex) {
            logger.error("fuzzyQueryUserInfo: ex={}", ex);
        }
        if (result == null) {
            return null;
        }
        return result.getData();
    }

    @Override
    public SearchResult<UserProductRoleStaticsInfosDTO> getAllUserProductRoleInfos(Integer staff, Long roleId,
                                                                                   Long orgPosId, Integer stationId, Integer locked, String keyword, Integer curPage, Integer pageSize) {
        Map<String, Object> params = new HashMap<>();
        params.put("staff", staff);
        if (roleId != null) {
            params.put("roleId", roleId);
        }
        params.put("orgPosId", orgPosId);
        params.put("stationId", stationId);
        params.put("locked", locked);
        if (keyword != null) {
            params.put("keyword", keyword);
        }
        params.put("curPage", curPage);
        params.put("pageSize", pageSize);
        AllUserProductRoleInfosResponseResult roleInfosResponseResult = null;
        try {
            roleInfosResponseResult = rpcTemplate.get(iusUrl + ALL_USER_ROLE_INFOS, params,
                new RpcObjectHandler<>(AllUserProductRoleInfosResponseResult.class));
        } catch (Exception ex) {
            logger.error("[getAllUserProductRoleInfos] ex={}", ex);
        }
        if (roleInfosResponseResult == null) {
            return null;
        }
        return roleInfosResponseResult.getData();
    }

    @Override
    //    @Cacheable(value = Constants.EHC_CACHE_STRING, key = "'listUserByRoleId'+#productCode+#roleId")
    public List<UserBaseDTO> listUserByRoleId(String productCode, Long roleId) {
        logger.info("[listUserByRoleId] productCode={}, roleId={}", productCode, roleId);
        Map<String, Object> params = new HashMap<>();
        params.put("roleId", roleId);
        RoleResponseResult result = null;
        try {
            result = rpcTemplate.post(String.format(iusUrl + LIST_USER_BY_ROLE_ID, productCode), params, 500,
                new RpcObjectHandler<>(RoleResponseResult.class));
        } catch (Exception ex) {
            logger.error("[listUserByRoleId] roleId={}, ex={}", roleId, ex);
        }
        if (result == null) {
            return null;
        }
        return result.getData();
    }

    @Override
    public List<UserUniteOrgMetaDTO> fuzzyQueryUserInformation(int queryType, String keyword) {
        String baseUid = LoginUserUtils.getLoginUserEmail();
        UserInfoDTO baseUser = getUserInfo(baseUid);
        Assert.notNull(baseUser, "查询用户信息失败，获取基础信息异常。");
        Assert.notNull(baseUser.getDeptIdLevel3(), "查询用户信息失败，获取用户【" + baseUid + "】三级部门失败");
        List<UserUniteOrgMetaDTO> result = fuzzyQueryUserInfo((long) baseUser.getDeptIdLevel3(), -1, 0, queryType,
            keyword);
        return result;
    }

    @Override
    public UserVO lv3UserInfo(String uid) {
        UserVO user = new UserVO();
        user.setUserEmail(uid);
        UserInfoDTO userInfoDTO = getUserInfo(uid);
        if (userInfoDTO == null) {
            return user;
        }
        if (!StringUtils.hasText(userInfoDTO.getDeptLevel3())) {
            user.setUserName(userInfoDTO.getName());
            user.setOrgId((long) userInfoDTO.getDeptIdLevel3());
            user.setOrgName("无三级部门");
            return user;
        }
        user.setUserName(userInfoDTO.getName());
        user.setOrgId((long) userInfoDTO.getDeptIdLevel3());
        user.setOrgName(userInfoDTO.getDeptLevel3());
        return user;
    }

    @Override
    public List<String> listProductHasUid(String uid) {
        if (StringUtils.isEmpty(uid)) {
            logger.info("[listProductHasUid] uid -> null");
            return null;
        }
        Map<String, Object> params = new HashMap<>();
        params.put("uid", uid);
        AjaxResult<List<String>> result = null;
        try {
            result = rpcTemplate.post(iusUrl + LIST_PRODUCT_HAS_UID, params, 500,
                new RpcObjectHandler<>(AjaxResult.class));
        } catch (Exception e) {
            logger.error("[listProductHasUid] 接口调用错误 e:{}", e);
        }
        if (result == null) {
            return null;
        }
        return result.getData();
    }

    @Override
    public List<ThirdOrgPosDTO> getThirdOrgPosName(String name) {
        Map<String, Object> params = new HashMap<>();
        params.put("levels", THIRD_ORG_POS_LEVEL);
        AjaxResult<List<ThirdOrgPosDTO>> result = null;
        try {
            result = rpcTemplate.get(iusUrl + THIRD_ORG_POS_URL, params, new RpcObjectHandler<>(AjaxResult.class));
        } catch (Exception e) {
            logger.error("[getThirdOrgPosName] 接口调用错误 e:{}", e);
        }
        if (result == null) {
            return null;
        }
        //搜索
        List<ThirdOrgPosDTO> thirdOrgPosDTOS = JSON.parseArray(JSON.toJSONString(result.getData()),
            ThirdOrgPosDTO.class);
        return thirdOrgPosNameFuzzySearch(thirdOrgPosDTOS, name);
    }

    private List<ThirdOrgPosDTO> thirdOrgPosNameFuzzySearch(List<ThirdOrgPosDTO> orgPosDTOList, String name) {
        if (!CollectionUtils.isEmpty(orgPosDTOList)) {
            orgPosDTOList = orgPosDTOList.stream().filter(f -> f.getOrgPosName().contains(name))
                .collect(Collectors.toList());
            return orgPosDTOList;
        }
        return null;
    }

}
