/**
 * @(#)ChangeFlowController.java, 2022/11/14.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.mail.yanxuan.change.web.controller;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.github.pagehelper.PageInfo;
import com.netease.mail.yanxuan.change.biz.biz.ChangeFlowBiz;
import com.netease.mail.yanxuan.change.biz.config.TitleConfig;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowService;
import com.netease.mail.yanxuan.change.biz.service.InteriorChangeConfigService;
import com.netease.mail.yanxuan.change.common.bean.AjaxResult;
import com.netease.mail.yanxuan.change.dal.meta.model.po.QueryCommanderEmailReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCancelReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowCreateReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowDeliverReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowSubmitReq;
import com.netease.mail.yanxuan.change.dal.meta.model.rpc.CommanderResponse;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.BasicChangeFlowVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeFlowListVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeFlowVO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.InterfaceInputDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.exec.UserBaseContainerDTO;

import lombok.extern.slf4j.Slf4j;

/**
 * @Author zcwang
 * @Date 2022/11/14
 */
@Slf4j
@RestController
@RequestMapping("/change/flow")
public class ChangeFlowController {

    @Autowired
    private ChangeFlowService changeFlowService;

    @Autowired
    private ChangeFlowBiz changeFlowBiz;

    @Autowired
    private InteriorChangeConfigService interiorChangeConfigService;

    /**
     * 变更工单详情查询
     * 
     * @return
     */
    @GetMapping("/detail")
    public AjaxResult<ChangeFlowVO> detail(@RequestParam Long id) {
        return AjaxResult.success(changeFlowBiz.detail(id));
    }

    /**
     * 变更工单列表查询
     * 
     * @return
     */
    @GetMapping("/query")
    public AjaxResult<ChangeFlowListVO> query(@RequestParam(name = "page", defaultValue = "1") Integer page,
                                              @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                              @RequestParam(name = "itemId", required = false) String itemId,
                                              @RequestParam(name = "supplier", required = false) String supplier,
                                              @RequestParam(name = "parentChangeClassId", required = false) Long parentChangeClassId,
                                              @RequestParam(name = "sonChangeClassId", required = false) Long sonChangeClassId,
                                              @RequestParam(name = "changeState", required = false) Integer changeState,
                                              @RequestParam(name = "changeCommander", required = false) String changeCommander,
                                              @RequestParam(name = "changeDepartment", required = false) String changeDepartment,
                                              @RequestParam(name = "changeExecDepartment", required = false) String changeExecDepartment,
                                              @RequestParam(name = "startTime", required = false) Long startTime,
                                              @RequestParam(name = "endTime", required = false) Long endTime) {
        ChangeFlowListQueryReq changeFlowListQueryReq = new ChangeFlowListQueryReq();
        changeFlowListQueryReq.setItemId(itemId);
        changeFlowListQueryReq.setSupplier(supplier);
        changeFlowListQueryReq.setParentChangeClassId(parentChangeClassId);
        changeFlowListQueryReq.setSonChangeClassId(sonChangeClassId);
        changeFlowListQueryReq.setState(changeState);
        changeFlowListQueryReq.setChangeCommander(changeCommander);
        changeFlowListQueryReq.setChangeDepartment(changeDepartment);
        changeFlowListQueryReq.setChangeExecDepartment(changeExecDepartment);
        changeFlowListQueryReq.setStartTime(startTime);
        changeFlowListQueryReq.setEndTime(endTime);
        return AjaxResult.success(changeFlowBiz.query(page, pageSize, changeFlowListQueryReq));
    }

    /**
     * 返回提示文案
     * 
     * @return
     */
    @GetMapping("/title")
    public AjaxResult<TitleConfig> title() {
        return AjaxResult.success(changeFlowService.getTitleConfig());
    }

    /**
     * 发起变更，如果发起人=审核人，自动跳转到第一个节点
     * 
     * @return
     */
    @PostMapping("/create")
    public AjaxResult<String> createAndSubmit(@RequestBody @Valid ChangeFlowCreateReq changeFlowCreateReq) {
        return AjaxResult.success(changeFlowBiz.createAndSubmit(changeFlowCreateReq));
    }

    /**
     * 变更工单提交
     * 
     * @return
     */
    @PostMapping("/submit")
    public AjaxResult<String> submit(@RequestBody @Valid ChangeFlowSubmitReq changeFlowSubmitReq) {
        return AjaxResult.success(changeFlowBiz.submit(changeFlowSubmitReq));
    }

    /**
     * 工单转交
     * 
     * @return
     */
    @PostMapping("/deliver")
    public AjaxResult<Void> deliver(@RequestBody @Valid ChangeFlowDeliverReq req) {
        changeFlowBiz.deliver(req);
        return AjaxResult.success();
    }

    /**
     * 工单取消
     * 
     * @return
     */
    @PostMapping("/cancel")
    public AjaxResult<Boolean> cancel(@RequestBody @Valid ChangeFlowCancelReq changeFlowCancelReq) {
        return AjaxResult.success(changeFlowBiz.cancel(changeFlowCancelReq));
    }

    /**
     * 引用变更工单
     * 
     * @return
     */
    @GetMapping("/quote")
    public AjaxResult<BasicChangeFlowVO> quote(@RequestParam Long flowId) {
        return AjaxResult.success(changeFlowBiz.quote(flowId));
    }

    /**
     * 导出变更
     * 
     * @return
     */
    @PostMapping("/export")
    public AjaxResult<Void> export(@RequestParam(name = "itemId", required = false) String itemId,
                        @RequestParam(name = "supplier", required = false) String supplier,
                        @RequestParam(name = "parentChangeClassId", required = false) Long parentChangeClassId,
                        @RequestParam(name = "sonChangeClassId", required = false) Long sonChangeClassId,
                        @RequestParam(name = "changeState", required = false) Integer changeState,
                        @RequestParam(name = "changeCommander", required = false) String changeCommander,
                        @RequestParam(name = "changeDepartment", required = false) String changeDepartment,
                        @RequestParam(name = "changeExecDepartment", required = false) String changeExecDepartment,
                        @RequestParam(name = "startTime", required = false) Long startTime,
                        @RequestParam(name = "endTime", required = false) Long endTime, HttpServletResponse response) {
        ChangeFlowListQueryReq changeFlowListQueryReq = new ChangeFlowListQueryReq();
        changeFlowListQueryReq.setItemId(itemId);
        changeFlowListQueryReq.setSupplier(supplier);
        changeFlowListQueryReq.setParentChangeClassId(parentChangeClassId);
        changeFlowListQueryReq.setSonChangeClassId(sonChangeClassId);
        changeFlowListQueryReq.setState(changeState);
        changeFlowListQueryReq.setChangeCommander(changeCommander);
        changeFlowListQueryReq.setChangeDepartment(changeDepartment);
        changeFlowListQueryReq.setChangeExecDepartment(changeExecDepartment);
        changeFlowListQueryReq.setStartTime(startTime);
        changeFlowListQueryReq.setEndTime(endTime);
        changeFlowBiz.export(changeFlowListQueryReq, response);
        return AjaxResult.success();
    }

    /**
     * 获取审批人
     * 
     * @param interfaceInput
     * @return
     */
    @PostMapping("/getOperator")
    public AjaxResult<UserBaseContainerDTO> getOperator(@RequestBody InterfaceInputDTO interfaceInput) {
        return AjaxResult.success(changeFlowBiz.getOperator(interfaceInput));
    }

    /**
     * 查询负责人信息
     * @param req
     * @return
     */
    @PostMapping("/queryCommanderInfo")
    public AjaxResult<CommanderResponse> queryCommanderInfo(@RequestBody QueryCommanderEmailReq req) {
        return AjaxResult.success(interiorChangeConfigService.queryCommanderEmail(req));
    }
}
