/**
 * @(#)ChangeFlowExportCallback.java, 2022/12/8.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.mail.yanxuan.change.biz.callback;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.netease.mail.dp.des.process.api.entity.CallbackQueryReq;
import com.netease.mail.dp.des.process.api.entity.ExcelCell;
import com.netease.mail.dp.des.process.api.entity.ExcelTitle;
import com.netease.mail.dp.des.process.api.entity.PivotTable;
import com.netease.mail.dp.des.starter.callback.DesCallbackService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowExportQuery;
import com.netease.mail.yanxuan.change.biz.service.InteriorChangeConfigService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeTypeService;
import com.netease.mail.yanxuan.change.biz.service.rpc.ItemService;
import com.netease.mail.yanxuan.change.biz.service.rpc.IusService;
import com.netease.mail.yanxuan.change.biz.service.rpc.QCService;
import com.netease.mail.yanxuan.change.common.enums.BusinessFormEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeStatusEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeSubjectEnum;
import com.netease.mail.yanxuan.change.common.enums.MainSpuStatusEnum;
import com.netease.mail.yanxuan.change.common.util.DateUtils;
import com.netease.mail.yanxuan.change.common.util.ExcelUtil;
import com.netease.mail.yanxuan.change.dal.entity.ChangeExecRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeType;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeExecRecordMapper;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeRecordMapper;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeGoodsPrincipalPO;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ItemVO;
import com.netease.mail.yanxuan.change.integration.excel.ChangeFlowExcelDTO;
import com.netease.mail.yanxuan.change.integration.excel.ChangeFlowExcelVo;
import com.netease.mail.yanxuan.change.integration.flow.UserQueryDTO;
import com.netease.mail.yanxuan.change.integration.flow.ius.rsp.IusUserInfoRsp;
import com.netease.mail.yanxuan.change.integration.item.SimplePhyCateGoryResultCo;
import com.netease.mail.yanxuan.change.integration.item.meta.SpuTO;
import com.netease.mail.yanxuan.change.integration.item.param.BatchQuerySpuInfoParam;
import com.netease.mail.yanxuan.change.integration.item.param.CommonIdsParamQuery;
import com.netease.mail.yanxuan.change.integration.qc.meta.QcCategoryVO;

import lombok.extern.slf4j.Slf4j;

/**
 * @Author zcwang
 * @Date 2022/12/8
 */
@Slf4j
@Component
public class ChangeFlowExportCallback implements DesCallbackService {

    @Autowired
    private ChangeRecordMapper changeRecordMapper;
    
    @Autowired
    private ChangeTypeService changeTypeService;

    @Autowired
    private ChangeExecRecordMapper changeExecRecordMapper;

    @Autowired
    private ItemService itemService;

    @Autowired
    private InteriorChangeConfigService interiorChangeConfigService;

    @Autowired
    private QCService qcService;

    @Autowired
    private IusService iusService;

    @Override
    public ExcelTitle queryExcelTitle(CallbackQueryReq req) {
        log.info("[op: change flow export callback queryExcelTitle] req={}", JSON.toJSONString(req));
        ChangeFlowExportQuery query = JSON.parseObject(req.getParamJson(), ChangeFlowExportQuery.class);
        ChangeFlowListQueryReq changeFlowListQueryReq = new ChangeFlowListQueryReq();
        BeanUtils.copyProperties(query, changeFlowListQueryReq);
        List<ChangeRecord> list = changeRecordMapper.selectByCondition(changeFlowListQueryReq);
        ExcelTitle excelTitle = ExcelUtil.fetchExceltile(ChangeFlowExcelVo.class);
        excelTitle.setTotal((long) list.size());
        log.info("[op: change flow export callback queryExcelTitle] result={}", JSON.toJSONString(excelTitle));
        return excelTitle;
    }

    @Override
    public List<List<ExcelCell>> queryExcelData(CallbackQueryReq req) {
        log.info("[op: address book export callback queryExcelData] req={}", JSON.toJSONString(req));
        ChangeFlowExportQuery query = JSON.parseObject(req.getParamJson(), ChangeFlowExportQuery.class);
        query.setOffset(req.getOffset());
        query.setLimit(req.getLimit());
        List<ChangeFlowExcelDTO> list = getExportList(query);
        return list.stream().map(ChangeFlowExcelVo::init).collect(Collectors.toList());
    }

    private List<ChangeFlowExcelDTO> getExportList(ChangeFlowExportQuery query) {
        ChangeFlowListQueryReq changeFlowListQueryReq = new ChangeFlowListQueryReq();
        BeanUtils.copyProperties(query, changeFlowListQueryReq);
        List<ChangeRecord> changeRecords = changeRecordMapper.selectByConditionAndLimit(changeFlowListQueryReq);
        log.debug("[getExportList] changeRecords:{}", changeRecords);
        if (CollectionUtils.isEmpty(changeRecords)) {
            return Lists.newArrayList();
        }
        List<ChangeFlowExcelDTO> changeFlowExcelDTOList = new ArrayList<>();
        // 导出excel，按照商品/供应商和行动项，n*n重复拆分，例2个商品3个行动项，拆分成2*3=6行，其他属性一致
        for (ChangeRecord record : changeRecords) {
            ChangeSubjectEnum changeSubjectEnum = ChangeSubjectEnum.getChangeSubjectEnum(record.getChangeSubject());
            switch (changeSubjectEnum) {
                case PRODUCT:
                    String changeItemJsonStr = record.getChangeItem();
                    List<ItemVO> itemVOS = JSON.parseArray(changeItemJsonStr, ItemVO.class);
                    List<ChangeExecRecord> changeExecRecords = changeExecRecordMapper.selectByChangeRecordId(record.getId());
                    List<IusUserInfoRsp> userListInfo = new ArrayList<>();
                    try {
                        Set<String> userEmail = changeExecRecords.stream().map(ChangeExecRecord::getChangeExecUserEmail).collect(Collectors.toSet());
                        userListInfo = iusService.queryUserListInfo(UserQueryDTO.builder().uids(new ArrayList<>(userEmail)).build());
                    } catch (Exception ex) {
                        log.info("query user info has ex", ex);
                    }
                    List<Long> itemIds = itemVOS.stream().map(ItemVO::getItemId).collect(Collectors.toList());
                    // 批量查询spu信息
                    List<SpuTO> spuTOS = itemService.batchQuerySpuInfo(BatchQuerySpuInfoParam.builder().ids(itemIds)
                            .commonProps(new ArrayList<>()).spuProps(Arrays.asList("itemSetupType", "businessForm")).build());
                    // 批量查询物理类目
                    Map<Long, List<SimplePhyCateGoryResultCo>> categoryChain = itemService
                            .queryBySpuIds(CommonIdsParamQuery.builder().ids(itemIds).build());
                    // 查询商品对应负责人信息
                    Map<Long, ChangeGoodsPrincipalPO> longChangeGoodsPrincipalPOMap = interiorChangeConfigService
                            .queryGoodsPrincipalInfo(itemIds);
                    log.debug("[detail] spuTOS:{}", JSON.toJSONString(spuTOS));
                    for (ItemVO itemVO : itemVOS) {
                        for (ChangeExecRecord changeExecRecord : changeExecRecords) {
                            Long itemId = itemVO.getItemId();
                            // 组装通用信息，除商品信息，行动项以外其他字段一致
                            ChangeFlowExcelDTO changeFlowExcelDTO = this.obtainCommon(record, changeSubjectEnum);
                            changeFlowExcelDTO.setSupplier("/");
                            changeFlowExcelDTO.setChangeItem(itemVO.getItemName());
                            changeFlowExcelDTO.setSPUId(itemId);
                            List<QcCategoryVO> qcCategoryList = qcService.getQcCategoryListByItemId(itemId);
                            if (CollectionUtils.isNotEmpty(qcCategoryList)) {
                                String collect = qcCategoryList.stream().map(QcCategoryVO::getCategoryName)
                                        .collect(Collectors.joining("/"));
                                changeFlowExcelDTO.setQcCategory(collect);
                            } else {
                                changeFlowExcelDTO.setQcCategory("/");
                            }
                            List<SimplePhyCateGoryResultCo> simplePhyCateGoryResultCos = categoryChain.get(itemId);
                            if (CollectionUtils.isNotEmpty(simplePhyCateGoryResultCos)) {
                                String collect = simplePhyCateGoryResultCos.stream()
                                    .map(SimplePhyCateGoryResultCo::getName).collect(Collectors.joining("/"));
                                changeFlowExcelDTO.setPhyCategory(collect);
                            } else {
                                changeFlowExcelDTO.setPhyCategory("/");
                            }
                            Optional<SpuTO> optionalSpuTO = spuTOS.stream().filter(spu -> spu.getId() == itemId).findAny();
                            if (!optionalSpuTO.isPresent()) {

                            } else {
                                SpuTO spuTO = optionalSpuTO.get();
                                Map<String, String> propertyMap = spuTO.getPropertyMap();
                                changeFlowExcelDTO.setItemSetupType(Integer.valueOf(propertyMap.get("itemSetupType")));
                                String businessForm = propertyMap.get("businessForm");
                                int business = StringUtils.isBlank(businessForm) ? 0 : Integer.parseInt(businessForm);
                                changeFlowExcelDTO.setBusinessForm(BusinessFormEnum.getByType(business));
                                changeFlowExcelDTO.setStatus(MainSpuStatusEnum.getByStatus(spuTO.getStatus()));
                            }
                            ChangeGoodsPrincipalPO changeGoodsPrincipalPO = longChangeGoodsPrincipalPOMap.get(itemId);
                            changeFlowExcelDTO.setFunctionary(changeGoodsPrincipalPO.getFunctionaryName()
                                + changeGoodsPrincipalPO.getFunctionaryEmail());
                            changeFlowExcelDTO.setPurchase(
                                changeGoodsPrincipalPO.getPurchaseName() + changeGoodsPrincipalPO.getPurchaseEmail());
                            changeFlowExcelDTO.setGoodsSqe(
                                changeGoodsPrincipalPO.getGoodsSqeName() + changeGoodsPrincipalPO.getGoodsSqeEmail());
                            changeFlowExcelDTO.setGoodsProjectName(
                                changeGoodsPrincipalPO.getGoodsProjectName() + changeGoodsPrincipalPO.getGoodsProjectEmail());
                            // 组装行动项相关信息
                            obtainChangeExec(changeExecRecord, changeFlowExcelDTO, userListInfo);
                            changeFlowExcelDTO.setChangeResult(record.getChangeResult());
                            changeFlowExcelDTO
                                .setChangeEndTime(record.getState().equals(ChangeStatusEnum.END.getStatus())
                                    ? DateUtils.parseLongToString(record.getUpdateTime(), DateUtils.DATE_TIME_FORMAT)
                                    : "");
                            changeFlowExcelDTOList.add(changeFlowExcelDTO);
                        }
                    }
                case SUPPLIER:
                    // 供应商变更，以行动项循环
                case OTHER:
                default:
            }

        }
        return changeRecords.stream().map(record -> {
            ChangeFlowExcelDTO changeFlowExcelDTO = new ChangeFlowExcelDTO();
            return changeFlowExcelDTO;
        }).collect(Collectors.toList());

    }

    private void obtainChangeExec(ChangeExecRecord changeExecRecord, ChangeFlowExcelDTO changeFlowExcelDTO,
        List<IusUserInfoRsp> userListInfo) {
        changeFlowExcelDTO.setExecDepartment(changeExecRecord.getChangeExecDepartment());
        Optional<IusUserInfoRsp> anyExeUser = userListInfo.stream()
            .filter(u -> u.getUid().equals(changeExecRecord.getChangeExecUserEmail())).findAny();
        String execUserName = "";
        if (anyExeUser.isPresent()) {
            execUserName = anyExeUser.get().getName();
        }
        changeFlowExcelDTO.setChangeExecUser(execUserName + changeExecRecord.getChangeExecUserEmail());
        changeFlowExcelDTO.setChangeExecRiskDesc(changeExecRecord.getChangeRiskDesc());
        changeFlowExcelDTO.setChangeExecProject(changeExecRecord.getChangeExecProject());
        changeFlowExcelDTO.setChangeChecking(changeExecRecord.getChangeChecking());
        changeFlowExcelDTO.setChangeExecFinishTime(changeExecRecord.getChangeExecFinishTime());
        changeFlowExcelDTO.setChangeExecFinishDesc(changeExecRecord.getChangeExecFinishDesc());
    }

    /**
     * 组装通用字段
     * @param record
     * @param changeSubjectEnum
     * @return
     */
    private ChangeFlowExcelDTO obtainCommon(ChangeRecord record, ChangeSubjectEnum changeSubjectEnum) {
        ChangeFlowExcelDTO changeFlowExcelDTO = new ChangeFlowExcelDTO();
        changeFlowExcelDTO.setFlowId(record.getFlowId());
        String createTimeStr = DateUtils.parseLongToString(record.getCreateTime(), DateUtils.DATE_TIME_FORMAT);
        changeFlowExcelDTO.setCreateTime(createTimeStr);
        changeFlowExcelDTO.setChangeCreator(record.getCreator());
        changeFlowExcelDTO.setChangeSubject(changeSubjectEnum.getDesc());
        ChangeType parentChangeType = changeTypeService.getChangeTypeById(record.getParentChangeClassId());
        ChangeType sonChangeType = changeTypeService.getChangeTypeById(record.getSonChangeClassId());
        changeFlowExcelDTO.setChangeType(parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
        changeFlowExcelDTO.setChangeLevel(record.getChangeLevel());
        changeFlowExcelDTO.setChangeDepartment(record.getChangeDepartment());
        changeFlowExcelDTO.setChangeCommander(record.getChangeCommander());
        changeFlowExcelDTO.setChangeReason(record.getChangeReason());
        changeFlowExcelDTO.setChangeContent(record.getChangeContent());
        changeFlowExcelDTO.setChangeRiskDesc(record.getChangeRiskDesc());
        changeFlowExcelDTO.setChangeProfit(record.getChangeProfit());
        changeFlowExcelDTO.setChangeProfitDesc(record.getChangeProfitDesc());
        return changeFlowExcelDTO;
    }

    @Override
    public PivotTable queryPivotTable(CallbackQueryReq callbackQueryReq) {
        return null;
    }
}