/**
 * @(#)ChangeFlowExportCallback.java, 2022/12/8.
 * <p/>
 * Copyright 2022 Netease, Inc. All rights reserved.
 * NETEASE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.netease.mail.yanxuan.change.biz.callback;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.netease.mail.yanxuan.change.biz.service.ChangeFlowExecService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.netease.mail.dp.des.process.api.entity.CallbackQueryReq;
import com.netease.mail.dp.des.process.api.entity.ExcelCell;
import com.netease.mail.dp.des.process.api.entity.ExcelTitle;
import com.netease.mail.dp.des.process.api.entity.PivotTable;
import com.netease.mail.dp.des.starter.callback.DesCallbackService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowExportQuery;
import com.netease.mail.yanxuan.change.biz.service.InteriorChangeConfigService;
import com.netease.mail.yanxuan.change.biz.service.change.ChangeTypeService;
import com.netease.mail.yanxuan.change.biz.service.rpc.ItemService;
import com.netease.mail.yanxuan.change.biz.service.rpc.IusService;
import com.netease.mail.yanxuan.change.biz.service.rpc.QCService;
import com.netease.mail.yanxuan.change.biz.service.rpc.SupplierService;
import com.netease.mail.yanxuan.change.common.enums.BusinessFormEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeLevelEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeResultEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeStatusEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeSubjectEnum;
import com.netease.mail.yanxuan.change.common.enums.CreateSourceEnum;
import com.netease.mail.yanxuan.change.common.enums.ItemSetupTypeEnum;
import com.netease.mail.yanxuan.change.common.enums.MainSpuStatusEnum;
import com.netease.mail.yanxuan.change.common.enums.ProfitTypeEnum;
import com.netease.mail.yanxuan.change.common.util.DateUtils;
import com.netease.mail.yanxuan.change.common.util.ExcelUtil;
import com.netease.mail.yanxuan.change.dal.entity.ChangeExecRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeType;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeExecRecordMapper;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeRecordMapper;
import com.netease.mail.yanxuan.change.dal.meta.model.po.ChangeGoodsPrincipalPO;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ItemVO;
import com.netease.mail.yanxuan.change.integration.excel.ChangeFlowExcelDTO;
import com.netease.mail.yanxuan.change.integration.excel.ChangeFlowExcelVo;
import com.netease.mail.yanxuan.change.integration.flow.UserQueryDTO;
import com.netease.mail.yanxuan.change.integration.flow.ius.req.IusDepartmentReq;
import com.netease.mail.yanxuan.change.integration.flow.ius.rsp.IusUserInfoRsp;
import com.netease.mail.yanxuan.change.integration.flow.ius.rsp.SecondaryDepartments;
import com.netease.mail.yanxuan.change.integration.flow.supplier.rsp.SupplierSimpleRsp;
import com.netease.mail.yanxuan.change.integration.item.SimplePhyCateGoryResultCo;
import com.netease.mail.yanxuan.change.integration.item.meta.SpuTO;
import com.netease.mail.yanxuan.change.integration.item.param.BatchQuerySpuInfoParam;
import com.netease.mail.yanxuan.change.integration.item.param.CommonIdsParamQuery;
import com.netease.mail.yanxuan.change.integration.qc.meta.QcCategoryVO;

import lombok.extern.slf4j.Slf4j;

/**
 * @Author zcwang
 * @Date 2022/12/8
 */
@Slf4j
@Component
public class ChangeFlowExportCallback implements DesCallbackService {

    @Autowired
    private ChangeRecordMapper changeRecordMapper;
    
    @Autowired
    private ChangeTypeService changeTypeService;

    @Autowired
    private ChangeExecRecordMapper changeExecRecordMapper;

    @Autowired
    private ItemService itemService;

    @Autowired
    private InteriorChangeConfigService interiorChangeConfigService;

    @Autowired
    private QCService qcService;

    @Autowired
    private IusService iusService;

    @Autowired
    private SupplierService supplierService;

    @Autowired
    private ChangeFlowExecService changeFlowExecService;

    @Override
    public ExcelTitle queryExcelTitle(CallbackQueryReq req) {
        log.info("[op: change flow export callback queryExcelTitle] req={}", JSON.toJSONString(req));
        ChangeFlowExportQuery query = JSON.parseObject(req.getParamJson(), ChangeFlowExportQuery.class);
        ChangeFlowListQueryReq changeFlowListQueryReq = new ChangeFlowListQueryReq();
        BeanUtils.copyProperties(query, changeFlowListQueryReq);
        
        // 处理 changeExecUser 参数（与 query 方法保持一致）
        if (StringUtils.isNotBlank(changeFlowListQueryReq.getChangeExecUser())) {
            List<Long> recordIds = changeFlowExecService.queryByExecUser(changeFlowListQueryReq.getChangeExecUser());
            changeFlowListQueryReq.setChangeRecordIds(recordIds);
            log.info("[queryExcelTitle] recordIds: {}", recordIds);
            if (CollectionUtils.isEmpty(recordIds)) {
                ExcelTitle excelTitle = ExcelUtil.fetchExceltile(ChangeFlowExcelVo.class);
                excelTitle.setTotal(0L);
                return excelTitle;
            }
        }
        
        List<ChangeRecord> list = changeRecordMapper.selectByCondition(changeFlowListQueryReq);
        ExcelTitle excelTitle = ExcelUtil.fetchExceltile(ChangeFlowExcelVo.class);
        excelTitle.setTotal((long) list.size());
        log.info("[op: change flow export callback queryExcelTitle] result={}", JSON.toJSONString(excelTitle));
        return excelTitle;
    }

    @Override
    public List<List<ExcelCell>> queryExcelData(CallbackQueryReq req) {
        log.info("[op: change flow export callback queryExcelData] req={}", JSON.toJSONString(req));
        ChangeFlowExportQuery query = JSON.parseObject(req.getParamJson(), ChangeFlowExportQuery.class);
        query.setOffset(req.getOffset());
        query.setLimit(req.getLimit());
        List<ChangeFlowExcelDTO> list = getExportList(query);
        log.debug("[op: change flow export callback queryExcelData] size:{}", list.size());
        return list.stream().map(ChangeFlowExcelVo::init).collect(Collectors.toList());
    }

    private List<ChangeFlowExcelDTO> getExportList(ChangeFlowExportQuery query) {
        ChangeFlowListQueryReq changeFlowListQueryReq = new ChangeFlowListQueryReq();
        BeanUtils.copyProperties(query, changeFlowListQueryReq);
        
        // 处理 changeExecUser 参数（与 query 方法保持一致）
        if (StringUtils.isNotBlank(changeFlowListQueryReq.getChangeExecUser())) {
            List<Long> recordIds = changeFlowExecService.queryByExecUser(changeFlowListQueryReq.getChangeExecUser());
            changeFlowListQueryReq.setChangeRecordIds(recordIds);
            log.info("[getExportList] recordIds: {}", recordIds);
            if (CollectionUtils.isEmpty(recordIds)) {
                return Lists.newArrayList();
            }
        }
        
        List<ChangeRecord> changeRecords = changeRecordMapper.selectByConditionAndLimit(changeFlowListQueryReq);
        log.debug("[getExportList] changeRecords:{}", JSON.toJSONString(changeRecords));
        if (CollectionUtils.isEmpty(changeRecords)) {
            return Lists.newArrayList();
        }
        // 过滤不为供应商的创建人
        List<String> creatorList = changeRecords.stream()
            .filter(c -> !CreateSourceEnum.TONG_ZHOU.getType().equals(c.getCreateSource()))
            .map(ChangeRecord::getCreator).collect(Collectors.toList());
        List<String> allQueryDepartmentList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(creatorList)) {
            allQueryDepartmentList.addAll(creatorList);
        }
        List<String> changeCommanderList = changeRecords.stream().map(ChangeRecord::getChangeCommander).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(changeCommanderList)) {
            allQueryDepartmentList.addAll(changeCommanderList);
        }
        HashMap<String, List<SecondaryDepartments>> orgMap = null;
        if (CollectionUtils.isNotEmpty(allQueryDepartmentList)) {
            // 查询创建人部门名
            IusDepartmentReq iusDepartmentReq = new IusDepartmentReq();
            iusDepartmentReq.setUids(new ArrayList<>(allQueryDepartmentList));
            iusDepartmentReq.setIcac(true);
            orgMap = iusService.queryDepartment(iusDepartmentReq);
        }
        // 提前查询所有创建人及行动项用户名
        List<String> allExecUserEmailList = new ArrayList<>();
        for (ChangeRecord changeRecord : changeRecords) {
            List<ChangeExecRecord> changeExecRecords = changeExecRecordMapper.selectByChangeRecordId(changeRecord.getId());
            List<String> collect = changeExecRecords.stream()
                    .map(ChangeExecRecord::getChangeExecUserEmail)
                    .filter(StringUtils::isNotBlank).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(collect)) {
                allExecUserEmailList.addAll(collect);
            }
        }
        List<IusUserInfoRsp> userListInfo = new ArrayList<>();
        try {
            if (CollectionUtils.isNotEmpty(allExecUserEmailList)) {
                List<String> userEmailList = allExecUserEmailList.stream().distinct().collect(Collectors.toList());
                userListInfo = iusService.queryUserListInfo(UserQueryDTO.builder().uids(userEmailList).build());
            }
        } catch (Exception e) {
            log.info("query user info has ex", e);
        }
        // 提前查询所有商品
        List<Long> allItemList = new ArrayList<>();
        List<String> allChangeProduct = changeRecords.stream()
            .filter(changeRecord -> changeRecord.getChangeSubject().equals(ChangeSubjectEnum.PRODUCT.getType()))
            .map(ChangeRecord::getChangeItem).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(allChangeProduct)) {
            for (String changeItemJsonStr : allChangeProduct) {
                List<ItemVO> itemVOS = JSON.parseArray(changeItemJsonStr, ItemVO.class);
                List<Long> itemIds = itemVOS.stream().map(ItemVO::getItemId).collect(Collectors.toList());
                allItemList.addAll(itemIds);
            }
        }
        log.debug("[getExportList] allItemList:{}", allItemList);
        List<SpuTO> spuTOS = new ArrayList<>();
        Map<Long, List<SimplePhyCateGoryResultCo>> categoryChain = new HashMap<>();
        Map<Long, ChangeGoodsPrincipalPO> longChangeGoodsPrincipalPOMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(allItemList)) {
            allItemList = allItemList.stream().distinct().collect(Collectors.toList());
            // 批量查询spu信息
            spuTOS = itemService.batchQuerySpuInfo(BatchQuerySpuInfoParam.builder().ids(allItemList)
                    .commonProps(new ArrayList<>()).spuProps(Arrays.asList("itemSetupType", "businessForm")).build());
            // 批量查询物理类目
            categoryChain = itemService
                    .queryBySpuIds(CommonIdsParamQuery.builder().ids(allItemList).build());
            // 查询商品对应负责人信息
            longChangeGoodsPrincipalPOMap = interiorChangeConfigService
                    .queryGoodsPrincipalInfo(allItemList);
            log.debug("[getExportList] spuTOS:{}", spuTOS.size());
        }
        List<ChangeFlowExcelDTO> changeFlowExcelDTOList = new ArrayList<>();
        // 导出excel，按照商品/供应商和行动项，n*n重复拆分，例2个商品3个行动项，拆分成2*3=6行，其他属性一致
        for (ChangeRecord record : changeRecords) {
            List<ChangeExecRecord> changeExecRecords = changeExecRecordMapper.selectByChangeRecordId(record.getId());
            ChangeSubjectEnum changeSubjectEnum = ChangeSubjectEnum.getChangeSubjectEnum(record.getChangeSubject());
            switch (changeSubjectEnum) {
                case PRODUCT:
                    String changeItemJsonStr = record.getChangeItem();
                    List<ItemVO> itemVOS = JSON.parseArray(changeItemJsonStr, ItemVO.class);
                    for (ItemVO itemVO : itemVOS) {
                        for (ChangeExecRecord changeExecRecord : changeExecRecords) {
                            Long itemId = itemVO.getItemId();
                            // 组装通用信息，除商品信息，行动项以外其他字段一致
                            ChangeFlowExcelDTO changeFlowExcelDTO = this.obtainCommon(record, changeSubjectEnum, orgMap);
                            String changeSupplier = StringUtils.isBlank(record.getChangeSupplier()) ? "" : record.getChangeSupplier();
                            String changeSupplierName = "";
                            if (StringUtils.isNotBlank(changeSupplier)) {
                                List<SupplierSimpleRsp> supplierSimple = supplierService.getSupplierName(changeSupplier);
                                if (CollectionUtils.isNotEmpty(supplierSimple)) {
                                    SupplierSimpleRsp supplier = supplierSimple.get(0);
                                    changeSupplierName = supplier.getSupplierName();
                                }
                            }
                            changeFlowExcelDTO.setSupplier(
                                    StringUtils.isBlank(changeSupplier) ? "/" : changeSupplier + changeSupplierName);
                            changeFlowExcelDTO.setChangeItem(itemVO.getItemName());
                            changeFlowExcelDTO.setSPUId(itemId);
                            List<QcCategoryVO> qcCategoryList = qcService.getQcCategoryListByItemId(itemId);
                            if (CollectionUtils.isNotEmpty(qcCategoryList)) {
                                String collect = qcCategoryList.stream().map(QcCategoryVO::getCategoryName)
                                        .collect(Collectors.joining("/"));
                                changeFlowExcelDTO.setQcCategory(collect);
                            } else {
                                changeFlowExcelDTO.setQcCategory("/");
                            }
                            List<SimplePhyCateGoryResultCo> simplePhyCateGoryResultCos = categoryChain.get(itemId);
                            if (CollectionUtils.isNotEmpty(simplePhyCateGoryResultCos)) {
                                String collect = simplePhyCateGoryResultCos.stream()
                                    .map(SimplePhyCateGoryResultCo::getName).collect(Collectors.joining("/"));
                                changeFlowExcelDTO.setPhyCategory(collect);
                            } else {
                                changeFlowExcelDTO.setPhyCategory("/");
                            }
                            Optional<SpuTO> optionalSpuTO = spuTOS.stream().filter(spu -> spu.getId() == itemId).findAny();
                            if (!optionalSpuTO.isPresent()) {
                                changeFlowExcelDTO.setItemSetupType(null);
                                changeFlowExcelDTO.setBusinessForm(null);
                                changeFlowExcelDTO.setStatus(null);
                            } else {
                                SpuTO spuTO = optionalSpuTO.get();
                                Map<String, String> propertyMap = spuTO.getPropertyMap();
                                Integer itemSetupType = Integer.valueOf(propertyMap.get("itemSetupType"));
                                changeFlowExcelDTO.setItemSetupType(ItemSetupTypeEnum.getByType(itemSetupType));
                                String businessForm = propertyMap.get("businessForm");
                                int business = StringUtils.isBlank(businessForm) ? 0 : Integer.parseInt(businessForm);
                                changeFlowExcelDTO.setBusinessForm(BusinessFormEnum.getByType(business));
                                changeFlowExcelDTO.setStatus(MainSpuStatusEnum.getByStatus(spuTO.getStatus()));
                            }
                            ChangeGoodsPrincipalPO changeGoodsPrincipalPO = longChangeGoodsPrincipalPOMap.get(itemId);
                            try {
                                if (null == changeGoodsPrincipalPO) {
                                    changeFlowExcelDTO.setFunctionary("/");
                                    changeFlowExcelDTO.setPurchase("/");
                                    changeFlowExcelDTO.setGoodsProject("/");
                                    changeFlowExcelDTO.setGoodsSqe("/");
                                    changeFlowExcelDTO.setGoodsCompliance("/");
                                } else {
                                    buildGoodsPrincipal(changeGoodsPrincipalPO, changeFlowExcelDTO);
                                }
                            } catch (Exception e) {
                                log.debug("[getExportList] changeGoodsPrincipalPO has ex:{}",
                                    JSON.toJSONString(changeGoodsPrincipalPO));
                            }
                            // 组装行动项相关信息
                            obtainChangeExec(changeExecRecord, changeFlowExcelDTO, userListInfo);
                            this.obtainOther(record, changeFlowExcelDTO);
                            changeFlowExcelDTOList.add(changeFlowExcelDTO);
                        }
                    }
                    break;
                case SUPPLIER:
                case OTHER:
                default:
                    // 供应商变更，以行动项循环
                    for (ChangeExecRecord changeExecRecord : changeExecRecords) {
                        // 组装通用信息，除商品信息，行动项以外其他字段一致
                        ChangeFlowExcelDTO changeFlowExcelDTO = this.obtainCommon(record, changeSubjectEnum, orgMap);
                        String changeSupplier = StringUtils.isBlank(record.getChangeSupplier()) ? "" : record.getChangeSupplier();
                        String changeSupplierName = "";
                        if (StringUtils.isNotBlank(changeSupplier)) {
                            List<SupplierSimpleRsp> supplierSimple = supplierService.getSupplierName(changeSupplier);
                            if (CollectionUtils.isNotEmpty(supplierSimple)) {
                                SupplierSimpleRsp supplier = supplierSimple.get(0);
                                changeSupplierName = supplier.getSupplierName();
                            }
                        }
                        changeFlowExcelDTO.setSupplier(
                            StringUtils.isBlank(changeSupplier) ? "/" : changeSupplier + changeSupplierName);
                        changeFlowExcelDTO.setChangeItem(null);
                        changeFlowExcelDTO.setSPUId(null);
                        changeFlowExcelDTO.setQcCategory(null);
                        changeFlowExcelDTO.setPhyCategory(null);
                        changeFlowExcelDTO.setItemSetupType(null);
                        changeFlowExcelDTO.setBusinessForm(null);
                        changeFlowExcelDTO.setStatus(null);
                        changeFlowExcelDTO.setFunctionary(null);
                        changeFlowExcelDTO.setPurchase(null);
                        changeFlowExcelDTO.setGoodsSqe(null);
                        changeFlowExcelDTO.setGoodsProject(null);
                        changeFlowExcelDTO.setGoodsCompliance(null);
                        // 组装行动项相关信息
                        this.obtainChangeExec(changeExecRecord, changeFlowExcelDTO, userListInfo);
                        this.obtainOther(record, changeFlowExcelDTO);
                        changeFlowExcelDTOList.add(changeFlowExcelDTO);
                    }
                    break;
            }

        }
        return changeFlowExcelDTOList;

    }

    private void obtainOther(ChangeRecord record, ChangeFlowExcelDTO changeFlowExcelDTO) {
        ChangeResultEnum resultEnum = ChangeResultEnum.getByValue(record.getChangeResult());
        log.debug("[obtainOther] record:{}, resultEnum:{}", JSON.toJSONString(record), resultEnum);
        changeFlowExcelDTO.setChangeResult(resultEnum == null ? "/" : resultEnum.getDesc());
        if (record.getState().equals(ChangeStatusEnum.CANCEL.getStatus())
                || record.getState().equals(ChangeStatusEnum.END.getStatus())) {
            changeFlowExcelDTO.setChangeEndTime(DateUtils.parseLongToString(record.getUpdateTime(), DateUtils.DATE_TIME_FORMAT));
        } else {
            changeFlowExcelDTO.setChangeEndTime("/");
        }
    }

    private void buildGoodsPrincipal(ChangeGoodsPrincipalPO changeGoodsPrincipalPO, ChangeFlowExcelDTO changeFlowExcelDTO) {
        String functionaryName = StringUtils.isBlank(changeGoodsPrincipalPO.getFunctionaryName()) ? "" : changeGoodsPrincipalPO.getFunctionaryName();
        String functionaryEmail = StringUtils.isBlank(changeGoodsPrincipalPO.getFunctionaryEmail()) ? "" : changeGoodsPrincipalPO.getFunctionaryEmail();
        String purchaseName = StringUtils.isBlank(changeGoodsPrincipalPO.getPurchaseName()) ? "" : changeGoodsPrincipalPO.getPurchaseName();
        String purchaseEmail = StringUtils.isBlank(changeGoodsPrincipalPO.getPurchaseEmail()) ? "" : changeGoodsPrincipalPO.getPurchaseEmail();
        String goodsProjectName = StringUtils.isBlank(changeGoodsPrincipalPO.getGoodsProjectName()) ? "" : changeGoodsPrincipalPO.getGoodsProjectName();
        String goodsProjectEmail = StringUtils.isBlank(changeGoodsPrincipalPO.getGoodsProjectEmail()) ? "" : changeGoodsPrincipalPO.getGoodsProjectEmail();
        String goodsSqeName = StringUtils.isBlank(changeGoodsPrincipalPO.getGoodsSqeName()) ? "" : changeGoodsPrincipalPO.getGoodsSqeName();
        String goodsSqeEmail = StringUtils.isBlank(changeGoodsPrincipalPO.getGoodsSqeEmail()) ? "" : changeGoodsPrincipalPO.getGoodsSqeEmail();
        String goodsComplianceName = StringUtils.isBlank(changeGoodsPrincipalPO.getGoodsComplianceName()) ? "" : changeGoodsPrincipalPO.getGoodsComplianceName();
        String goodsComplianceEmail = StringUtils.isBlank(changeGoodsPrincipalPO.getGoodsComplianceEmail()) ? "" : changeGoodsPrincipalPO.getGoodsComplianceEmail();

        String functionary = functionaryName + functionaryEmail;
        String purchase = purchaseName + purchaseEmail;
        String goodsProject = goodsProjectName + goodsProjectEmail;
        String goodsSqe = goodsSqeName + goodsSqeEmail;
        String goodsCompliance = goodsComplianceName + goodsComplianceEmail;
        changeFlowExcelDTO.setFunctionary(StringUtils.isBlank(functionary) ? "/" :functionary);
        changeFlowExcelDTO.setPurchase(StringUtils.isBlank(purchase) ? "/" :purchase);
        changeFlowExcelDTO.setGoodsProject(StringUtils.isBlank(goodsProject) ? "/" :goodsProject);
        changeFlowExcelDTO.setGoodsSqe(StringUtils.isBlank(goodsSqe) ? "/" :goodsSqe);
        changeFlowExcelDTO.setGoodsCompliance(StringUtils.isBlank(goodsCompliance) ? "/" :goodsCompliance);
    }

    private void obtainChangeExec(ChangeExecRecord changeExecRecord, ChangeFlowExcelDTO changeFlowExcelDTO,
        List<IusUserInfoRsp> userListInfo) {
        changeFlowExcelDTO.setExecDepartment(changeExecRecord.getChangeExecDepartment());
        Optional<IusUserInfoRsp> anyExeUser = userListInfo.stream()
            .filter(u -> u.getUid().equals(changeExecRecord.getChangeExecUserEmail())).findAny();
        String execUserName;
        if (anyExeUser.isPresent()) {
            execUserName = anyExeUser.get().getName();
        } else {
            // 批量查询没查到的人名，通过全量查询单独再查一次
            IusUserInfoRsp user = iusService.queryUserInfo(changeExecRecord.getChangeExecUserEmail());
            execUserName = user == null || StringUtils.isBlank(user.getName()) ? "" : user.getName();
        }
        changeFlowExcelDTO.setChangeExecUser(execUserName + changeExecRecord.getChangeExecUserEmail());
        changeFlowExcelDTO.setChangeExecRiskDesc(changeExecRecord.getChangeRiskDesc());
        changeFlowExcelDTO.setChangeExecProject(changeExecRecord.getChangeExecProject());
        changeFlowExcelDTO.setChangeChecking(
            StringUtils.isBlank(changeExecRecord.getChangeChecking()) ? "/" : changeExecRecord.getChangeChecking());
        Long changeExecFinishTime = changeExecRecord.getChangeExecFinishTime();
        if (changeExecFinishTime == 0L) {
            changeFlowExcelDTO.setChangeExecFinishTime("/");
        } else {
            changeFlowExcelDTO.setChangeExecFinishTime(
                    DateUtils.parseLongToString(changeExecFinishTime, DateUtils.DATE_TIME_FORMAT));
        }
        changeFlowExcelDTO.setChangeExecFinishDesc(StringUtils.isBlank(changeExecRecord.getChangeExecFinishDesc()) ? "/"
            : changeExecRecord.getChangeExecFinishDesc());
    }

    /**
     * 组装通用字段
     * @param record
     * @param changeSubjectEnum
     * @return
     */
    private ChangeFlowExcelDTO obtainCommon(ChangeRecord record, ChangeSubjectEnum changeSubjectEnum, HashMap<String, List<SecondaryDepartments>> orgMap) {
        ChangeFlowExcelDTO changeFlowExcelDTO = new ChangeFlowExcelDTO();
        changeFlowExcelDTO.setFlowId(record.getFlowId());
        String createTimeStr = DateUtils.parseLongToString(record.getCreateTime(), DateUtils.DATE_TIME_FORMAT);
        changeFlowExcelDTO.setCreateTime(createTimeStr);
        // 供应商发起，展示供应商id，否则展示：用户名(部门名)
        if (CreateSourceEnum.TONG_ZHOU.getType().equals(record.getCreateSource())) {
            changeFlowExcelDTO.setChangeCreator(record.getCreateSupplier());
        } else {
            List<SecondaryDepartments> creator = orgMap.get(record.getCreator());
            if (!CollectionUtils.isEmpty(creator)) {
                Optional<SecondaryDepartments> anyOrg = creator.stream().filter(o -> o.getOrgPosLevel() == 97).findAny();
                String orgName = "无三级部门";
                if (anyOrg.isPresent()) {
                    orgName = anyOrg.get().getOrgPosName();
                }
                changeFlowExcelDTO.setChangeCreator(creator.get(0).getUserName() + "(" + orgName + ")");
            }
        }
        changeFlowExcelDTO.setChangeSubject(changeSubjectEnum.getDesc());
        ChangeType parentChangeType = changeTypeService.getChangeTypeById(record.getParentChangeClassId());
        ChangeType sonChangeType = changeTypeService.getChangeTypeById(record.getSonChangeClassId());
        changeFlowExcelDTO.setChangeType(parentChangeType.getTypeName() + ">" + sonChangeType.getTypeName());
        changeFlowExcelDTO.setChangeLevel(ChangeLevelEnum.getByType(record.getChangeLevel()));
        changeFlowExcelDTO.setChangeDepartment(record.getChangeDepartment());
        List<SecondaryDepartments> commander = orgMap.get(record.getChangeCommander());
        if (!CollectionUtils.isEmpty(commander)) {
            Optional<SecondaryDepartments> anyOrg = commander.stream().filter(o -> o.getOrgPosLevel() == 97).findAny();
            String orgName = "无三级部门";
            if (anyOrg.isPresent()) {
                orgName = anyOrg.get().getOrgPosName();
            }
            changeFlowExcelDTO.setChangeCommander(commander.get(0).getUserName() + "(" + orgName + ")");
        }
        changeFlowExcelDTO.setChangeReason(record.getChangeReason());
        changeFlowExcelDTO.setChangeContent(record.getChangeContent());
        changeFlowExcelDTO
            .setChangeRiskDesc(StringUtils.isBlank(record.getChangeRiskDesc()) ? "/" : record.getChangeRiskDesc());
        changeFlowExcelDTO.setChangeProfit(ProfitTypeEnum.getByType(record.getChangeProfit()));
        changeFlowExcelDTO.setChangeProfitDesc(record.getChangeProfitDesc());
        return changeFlowExcelDTO;
    }

    @Override
    public PivotTable queryPivotTable(CallbackQueryReq callbackQueryReq) {
        return null;
    }
}