package com.netease.mail.yanxuan.change.biz.biz;

import com.alibaba.fastjson.JSON;
import com.netease.mail.yanxuan.change.biz.meta.exception.ExceptionFactory;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowExecService;
import com.netease.mail.yanxuan.change.biz.service.ChangeFlowService;
import com.netease.mail.yanxuan.change.biz.service.ChangeSubFlowRecordService;
import com.netease.mail.yanxuan.change.biz.service.ChangeSubFlowService;
import com.netease.mail.yanxuan.change.biz.service.rpc.FlowService;
import com.netease.mail.yanxuan.change.biz.service.rpc.IusService;
import com.netease.mail.yanxuan.change.integration.flow.ius.IusRpcService;
import com.netease.mail.yanxuan.change.common.bean.CommonConstants;
import com.netease.mail.yanxuan.change.biz.util.PageUtils;
import com.netease.mail.yanxuan.change.common.bean.RequestLocalBean;
import com.netease.mail.yanxuan.change.common.bean.ResponseCode;
import com.netease.mail.yanxuan.change.common.enums.ChangeFlowEnum;
import com.netease.mail.yanxuan.change.common.enums.ChangeSubFlowStatusEnum;
import com.netease.mail.yanxuan.change.common.enums.FlowOperationTypeEnum;
import com.netease.mail.yanxuan.change.common.enums.FlowxOperationEnum;
import com.netease.mail.yanxuan.change.common.util.DateUtils;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.netease.mail.yanxuan.change.dal.entity.ChangeExecRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeRecord;
import com.netease.mail.yanxuan.change.dal.entity.ChangeSubFlowRecord;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeRecordMapper;
import com.netease.mail.yanxuan.change.dal.mapper.ChangeSubFlowRecordMapper;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeSubFlowCreateReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeSubFlowListQueryReq;
import com.netease.mail.yanxuan.change.dal.meta.model.req.ChangeSubFlowSubmitReq;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeFlowExecVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeFlowVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeSubFlowListVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.ChangeSubFlowVO;
import com.netease.mail.yanxuan.change.dal.meta.model.vo.PageVO;
import com.netease.mail.yanxuan.change.integration.flow.UserQueryDTO;
import com.netease.mail.yanxuan.change.integration.flow.ius.rsp.IusUserInfoRsp;
import com.netease.yanxuan.flowx.sdk.meta.controller.communal.AjaxResponse;
import com.netease.yanxuan.flowx.sdk.meta.dto.base.FlowDataDTO;
import com.netease.yanxuan.flowx.sdk.meta.dto.flow.FlowCreateReqDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import javax.validation.Valid;
import java.util.*;
import java.util.stream.Collectors;

@Component
@Slf4j
public class ChangeSubFlowBiz {

    @Autowired
    private ChangeSubFlowService changeSubFlowService;

    @Autowired
    private FlowService flowService;

    @Autowired
    private ChangeFlowExecService changeFlowExecService;

    @Autowired
    private ChangeFlowService changeFlowService;

    @Autowired
    private ChangeSubFlowRecordService changeSubFlowRecordService;

    @Autowired
    private ChangeSubFlowRecordMapper changeSubFlowRecordMapper;

    @Autowired
    private ChangeRecordMapper changeRecordMapper;

    @Autowired
    private ChangeFlowBiz changeFlowBiz;

    @Autowired
    private IusRpcService iusRpcService;

    @Autowired
    private IusService iusService;


    public String createAndSubmit(@Valid ChangeSubFlowCreateReq changeSubFlowCreateReq) {
        String uid = RequestLocalBean.getUid();
        String name = RequestLocalBean.getName();
        Map<String, Object> content = new HashMap<>(10);
        content.put("createUserName", uid);
        content.put("createUser", uid);
        content.put("createTime", System.currentTimeMillis());
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.PASS.getValue());
        String flowName = "";
        // 组装工单创建数据
        FlowCreateReqDTO flowCreateReqDTO = buildFlowCreateReqDTO(ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), uid,
                JSON.toJSONString(content), FlowxOperationEnum.CREATE.getName(), name, flowName);
        // 创建工单
        String subFlowId = flowService.createFlow(flowCreateReqDTO);

        ChangeExecRecord execRecord = new ChangeExecRecord();
        changeSubFlowService.createSubFlow(execRecord);
        return "";
    }

    private FlowCreateReqDTO buildFlowCreateReqDTO(String topoId, String uid, String content, String operateResult,
                                                   String name, String flowName) {
        FlowCreateReqDTO flowCreateReqDTO = new FlowCreateReqDTO();
        flowCreateReqDTO.setTopoId(topoId);
        flowCreateReqDTO.setUid(uid);
        flowCreateReqDTO.setUserName(name);
        flowCreateReqDTO.setOperateResult(operateResult);
        flowCreateReqDTO.setWorkOrderId(StringUtils.joinWith("-", topoId, UUID.randomUUID().toString()));
        flowCreateReqDTO.setContent(content);
        flowCreateReqDTO.setFlowName(flowName);
        return flowCreateReqDTO;
    }

    public List<ChangeExecRecord> list(Integer page, Integer pageSize) {
        return changeSubFlowService.query((page - 1) * pageSize, pageSize);
    }

    /**
     * 子流程提交
     * @param changeSubFlowSubmitReq 子流程提交请求
     * @return 下一节点ID
     */
    public String submitSubFlow(ChangeSubFlowSubmitReq changeSubFlowSubmitReq) {
        log.info("[submitSubFlow] changeSubFlowSubmitReq:{}", JSON.toJSONString(changeSubFlowSubmitReq));
        String subFlowId = changeSubFlowSubmitReq.getSubFlowId();
        
        // 通过子流程工单ID查询行动工单记录
        ChangeSubFlowRecord subFlowRecord = changeSubFlowRecordService.getBySubFlowId(subFlowId);
        Assert.notNull(subFlowRecord, "行动工单记录不存在");
        
        // 检查工单节点（从行动工单获取）
        String currentNode = subFlowRecord.getSubFlowNode();
        this.checkNode(currentNode, Collections.singletonList(changeSubFlowSubmitReq.getCurrentNodeId()));
        
        // 通过行动工单记录ID查询行动项记录（取第一个）
        List<ChangeExecRecord> execRecords = changeFlowExecService.getBySubFlowRecordId(subFlowRecord.getId());
        Assert.notEmpty(execRecords, "行动项记录不存在");
        ChangeExecRecord execRecord = execRecords.get(0);
        
        String uid = RequestLocalBean.getUid();
        // 验证执行人权限
        if (!uid.equals(execRecord.getChangeExecUserEmail())) {
            throw ExceptionFactory.createBiz(ResponseCode.NO_AUTH, ResponseCode.NO_AUTH.getMsg());
        }
        
        // 获取工单详情
        FlowDataDTO flowDataDTO = flowService.flowDetail(subFlowId);
        if (flowDataDTO == null) {
            throw ExceptionFactory.createBiz(ResponseCode.DETAIL_FLOW_ERROR, "子流程工单查询错误，不存在");
        }
        
        return checkUpdateAndSubmit(subFlowId, flowDataDTO, uid, execRecord, subFlowRecord, currentNode, changeSubFlowSubmitReq);
    }

    /**
     * 检查并提交子流程节点
     */
    private String checkUpdateAndSubmit(String subFlowId, FlowDataDTO flowDataDTO, String uid, 
                                       ChangeExecRecord execRecord, ChangeSubFlowRecord subFlowRecord, 
                                       String currentNode, ChangeSubFlowSubmitReq changeSubFlowSubmitReq) {
        ChangeFlowEnum node = ChangeFlowEnum.getByNodeId(currentNode);
        Assert.notNull(node, "节点配置不存在");
        log.debug("[checkUpdateAndSubmit] subFlowId:{}, nodeEnum:{}, changeSubFlowSubmitReq:{}", 
                subFlowId, node, JSON.toJSONString(changeSubFlowSubmitReq));
        
        // 构建工单内容
        Map<String, Object> content = new HashMap<>(CommonConstants.INIT_HASH_MAP_SIZE);
        content.put("updateTime", System.currentTimeMillis());
        content.put(CommonConstants.FLOW_OPERATION_KEY, FlowOperationTypeEnum.PASS.getValue());
        
        switch (node) {
            case CHANGE_SUB_FLOW_START:
                // 确认行动方案节点，直接提交到下一节点
                String nextNodeId = flowService.submitFlow(subFlowId, flowDataDTO, uid,
                        ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), JSON.toJSONString(content), true,
                        FlowxOperationEnum.SUBMIT.getName(), "提交工单", execRecord.getCreateTime());
                // 更新行动工单节点ID
                subFlowRecord.setSubFlowNode(nextNodeId);
                // TODO: 根据业务逻辑设置状态 subFlowRecord.setStatus(...)
                subFlowRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeSubFlowRecordService.update(subFlowRecord);
                return nextNodeId;
                
            case CHANGE_SUB_FLOW_SUBMIT:
                // 审批行动方案节点，直接提交到执行节点
                String execNodeId = flowService.submitFlow(subFlowId, flowDataDTO, uid,
                        ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), JSON.toJSONString(content), true,
                        FlowxOperationEnum.SUBMIT.getName(), "提交工单", execRecord.getCreateTime());
                // 更新行动工单节点ID
                subFlowRecord.setSubFlowNode(execNodeId);
                // TODO: 根据业务逻辑设置状态 subFlowRecord.setStatus(...)
                subFlowRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeSubFlowRecordService.update(subFlowRecord);
                return execNodeId;
                
            case CHANGE_SUB_FLOW_EXE:
                // 执行行动方案节点，直接提交到提交执行结果节点
                String confirmNodeId = flowService.submitFlow(subFlowId, flowDataDTO, uid,
                        ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), JSON.toJSONString(content), true,
                        FlowxOperationEnum.SUBMIT.getName(), "提交工单", execRecord.getCreateTime());
                // 更新行动工单节点ID
                subFlowRecord.setSubFlowNode(confirmNodeId);
                // TODO: 根据业务逻辑设置状态 subFlowRecord.setStatus(...)
                subFlowRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeSubFlowRecordService.update(subFlowRecord);
                return confirmNodeId;
                
            case CHANGE_SUB_FLOW_CONFIRM:
                // 提交执行结果节点，需要填写完成时间和完成情况
                Long finishTime = changeSubFlowSubmitReq.getChangeExecFinishTime();
                String finishDesc = changeSubFlowSubmitReq.getChangeExecFinishDesc();
                Assert.notNull(finishTime, "行动完成时间不可为空");
                Assert.isTrue(StringUtils.isNotBlank(finishDesc), "行动完成情况不可为空");
                
                // 更新完成信息（更新所有关联的行动项）
                List<ChangeExecRecord> allExecRecords = changeFlowExecService.getBySubFlowRecordId(subFlowRecord.getId());
                for (ChangeExecRecord exec : allExecRecords) {
                    exec.setChangeExecFinishTime(finishTime);
                    exec.setChangeExecFinishDesc(finishDesc);
                    exec.setUpdateTime(DateUtils.getCurrentTime());
                    changeFlowExecService.update(exec);
                }
                
                // 提交到结束节点
                String endNodeId = flowService.submitFlow(subFlowId, flowDataDTO, uid,
                        ChangeFlowEnum.CHANGE_SUB_FLOW.getTopoId(), JSON.toJSONString(content), true,
                        FlowxOperationEnum.SUBMIT.getName(), "提交工单", execRecord.getCreateTime());
                // 更新行动工单节点ID
                subFlowRecord.setSubFlowNode(endNodeId);
                // TODO: 根据业务逻辑设置状态 subFlowRecord.setStatus(...)
                subFlowRecord.setUpdateTime(DateUtils.getCurrentTime());
                changeSubFlowRecordService.update(subFlowRecord);
                
                // TODO: 子流程完成后，检查主流程是否可以进入确认节点
                log.info("[CHANGE_SUB_FLOW_CONFIRM] 子流程完成，subFlowId:{}, subFlowRecordId:{}", subFlowId, subFlowRecord.getId());
                
                return endNodeId;
                
            default:
                throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, "不支持的子流程节点类型");
        }
    }

    /**
     * 检查节点是否匹配
     */
    private void checkNode(String currentNode, List<String> expectedNodes) {
        if (!expectedNodes.contains(currentNode)) {
            throw ExceptionFactory.createBiz(ResponseCode.BAD_REQUEST, 
                    String.format("当前节点[%s]与预期节点[%s]不匹配", currentNode, expectedNodes));
        }
    }

    /**
     * 根据变更行动工单ID查询详情
     * 说明：展示内容和主单的detail一样，但行动项只展示该变更行动工单下绑定的行动项
     *
     * @param subFlowId 变更行动工单ID（即topo）
     * @return 变更工单详情
     */
    public ChangeFlowVO detailBySubFlowId(String subFlowId) {
        if (StringUtils.isBlank(subFlowId)) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "变更行动工单ID不能为空");
        }

        // 根据 subFlowId 查询变更行动工单记录
        ChangeSubFlowRecord subFlowRecord = changeSubFlowRecordService.getBySubFlowId(subFlowId);
        if (subFlowRecord == null) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "变更行动工单不存在");
        }

        Long changeRecordId = subFlowRecord.getChangeRecordId();
        Long subFlowRecordId = subFlowRecord.getId();

        // 获取主单信息
        ChangeRecord changeRecord = changeFlowService.getById(changeRecordId);
        if (changeRecord == null) {
            throw ExceptionFactory.createBiz(ResponseCode.ERROR_FLOW_ID, "主单记录不存在");
        }

        // 调用主单的 detail 方法获取完整信息
        ChangeFlowVO changeFlowVO = changeFlowBiz.detail(changeRecord.getFlowId());

        // 只保留该变更行动工单下绑定的行动项
        List<ChangeExecRecord> execRecords = changeFlowExecService.getBySubFlowRecordId(subFlowRecordId);
        if (CollectionUtils.isNotEmpty(execRecords)) {
            // 转换为 ChangeFlowExecVO
            List<ChangeFlowExecVO> filteredExecList = execRecords.stream()
                    .map(c -> ChangeFlowExecVO.builder()
                            .changeExecId(c.getId())
                            .changeRecordId(c.getChangeRecordId())
                            .changeExecDepartment(c.getChangeExecDepartment())
                            .changeExecUserType(c.getChangeExecUserType())
                            .changeExecUser(c.getChangeExecUser())
                            .changeExecUserEmail(c.getChangeExecUserEmail())
                            .changeRiskDesc(c.getChangeRiskDesc())
                            .changeExecProject(c.getChangeExecProject())
                            .changeChecking(c.getChangeChecking())
                            .changeExecFinishTime(c.getChangeExecFinishTime())
                            .changeExecFinishDesc(c.getChangeExecFinishDesc())
                            .subFlowId(c.getSubFlowId())
                            .build())
                    .collect(Collectors.toList());

            // 填充用户名信息
            try {
                Set<String> userEmail = filteredExecList.stream()
                        .map(ChangeFlowExecVO::getChangeExecUserEmail)
                        .collect(Collectors.toSet());
                AjaxResponse<List<IusUserInfoRsp>> userListInfo = iusRpcService.queryUserListInfo(
                        UserQueryDTO.builder().uids(new ArrayList<>(userEmail)).build());
                List<IusUserInfoRsp> data = userListInfo.getData();
                filteredExecList.forEach(i -> {
                    Optional<IusUserInfoRsp> anyExeUser = data.stream()
                            .filter(u -> u.getUid().equals(i.getChangeExecUserEmail()))
                            .findAny();
                    if (anyExeUser.isPresent()) {
                        i.setChangeExecUserName(anyExeUser.get().getName());
                    } else {
                        // 批量查询没查到的人名，通过全量查询单独再查一次
                        IusUserInfoRsp user = iusService.queryUserInfo(i.getChangeExecUserEmail());
                        if (user != null && StringUtils.isNotBlank(user.getName())) {
                            i.setChangeExecUserName(user.getName());
                        }
                    }
                });
            } catch (Exception ex) {
                log.error("query user info has ex", ex);
            }

            changeFlowVO.setChangeExecProjectList(filteredExecList);
        } else {
            changeFlowVO.setChangeExecProjectList(new ArrayList<>());
        }

        return changeFlowVO;
    }

    /**
     * 查询变更行动工单列表
     *
     * @param page 页码
     * @param pageSize 每页大小
     * @param queryReq 查询条件
     * @return 变更行动工单列表
     */
    public ChangeSubFlowListVO querySubFlowList(Integer page, Integer pageSize, ChangeSubFlowListQueryReq queryReq) {
        log.info("[querySubFlowList] page:{}, pageSize:{}, queryReq:{}", page, pageSize, JSON.toJSONString(queryReq));

        // 根据主单条件（flowId, itemId, supplier, onlyMyFollowed）查询主单，获取 changeRecordIds
        if (queryReq.getFlowId() != null || StringUtils.isNotBlank(queryReq.getItemId())
                || StringUtils.isNotBlank(queryReq.getSupplier()) || queryReq.getOnlyMyFollowed()) {
            List<Long> changeRecordIds = filterFlowRecordIds(queryReq);
            if (CollectionUtils.isEmpty(changeRecordIds)) {
                // 如果主单查询结果为空，直接返回
                PageVO pageVO = PageUtils.buildPageVo(0L, pageSize, page);
                ChangeSubFlowListVO result = new ChangeSubFlowListVO();
                result.setPageVo(pageVO);
                result.setChangeSubFlowList(new ArrayList<>());
                return result;
            }
            queryReq.setChangeRecordIds(changeRecordIds);
        }

        // 清空主单相关的查询条件，因为已经通过 changeRecordIds 过滤了
        // changeExecUser 和 changeExecDepartment 直接在子单表中查询，不需要额外处理
        queryReq.setOnlyMyFollowed(null);
        queryReq.setCurrentUser(null);

        // 进行分页查询子单
        PageHelper.startPage(page, pageSize);
        List<ChangeSubFlowRecord> subFlowRecords = changeSubFlowRecordMapper.selectByCondition(queryReq);
        PageInfo<ChangeSubFlowRecord> subFlowRecordPageInfo = new PageInfo<>(subFlowRecords);

        // 第四步：批量获取主单信息和行动项信息，构建返回列表
        List<ChangeSubFlowVO> list = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(subFlowRecords)) {
            // 批量获取主单信息
            List<Long> changeRecordIds = subFlowRecords.stream().map(ChangeSubFlowRecord::getChangeRecordId).distinct()
                .collect(Collectors.toList());
            final Map<Long, ChangeRecord> changeRecordMap;
            if (CollectionUtils.isNotEmpty(changeRecordIds)) {
                List<ChangeRecord> changeRecords = changeFlowService.getByIds(changeRecordIds);
                changeRecordMap = changeRecords.stream().filter(Objects::nonNull)
                    .collect(Collectors.toMap(ChangeRecord::getId, r -> r));
            } else {
                changeRecordMap = new HashMap<>();
            }

            // 批量获取行动项信息（用于获取行动人、部门、完成时间等）
            List<Long> subFlowRecordIds = subFlowRecords.stream().map(ChangeSubFlowRecord::getId)
                .collect(Collectors.toList());
            Map<Long, List<ChangeExecRecord>> execRecordMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(subFlowRecordIds)) {
                List<ChangeExecRecord> allExecRecords = changeFlowExecService.getBySubFlowRecordIds(subFlowRecordIds);
                if (CollectionUtils.isNotEmpty(allExecRecords)) {
                    execRecordMap = allExecRecords.stream().filter(exec -> exec.getSubFlowRecordId() != null)
                        .collect(Collectors.groupingBy(ChangeExecRecord::getSubFlowRecordId));
                }
            }

            // 构建返回列表
            for (ChangeSubFlowRecord subFlowRecord: subFlowRecords) {
                ChangeSubFlowVO vo = new ChangeSubFlowVO();
                vo.setSubFlowId(subFlowRecord.getSubFlowId());
                vo.setStatus(subFlowRecord.getStatus());
                vo.setStatusDesc(ChangeSubFlowStatusEnum.getDescByStatus(subFlowRecord.getStatus()));
                vo.setCreateTime(subFlowRecord.getCreateTime());

                // 获取主单信息
                ChangeRecord changeRecord = changeRecordMap.get(subFlowRecord.getChangeRecordId());
                if (changeRecord != null) {
                    vo.setFlowId(changeRecord.getFlowId());
                }

                // 获取行动项信息（取第一个行动项的信息作为展示）
                List<ChangeExecRecord> execRecords = execRecordMap.get(subFlowRecord.getId());
                if (CollectionUtils.isNotEmpty(execRecords)) {
                    ChangeExecRecord firstExec = execRecords.get(0);
                    vo.setChangeExecUserEmail(firstExec.getChangeExecUserEmail());
                    vo.setChangeExecDepartment(firstExec.getChangeExecDepartment());
                    // 完成时间：如果有完成时间则使用，否则使用预期完成时间（从主单获取）
                    if (firstExec.getChangeExecFinishTime() != null) {
                        vo.setChangeExecFinishTime(firstExec.getChangeExecFinishTime());
                    } else if (changeRecord != null) {
                        vo.setChangeExecFinishTime(changeRecord.getChangeConfirmResultTime());
                    }
                }

                list.add(vo);
            }

            // 批量查询用户名信息
            try {
                Set<String> userEmails = list.stream().map(ChangeSubFlowVO::getChangeExecUserEmail)
                    .filter(StringUtils::isNotBlank).collect(Collectors.toSet());
                if (CollectionUtils.isNotEmpty(userEmails)) {
                    AjaxResponse<List<IusUserInfoRsp>> userListInfo = iusRpcService
                        .queryUserListInfo(UserQueryDTO.builder().uids(new ArrayList<>(userEmails)).build());
                    List<IusUserInfoRsp> userData = userListInfo.getData();
                    Map<String, String> userNameMap = new HashMap<>();
                    if (CollectionUtils.isNotEmpty(userData)) {
                        userNameMap = userData.stream()
                            .collect(Collectors.toMap(IusUserInfoRsp::getUid, IusUserInfoRsp::getName));
                    }

                    // 填充用户名和展示信息
                    for (ChangeSubFlowVO vo: list) {
                        String email = vo.getChangeExecUserEmail();
                        if (StringUtils.isNotBlank(email)) {
                            String name = userNameMap.get(email);
                            if (StringUtils.isBlank(name)) {
                                // 批量查询没查到的人名，通过全量查询单独再查一次
                                IusUserInfoRsp user = iusService.queryUserInfo(email);
                                if (user != null && StringUtils.isNotBlank(user.getName())) {
                                    name = user.getName();
                                }
                            }
                            vo.setChangeExecUserName(name);
                            // 构建展示信息：姓名+邮箱
                            if (StringUtils.isNotBlank(name)) {
                                vo.setChangeExecUserView(name + "(" + email + ")");
                            } else {
                                vo.setChangeExecUserView(email);
                            }
                        }
                    }
                }
            } catch (Exception ex) {
                log.error("query user info has ex", ex);
            }
        }

        PageVO pageVO = PageUtils.buildPageVo(subFlowRecordPageInfo.getTotal(), pageSize, page);
        ChangeSubFlowListVO result = new ChangeSubFlowListVO();
        result.setPageVo(pageVO);
        result.setChangeSubFlowList(list);
        return result;
    }

    /**
     * 根据主单条件查询主单记录ID列表
     *
     * @param queryReq 查询条件
     * @return 主单记录ID列表
     */
    private List<Long> filterFlowRecordIds(ChangeSubFlowListQueryReq queryReq) {
        // 构建主单查询条件
        ChangeFlowListQueryReq mainQueryReq = new ChangeFlowListQueryReq();
        mainQueryReq.setFlowId(queryReq.getFlowId());
        mainQueryReq.setItemId(queryReq.getItemId());
        mainQueryReq.setSupplier(queryReq.getSupplier());
        mainQueryReq.setOnlyMyFollowed(queryReq.getOnlyMyFollowed());

        String currentUser = RequestLocalBean.getUid();
        // 数据可见范围：变更负责人=当前用户的变更工单所关联的全部变更行动工单
        if (queryReq.getOnlyMyFollowed()) {
            if (StringUtils.isNotBlank(currentUser)) {
                mainQueryReq.setChangeCommander(currentUser);
            }
        }

        // 查询主单（不分页，只获取ID）
        List<ChangeRecord> changeRecords = changeRecordMapper.selectByCondition(mainQueryReq);

        if (CollectionUtils.isEmpty(changeRecords)) {
            return new ArrayList<>();
        }

        return changeRecords.stream().map(ChangeRecord::getId).distinct().collect(Collectors.toList());
    }

}
